#!/bin/bash
CURRENT_PATH=$(dirname "$(readlink -f "$0")")
LINE_ROOT=$(dirname "$(dirname "${CURRENT_PATH}")")
source "${LINE_ROOT}/bin/utils.sh"
export LOG_TAG="csbs_install"

SECURITY_PATH="/opt/huawei/dj/DJSecurity"
CA_PATH="${SECURITY_PATH}/server-ca"
CERT_PATH="${SECURITY_PATH}/server-cert"


function init_root_key()
{
    mkdir -p /opt/huawei/dj/paas
    if [[ -d "${LINE_ROOT}/paas" ]];then
        keystore_txt=$(find "${LINE_ROOT}/paas" -type f -name "*keystore.txt")
    else
        keystore_txt=$(find "${LINE_ROOT}/cfg" -type f -name "*keystore.txt")
    fi
    while read -r file;do
        /usr/bin/cp -f "${file}" /opt/huawei/dj/paas
        CHECK_RESULT $? "Copy default kmc keystore failed."
    done <<< "${keystore_txt}"
    set_common_permission /opt/huawei/dj/paas
    set_modify_permission /opt/huawei/dj/paas
    systemctl restart haveged
    CHECK_RESULT $? "Restart haveged failed."
    USER_PWD=$(decrypt_pwd "$(get_pwd djmanager_pwd "${LINE_ROOT}/cfg/util.conf")")
}


function modify_password_of_djmanager()
{
    if [[ -f /etc/pam.d/system-auth-local ]];then
        /usr/bin/cp -f /etc/pam.d/system-auth-local /etc/pam.d/system-auth.bak
        sed -i "/auth        sufficient    pam_unix.so nullok try_first_pass/d" /etc/pam.d/system-auth-local
        sed -i "/password    required      pam_pwhistory.so use_authtok remember=5 enforce_for_root/d" /etc/pam.d/system-auth-local
    fi
    sed -i '/djmanager/d' /etc/security/opasswd
    if command -v chpasswd >/dev/null 2>&1;then
        if ! (echo "djmanager:${USER_PWD}" | chpasswd);then
            log_warn "chpasswd command failed, use passwd."
            echo "${USER_PWD}" | passwd --stdin djmanager
            CHECK_RESULT $? "Passwd djmanager failed use pwd"
        fi
    else
        log_warn "chpasswd command not exist, use passwd."
        echo "${USER_PWD}" | passwd --stdin djmanager
        CHECK_RESULT $? "Passwd djmanager failed use pwd"
    fi
    mv -f /etc/pam.d/system-auth.bak /etc/pam.d/system-auth-local
}


function create_root_ca()
{
    mkdir -p ${SECURITY_PATH}/privkey ${CA_PATH}/demoCA/newcerts
    key_pwd=$(mkpasswd -l 30 | sed 's/{/(/g' | sed 's/}/)/g')
    echo "encrypt_password $(encrypt_pwd "${key_pwd}")" > ${SECURITY_PATH}/privkey/privkey.conf
    touch ${CA_PATH}/demoCA/index.txt
    rand_serial=$(openssl rand -base64 8 | sha256sum | cut -c1-8)
    echo "${rand_serial}" > ${CA_PATH}/demoCA/serial
    cd ${CA_PATH} || exit 1
    echo "${key_pwd}" | openssl genrsa -passout stdin -out ${CA_PATH}/ca-key.pem -aes256 2048
    CHECK_RESULT $? "create internal ca ${CA_PATH} key failed"
    log_info "create root ca ${CA_PATH} key successfully"
    echo "${key_pwd}" | openssl req -new -x509 -sha256 -key ${CA_PATH}/ca-key.pem -out ${CA_PATH}/ca-cert.pem -days 3650 \
    -subj "/C=CN/ST=SiChuan/O=Huawei/CN=CBS" -extensions v3_ca -passin stdin
    CHECK_RESULT $? "create internal ca ${CA_PATH} cert failed"
    log_info "generate root ca certificate finished"
}

function create_key_and_csr()
{
    cert_name=$1
    csr_cn=$2
    cert_passwd=$3
    echo "" > ${CA_PATH}/demoCA/index.txt.attr
    log_info "modify unique_subject value"
    echo "unique_subject = no" > ${CA_PATH}/demoCA/index.txt.attr
    echo "${cert_passwd}" | openssl genrsa -passout stdin -aes256 -out "${CA_PATH}/${cert_name}-key.pem" 2048
    CHECK_RESULT $? "there is no cert key for ${csr_cn}"
    log_info "ready to create ${cert_name} certification request"
    echo "${cert_passwd}" | openssl req -new -sha256 -key "${CA_PATH}/${cert_name}-key.pem" -out "${CA_PATH}/${cert_name}-csr.pem" -days 3650 \
    -subj "/C=CN/ST=SiChuan/O=Huawei/CN=${csr_cn}" -passin stdin
    CHECK_RESULT $? "Create certification request for ${cert_name} failed"
    log_info "Create certification request for ${cert_name} successfully."
}

function create_work_cert()
{
    cert_name=$1
    cert_passwd=$2
    cd ${CA_PATH} || exit 1
    log_info "ready to sign certification for ${cert_name}"
    echo "${cert_passwd}" | openssl ca -batch -in "${CA_PATH}/${cert_name}-csr.pem" -cert ${CA_PATH}/ca-cert.pem -keyfile ${CA_PATH}/ca-key.pem \
    -out "${CA_PATH}/${cert_name}-cert.pem" -md sha256 -days 3650 -passin stdin
    CHECK_RESULT $? "sign the certification failed"
    log_info "sign the certification successfully"
    mkdir -p "${CERT_PATH}/${cert_name}"
    if [[ "${cert_name}" == "haproxy" ]];then
        cat "${CA_PATH}/ca-cert.pem" >> "${CA_PATH}/${cert_name}-cert.pem"
        cat "${CA_PATH}/${cert_name}-key.pem" >> "${CA_PATH}/${cert_name}-cert.pem"
    fi
    mv  "${CA_PATH}/${cert_name}-cert.pem" "${CERT_PATH}/${cert_name}/"
    mv  "${CA_PATH}/${cert_name}-key.pem" "${CERT_PATH}/${cert_name}/"
    rm -rf "${CA_PATH}/${cert_name}-csr.pem"
}

function generate_certificate()
{
    key_pwd=$(decrypt_pwd "$(awk '{print $NF}' ${SECURITY_PATH}/privkey/privkey.conf)")
    COMPONENTS="scagent cli-client cps-monitor omm-ha gaussdb haproxy rabbitmq cms zookeeper karbor resource_manager alarm"
    for COMPONENT in ${COMPONENTS};do
        if [ "${COMPONENT}" = "gaussdb" ];then
            float_ip=$(get_info gaussdb_float_ip)
        else
            float_ip=$(get_info haproxy_float_ip)
        fi
        create_key_and_csr "${COMPONENT}" "${float_ip}" "${key_pwd}"
        create_work_cert "${COMPONENT}" "${key_pwd}"
    done
    rm -rf ${SECURITY_PATH}/server-ca/demoCA
}



function scp_cmd()
{
    password=$(get_safe_password "${USER_PWD}")
    /usr/bin/expect<<-EOF
    set timeout 60
    spawn scp -o StrictHostKeyChecking=no -r "$1" djmanager@$2:/home/djmanager
    expect {
        -nocase "password:" {send "${password}\r"}
    }
    expect {
        -nocase "password:" {exit 1}
    }
    lassign [wait] pid spawnid os_error_flag value
    exit \$value
EOF
}

function sync_root_ca()
{
    node_list=$(get_info manage_ip list)
    cur_node=$(get_info manage_ip)
    for node in ${node_list//,/ }; do
        if [[ "${cur_node}" == "${node}" ]];then
            continue
        fi
        log_info "begin to sync ca to ${node}"
        scp_times=0
        while [[ ${scp_times} -lt 360 ]]
        do
            if scp_cmd "${SECURITY_PATH}" "${node}";then
                log_info "sync ca to ${node} successfully."
                break
            fi
            log_warn "failed to sync ca to ${node},retry: ${scp_times}."
            sleep 5
            scp_times=$((scp_times+1))
        done
        if [[ "${scp_times}" == 360 ]];then
            log_error "sync ca file failed."
            exit 1
        fi
    done
    return 0
}


function main()
{
    init_root_key
    modify_password_of_djmanager
    [[ -d "${LINE_ROOT}/DJSecurity" ]] && /usr/bin/cp -rf "${LINE_ROOT}/DJSecurity" /opt/huawei/dj && exit 0
    node_index=$(get_info node_index)
    if [ "${node_index}" = "1" ];then
        create_root_ca
        sync_root_ca; CHECK_RESULT $? "Sync root ca failed!"
    else
        wait_times=0
        while [[ ${wait_times} -lt 360 ]]
        do
            if [[ -f /home/djmanager/DJSecurity/server-ca/ca-cert.pem ]];then
                log_info "root ca is existed."
                break
            fi
            usermod --unlock djmanager
            faillock --user djmanager --reset
            log_info "waiting sync root ca..."
            sleep 5
            wait_times=$((wait_times+1))
        done
        if [[ ${wait_times} == 360 ]];then
            log_error "waiting timeout."
            exit 1
        fi
        /usr/bin/mv -f /home/djmanager/DJSecurity ${SECURITY_PATH}
    fi
    generate_certificate
}

main
