import os
import secrets
import stat
import sys
import threading
import time

from basesdk import cms_info
from basesdk import security_harden
from basesdk import utils
from kmc import kmc
from safety_lib import safe_command

LOG = utils.get_logger("rabbitmq_watch")
RABBIT_CFG = 'rabbitmq.pwd'
GENERATE_CONFIG = 'generate_config'
RABBITMQ_CFG_PATH = "/opt/huawei/dj/bin/rabbitmq/etc/rabbitmq/rabbitmq.config"
RABBITMQ_CFG_BAK = "/opt/huawei/dj/etc/rabbitmq/rabbitmq.config"
RABBITMQ_ENV_PATH = "/opt/huawei/dj/bin/rabbitmq/etc/rabbitmq/rabbitmq-env.conf"
RABBITMQ_ENV_BAK = "/opt/huawei/dj/etc/rabbitmq/rabbitmq-env.conf"
RABBITMQ_CTL = "/opt/huawei/dj/bin/rabbitmq/sbin/rabbitmqctl"
RABBITMQ_TOOL = "/opt/huawei/dj/tools/rabbitmq"


def modify_password():
    try:
        rabbitmq_pwd = cms_info.get_cms_info(RABBIT_CFG)
        if not rabbitmq_pwd:
            LOG.error("Get rabbitmq.pwd failed.")
            return 1
        rabbitmq_decrypt = kmc.API().decrypt(0, rabbitmq_pwd)
        cmd_ret = safe_command.run_command(
            [RABBITMQ_CTL, 'change_password', 'rabbit'], sensitive_args=[rabbitmq_decrypt])
        if cmd_ret.exist_code != 0:
            LOG.error("Refresh rabbitmq conf failed.")
        return cmd_ret.exist_code
    except Exception:
        LOG.error("Modify password failed.")
        return 1


def generate_config_file():
    def _generate_config_file(encrypt_data, source_file, target_file):
        with open(source_file, 'r') as read_file:
            all_configuration = [line.replace("******", encrypt_data) for line in read_file.readlines()]
        flags = os.O_WRONLY | os.O_CREAT | os.O_TRUNC
        modes = stat.S_IWUSR | stat.S_IRUSR
        with os.fdopen(os.open(target_file, flags, modes), 'w') as write_file:
            write_file.writelines(all_configuration)
    try:
        cmd_ret = safe_command.run_command(['bash', f"{RABBITMQ_TOOL}/generate_encryptdata.sh"])
        if cmd_ret.exist_code != 0:
            raise Exception("encrypt rabbitmq data failed")
        _generate_config_file(cmd_ret.std_out, RABBITMQ_CFG_BAK, RABBITMQ_CFG_PATH)
        _generate_config_file(cmd_ret.std_out, RABBITMQ_ENV_BAK, RABBITMQ_ENV_PATH)
        return 0
    except Exception as err:
        LOG.error(f"Generate rabbitmq config file failed {err}.")
        return 1


class RabbitWatcher(threading.Thread):
    def __init__(self):
        super(RabbitWatcher, self).__init__()

    def run(self):
        while True:
            safe_command.run_command(['bash', f"{RABBITMQ_TOOL}/check_cluster.sh"], timeout=None)
            time.sleep(secrets.SystemRandom().randint(60, 240))


if __name__ == '__main__':
    security_harden.switch_to_openstack_user()
    if len(sys.argv) == 2 and sys.argv[1] == RABBIT_CFG:
        sys.exit(modify_password())
    if len(sys.argv) == 2 and sys.argv[1] == GENERATE_CONFIG:
        sys.exit(generate_config_file())
    if len(sys.argv) == 1:
        RabbitWatcher().start()
