import datetime
import time

import requests
import utils.common.log as logger
from utils.business.manageone_cmdb_util import ManageOneCmdbUtil
from utils.business.project_util import ProjectApi
from utils.common.exception import HCCIException
from utils.common.param_check import check_param_ip
from utils.common.ssh_util import Ssh

from plugins.eBackup.common import constant
from plugins.eBackup.common.util import Utils
from plugins.eBackup.scripts.upgrade.util import get_ebackup_driver_nodes


def enable_other_params(check_result, all_params):
    for each in all_params:
        if -1 == str(check_result.check_list).find(each):
            check_result.set_check_result(param_keys=[each], status=200,
                                          error_msg="")


def get_password_expired_days(cmd_outputs):
    expired_days = 999
    for item in cmd_outputs:
        key = item.split(":")[0].strip()
        value = item.split(":")[1].strip()
        if key.lower() == 'password expires':
            expired_time = value.lower()
            if expired_time == 'never':
                expired_days = 'never'
            else:
                expired_days = (datetime.datetime.strptime(expired_time,
                                                           '%b %d, %Y') - datetime.datetime.utcnow()).days + 1
                logger.info(f"The password will expire in {str(expired_time)}")
            break
    if expired_days == 'never':
        logger.info("The password will never expire.")
    else:
        logger.info("The password will expire in %s days" % expired_days)
    return expired_days


def do_check_age(ssh_client, account, goto_expire_days):
    cmd_output = Ssh.ssh_exec_command_return_list(ssh_client, f"chage -l {account}")
    expired_days = get_password_expired_days(cmd_output)
    if expired_days == 'never':
        return constant.CHECK_SUC, ""
    if expired_days <= goto_expire_days:
        logger.info(f"The password of the {account} user  will expire in {expired_days} days.")
        return constant.USER_PWD_ABOUT_TO_EXPIRE, str(expired_days)
    return constant.CHECK_SUC, ""


def check_ssh_login(server_ip, account, account_password, account_root_password, goto_expire_days=7):
    # 0:success
    # 1:普通用户密码错误
    # 2:普通用户密码过期或者OS语言不是英语
    # 3:普通用户密码即将过期
    # 4:root用户密码错误
    ssh_client = None
    try:
        ssh_client = Ssh.ssh_create_client(server_ip, account, account_password)
    except Exception as e:
        logger.error("The password of the %s user is incorrect on host %s.The reason is:%s", server_ip, account, str(e))
        Utils.close_ssh_clinet(ssh_client)
        return constant.USER_PWD_ERR, account
    logger.info("Login %s successfully by user %s", server_ip, account)

    try:
        result = Ssh.ssh_send_command(ssh_client, 'su - root', 'Password:', 60)
    except Exception as e:
        logger.error("Excute su - root on host %s failed.The reason is:%s", server_ip, str(e))
        Utils.close_ssh_clinet(ssh_client)
        return constant.USER_PWD_EXPIRED_OR_LANGUAGE_ERR, ""
    for sub_result in result:
        if "Account temporarily locked" in sub_result:
            logger.error("root has been locked.")
            return constant.ROOT_LOCKED, "root"
    logger.info(f"Excute su - root on host {server_ip} successfully by user {account}")

    try:
        Ssh.ssh_send_command(ssh_client, account_root_password, '#', 60)
    except Exception as e:
        logger.error("The password of the root user is incorrect on host %s.The reason is:%s", server_ip, str(e))
        Utils.close_ssh_clinet(ssh_client)
        return constant.ROOT_USER_PWD_ERR, 'root'
    logger.info("su - root successfuly on host %s", server_ip)

    logger.info(f"Begin to check the age of the {account} user.")
    try:
        return do_check_age(ssh_client, account, goto_expire_days)
    except Exception as e:
        logger.error(f"Check the the age of the {account} user failed. The reason is:{str(e)}")
        return constant.USER_PWD_ABOUT_TO_EXPIRE, ""
    finally:
        Utils.close_ssh_clinet(ssh_client)


def check_openstack_cps_params(openstack_reverse_proxy_ip, openstack_fsp_pwd, openstack_root_pwd, cps_admin_pwd):
    """
    登录fsp
    """
    ssh_client = None

    def do_login_root():
        Ssh.ssh_send_command(ssh_client, "su - root", "Password:", 30, retry_times=3)
        # root账户登录校验管理员密码
        Ssh.ssh_send_command(ssh_client, openstack_root_pwd, "#", 30)

    def do_login_openstack():
        # cps_admin密码校验
        Ssh.ssh_send_command(ssh_client, "TMOUT=0", "#", 300)
        Ssh.ssh_send_command(ssh_client, "source set_env", "1|2", 300)
        Ssh.ssh_send_command(ssh_client, "2", "CPS_USERNAME", 300)
        Ssh.ssh_send_command(ssh_client, "cps_admin", "CPS_PASSWORD", 300)
        resp = Ssh.ssh_send_command(ssh_client, cps_admin_pwd, "#", 300)
        if [x for x in resp if x.find("Authenticate Failed") >= 0]:
            raise HCCIException(650051)

    try:
        try:
            # 登录fsp校验fsp普通账户密码
            ssh_client = Ssh.ssh_create_client(openstack_reverse_proxy_ip, "fsp", openstack_fsp_pwd)
        except Exception as ex:
            logger.error("The ssh connection failed,description: {}".format(ex))
            return ['openstack_fsp_pwd'], 500, HCCIException(650050, 'openstack_fsp_pwd', ex)
        logger.info("finish create SSH client use fsp user.")

        try:
            do_login_root()
        except Exception as ex:
            logger.error(f"Failed to switch root privileges, description: {ex}.")
            return ['openstack_root_pwd'], 500, HCCIException(650050, 'openstack_root_pwd', ex)
        logger.info("finish login to root.")

        try:
            do_login_openstack()
        except Exception as ex:
            logger.error(f"Authenticate Failed: openstack_cps_admin_pwd is invalid:{str(ex)}")
            return ['openstack_cps_admin_pwd'], 500, HCCIException(650050, 'cps_admin_pwd', ex)
        logger.info("finish login openstack by cps_admin.")
        return ['openstack_cps_admin_pwd'], 200, ''
    finally:
        # 校验完毕关闭ssh连接
        if ssh_client:
            Ssh.ssh_close(ssh_client)
            logger.info("OpenStack SSH client is closed")


def check_ebackup_nodes(check_result, params_key, nodes, account_password,
                        account_root_password, unpassed_ordinary_user_nodes,
                        unpassed_root_nodes, password_goto_expire_msg,
                        password_lock_msg):
    is_check_ok = True
    ip_list = nodes.replace(" ", "").replace("|", ";").split(";")
    ret = None

    for ip in ip_list:
        try:
            ret = check_param_ip(ip)
        except HCCIException as error:
            check_result.set_check_result(param_keys=[params_key], status=500, error_msg=error)
        if not ret:
            is_check_ok = False
            logger.error("Ip(%s) address format error", ip)
            check_result.set_check_result(param_keys=[params_key], status=500, error_msg=HCCIException(650034))
    if is_check_ok:
        check_result.set_check_result(param_keys=[params_key], status=200, error_msg="")
        for ip in ip_list:
            ret_code, msg = check_ssh_login(ip, "hcp", account_password, account_root_password)
            if ret_code != constant.CHECK_SUC:
                is_check_ok = False
                handle_ebackup_login_result(ret_code, ip,
                                            unpassed_ordinary_user_nodes,
                                            unpassed_root_nodes,
                                            password_goto_expire_msg,
                                            password_lock_msg, msg)
    return is_check_ok


def handle_ebackup_login_result(ret_code, ebk_ip, unpassed_ordinary_user_nodes,
                                unpassed_root_nodes, password_goto_expire_msg,
                                password_lock_msg, msg):
    if constant.USER_PWD_ERR == ret_code or constant.USER_PWD_EXPIRED_OR_LANGUAGE_ERR == ret_code:
        unpassed_ordinary_user_nodes.append(ebk_ip)
    elif constant.USER_PWD_ABOUT_TO_EXPIRE == ret_code:
        password_goto_expire_msg.update({ebk_ip: f"The password will expire in {msg} days."})
    elif constant.ROOT_USER_PWD_ERR == ret_code:
        unpassed_root_nodes.append(ebk_ip)
    elif constant.ROOT_LOCKED == ret_code:
        password_lock_msg.update({ebk_ip: "root password has been locked."})


def handle_openstack_login_result(ret_code, msg, host_ip,
                                  unpassed_fsp_nodes, unpassed_root_nodes,
                                  password_goto_expire_msg):
    if constant.USER_PWD_ERR == ret_code or constant.USER_PWD_EXPIRED_OR_LANGUAGE_ERR == ret_code:
        unpassed_fsp_nodes.append(host_ip)
    elif constant.USER_PWD_ABOUT_TO_EXPIRE == ret_code:
        password_goto_expire_msg.update({host_ip: f"The password will expire in {msg} days."})
    elif constant.ROOT_USER_PWD_ERR == ret_code:
        unpassed_root_nodes.append(host_ip)


def check_openstack_params(check_result, param_dict, unpassed_fsp_nodes, unpassed_root_nodes, password_goto_expire_msg):
    reverse_proxy_ip = param_dict["openstack_reverse_proxy_ip"]
    reverse_ips = [reverse_proxy_ip]

    def do_check_login():
        driver_nodes = driver_host_dict["eBackupDriver"]
        [driver_nodes.append(ip) for ip in driver_host_dict["cpsnode"] if ip not in driver_nodes]
        for node in driver_nodes:
            ret_code, msg = check_ssh_login(node, "fsp", param_dict["openstack_fsp_pwd"],
                                            param_dict["openstack_root_pwd"])
            handle_openstack_login_result(ret_code, msg, node, unpassed_fsp_nodes, unpassed_root_nodes,
                                          password_goto_expire_msg)
        check_result.set_check_result(param_keys=["openstack_fsp_pwd"], status=200, error_msg="")

    # 去级联场景需要校验级联层的密码
    for each_ip in reverse_ips:
        ret_code, msg = check_ssh_login(each_ip, "fsp", param_dict["openstack_fsp_pwd"],
                                        param_dict["openstack_root_pwd"])
        handle_openstack_login_result(ret_code, msg, each_ip, unpassed_fsp_nodes, unpassed_root_nodes,
                                      password_goto_expire_msg)
        param_keys, status, error_msg = check_openstack_cps_params(each_ip, param_dict["openstack_fsp_pwd"],
                                                                   param_dict["openstack_root_pwd"],
                                                                   param_dict["openstack_cps_admin_pwd"])
        check_result.set_check_result(param_keys=param_keys, status=status, error_msg=error_msg)
        if 0 != ret_code:
            continue
        driver_host_dict = None
        try:
            driver_host_dict = get_ebackup_driver_nodes(each_ip, param_dict)
        except Exception as ex:
            check_result.set_check_result(param_keys=["openstack_fsp_pwd"], status=500,
                                          error_msg=HCCIException(650008, str(ex)))
        if not driver_host_dict:
            continue
        try:
            do_check_login()
        except Exception as ex:
            check_result.set_check_result(param_keys=["openstack_fsp_pwd"], status=500,
                                          error_msg=HCCIException(650008, str(ex)))


def do_check_web_login(web_ip, admin_password, ip_version, role, check_result):
    token, session = Utils.login(web_ip, admin_password, ip_version)
    header = {'Accept': 'application/json;version=2.2;charset=UTF-8', 'iBaseToken': token, 'Accept-Language': 'en',
              'Cookie': f'language=en; {session};DEVICE_ID=dev;sessionIdleTime=60000;'
                        f'MACHINEROLE={role};CSRF_IBASE_TOKEN={token}'}
    if ip_version == 4:
        url = f"https://{web_ip}:8088/rest/dev/vbackup_server"
    else:
        url = f"https://[{web_ip}]:8088/rest/dev/vbackup_server"
    rsp = requests.get(url=url, headers=header, verify=False)
    rsp_dict = rsp.json()
    error_code = rsp_dict["error"]["code"]
    if error_code is None:
        logger.error("errcode is empty.")
        raise Exception("errcode is empty.")
    if error_code != 0:
        error_des = rsp_dict["error"]["description"]
        logger.error(f"check admin permission fail [{web_ip}],the description is:{error_des}")
        check_result.set_check_result(param_keys=["eBackup_admin_pwd"], status=500,
                                      error_msg=HCCIException(650065, web_ip, error_des))
        return False
    check_result.set_check_result(param_keys=["eBackup_admin_pwd"], status=200, error_msg="")
    return True


def check_web_login(check_result, hosts_groups, hcp_password, root_password, admin_password, role):
    token = ''
    session = ''
    web_ip = Utils.find_float_ip(hosts_groups, hcp_password, root_password)
    if web_ip == "":
        check_result.set_check_result(param_keys=["eBackup_admin_pwd"], status=500,
                                      error_msg=HCCIException(650017, str(hosts_groups)))
        return False

    ip_version = Utils.check_ip_version(web_ip)
    try:
        return do_check_web_login(web_ip, admin_password, ip_version, role, check_result)
    except HCCIException as e:
        check_result.set_check_result(param_keys=["eBackup_admin_pwd"], status=500, error_msg=e)
        return False
    except Exception as e:
        check_result.set_check_result(param_keys=["eBackup_admin_pwd"], status=500,
                                      error_msg=HCCIException(650008, str(e)))
        return False
    finally:
        Utils.logout(web_ip, token, session, role, ip_version)


def check_is_empty(params_dict, check_result, params_key_list):
    ret = True
    check_list = params_key_list[:]
    for key in check_list:
        if params_dict[key] is None or 0 == len(params_dict[key]):
            ret = False
            check_result.set_check_result(param_keys=[key], status=500,
                                          error_msg=HCCIException(650035))
    return ret


def check_connectivity(check_result, host_ip, hcp_pwd, root_pwd, checked_address, key):
    ssh_client = None
    unpassed_ip = []

    def do_check_connectivity(ip):
        ping_cmd = f'ping -c2 {checked_address} >/dev/null && echo "Success"'
        ssh_client = Ssh.ssh_create_client(ip, "hcp", hcp_pwd)
        Ssh.ssh_send_command(ssh_client, 'su - root', 'Password:', 100)
        Ssh.ssh_send_command(ssh_client, root_pwd, '#', 100)
        result = Ssh.ssh_exec_command_return_list(ssh_client, ping_cmd)
        if str(result).find("Success") < 0:
            unpassed_ip.append(ip)

    for host in host_ip:
        try:
            return do_check_connectivity(host)
        except Exception as e:
            check_result.set_check_result(param_keys=[key], status=500, error_msg=HCCIException(650008, str(e)))
            return False
        finally:
            Utils.close_ssh_clinet(ssh_client)
    if not unpassed_ip:
        check_result.set_check_result(param_keys=[key], status=500,
                                      error_msg=HCCIException(650011, ';'.join(unpassed_ip), checked_address, key))
        return False
    else:
        check_result.set_check_result(param_keys=[key], status=200, error_msg="")
        return True


def is_exist_same_ips(check_result, key, host_ips):
    ip_list = []
    all_ip_list = host_ips.replace("|", ";").split(";")
    for ip in all_ip_list:
        if ip not in ip_list:
            ip_list.append(ip)
        else:
            check_result.set_check_result(param_keys=[key], status=500,
                                          error_msg=HCCIException(650010))
            return True
    return False


def get_ebackup_cluster_url(host_ips_list, hcp_password, root_password):
    web_ip = Utils.find_float_ip(host_ips_list,
                                 hcp_password,
                                 root_password)
    if not web_ip:
        logger.error("Find float failed.")
        raise Exception("Find float failed.")
    ip_version = Utils.check_ip_version(web_ip)
    if ip_version == 4:
        url = "https://%s:8088/rest/dev/vbackup_server" % web_ip
    else:
        url = "https://[%s]:8088/rest/dev/vbackup_server" % web_ip
    return web_ip, url, ip_version


def parse_ebackup_server_response(rsp, web_ip):
    rsp_dict = rsp.json()
    error_code = rsp_dict["error"]["code"]
    if error_code is None:
        logger.error("errcode is empty.")
        raise Exception("errcode is empty.")
    if error_code != 0:
        logger.error(
            "Query backup server info failed on [%s],the description "
            "is:%s" % (web_ip, rsp_dict["error"]["description"]))
        if "Insufficient user permission" in rsp_dict["error"]["description"]:
            raise HCCIException(650065, web_ip, rsp_dict["error"]["description"])
        raise Exception(rsp_dict["error"]["description"])

    all_host = []
    for host in rsp_dict["data"]:
        if host["MANAGEMENTIP"] not in all_host:
            all_host.append(host["MANAGEMENTIP"])
    return all_host


def is_in_same_cluster(check_result, key, host_ips_list, hcp_password,
                       root_password, admin_password, role):
    token = ''
    session = ''
    web_ip = ''
    ip_version = ''

    def do_is_in_same_cluster():
        logger.info(f"Begin to check if the hosts {str(host_ips_list)} are in the same cluster.")
        web_ip, url, ip_version = get_ebackup_cluster_url(host_ips_list, hcp_password, root_password)
        token, session = Utils.login(web_ip, admin_password, ip_version)

        header = {'Accept': 'application/json;version=2.2;charset=UTF-8', 'iBaseToken': token, 'Accept-Language': 'en',
                  'Cookie': f'language=en; {session};DEVICE_ID=dev;sessionIdleTime=60000;'
                            f'MACHINEROLE={role};CSRF_IBASE_TOKEN={token}'}
        rsp = requests.get(url=url, headers=header, verify=False)
        if rsp.status_code == 200:
            all_host = parse_ebackup_server_response(rsp, web_ip)
            logger.info("The host ips by user inputing are:" + str(host_ips_list))
            logger.info("The host ips by querying are:" + str(all_host))
            return is_same_hosts(host_ips_list, all_host, key, check_result)
        else:
            logger.error("Response status code is not 200.The url is" + url)
            raise Exception("Response status code is not 200.")

    try:
        return do_is_in_same_cluster()
    except Exception as e:
        logger.error("Exception occurs:" + str(e))
        check_result.set_check_result(param_keys=[key], status=500,
                                      error_msg=HCCIException(650008, str(e)))
        return False
    finally:
        Utils.logout(web_ip, token, session, role, ip_version)


def is_same_hosts(host_ips_list, all_host, key, check_result):
    if len(host_ips_list) == len(all_host):
        is_true = True
        for ip in host_ips_list:
            if ip not in all_host:
                is_true = False
                break
        if is_true:
            check_result.set_check_result(param_keys=[key], status=200,
                                          error_msg="")
            return True
    error_code = 650020
    if key == "eBackup_Workflow_nodes":
        error_code = 650070
    check_result.set_check_result(
        param_keys=[key],
        status=500,
        error_msg=HCCIException(error_code, ';'.join(host_ips_list), ';'.join(all_host), key))
    return False


def check_ebackup_role(check_result, key, host_ips_list, hcp_password, root_password):
    ebk_role_dict = {'eBackup_Workflow_nodes': ['2', '3'],
                     'eBackup_Datamover_nodes': ['0', '1']}
    unpassed_ips = []
    query_role_cmd = '''cat /opt/huawei-data-protection/ebackup/conf/hcpconf.ini | grep MachineRole |
                            awk -F "=" '{print int($2)}' '''

    def do_check_ebackup_role():
        roles = ebk_role_dict[key]
        for ip in host_ips_list:
            ssh_client = Ssh.ssh_create_client(ip, 'hcp', hcp_password)
            Ssh.ssh_send_command(ssh_client, 'su - root', 'Password:', 100)
            Ssh.ssh_send_command(ssh_client, root_password, '#', 100)
            result = Ssh.ssh_exec_command_return_list(ssh_client, query_role_cmd)
            logger.info(f"The server role is {str(result[0])} of host {ip}")
            role = result[1].strip().replace('\n', '')
            if role not in roles:
                logger.error(f"The node[{key}] is not a {key} node")
                unpassed_ips.append(ip)
            Ssh.ssh_close(ssh_client)
        if unpassed_ips:
            error_code = 650023
            if key == "eBackup_Workflow_nodes":
                error_code = 650071
            logger.error('These nodes %s are not %s role node.' % (str(host_ips_list), key))
            check_result.set_check_result(param_keys=[key], status=500,
                                          error_msg=HCCIException(error_code, ';'.join(host_ips_list), key, key))
            return False
        else:
            check_result.set_check_result(param_keys=[key], status=200, error_msg="")
            return True

    try:
        return do_check_ebackup_role()
    except Exception as e:
        check_result.set_check_result(param_keys=[key], status=500, error_msg=HCCIException(650008, str(e)))
        return False


def parse_ebackup_node_info_on_cmdb(ebackup_nodes_list, deploy_info_dict):
    for node in ebackup_nodes_list:
        if -1 != node["name"].find("eBackup_service"):
            group_name = node["name"][:-1] + 's'
            if group_name not in deploy_info_dict['eBackup_Datamover_nodes']:
                deploy_info_dict['eBackup_Datamover_nodes'].update({group_name: [node['ipAddresses'][0]['ip']]})
            else:
                deploy_info_dict['eBackup_Datamover_nodes'][group_name].append(node['ipAddresses'][0]['ip'])
        else:
            logger.error("Parse ebackup nodes info failed.")
            raise Exception("Parse ebackup nodes info failed.")


def parse_openstack_proxy_ip_on_cmdb(openstack_info, deploy_info_dict):
    for openstack in openstack_info:
        if openstack['indexName'] != 'OpenStack':
            continue
        for item in openstack['extendInfos']:
            if -1 != str(item).find('openstack_reverse_proxy_ip'):
                deploy_info_dict['openstack_reverse_proxy_ip'] = item['value']
                break
        if 0 != len(deploy_info_dict['openstack_reverse_proxy_ip']):
            logger.info("The openstack reverse proxy ip ""is:%s",
                        str(deploy_info_dict['openstack_reverse_proxy_ip']))
            break


def get_deploy_info_from_cmdb(project_id, region_id):
    def do_get_deploy_info_from_cmdb():
        logger.info("Begin to query ebackup deploy info from cmdb...")
        deploy_info_dict = {'eBackup_Datamover_nodes': {}, 'openstack_reverse_proxy_ip': ''}
        cmdb = ManageOneCmdbUtil(project_id)
        ebackup_nodes_list = cmdb.get_deploy_node_info(region_id, 'eBackup')
        if 0 == len(ebackup_nodes_list):
            logger.error("Query ebackup nodes info failed.")
            raise Exception("Query ebackup nodes info failed.")
        parse_ebackup_node_info_on_cmdb(ebackup_nodes_list, deploy_info_dict)

        openstack_info = cmdb.get_cloud_service_info(region_id, 'OpenStack')
        if 0 == len(openstack_info):
            logger.error("Query openstack deploy info failed.")
            raise Exception("Query openstack deploy info failed.")
        parse_openstack_proxy_ip_on_cmdb(openstack_info, deploy_info_dict)
        if 0 == len(deploy_info_dict['openstack_reverse_proxy_ip']):
            logger.error("Get openstack reverse proxy ip failed.")
            raise Exception("Get openstack reverse proxy ip failed.")

        logger.info("Query ebackup deploy info completed:%s" % str(deploy_info_dict))
        return deploy_info_dict

    try:
        return do_get_deploy_info_from_cmdb()
    except Exception as e:
        logger.info("Exception occurs when query ebackup deploy info: " + str(e))
        raise e


def init_params(*args):
    if len(args) == 6:
        check_data, db_opt, project_id, check_result, all_params, scene = args
    else:
        (check_data, db_opt, project_id, check_result, all_params), scene = args, 'upgrade'
    check_data.update(db_opt.get_need_check_cloud_params(project_id, "OpenStack"))
    if scene != "migrate":
        check_data.update({"eBackup_Datamover_nodes": check_data.get('eBackup_Datamover_nodes').replace(",", ";")})
    check_data.update({"current_region_id": ProjectApi().get_regionlist_by_project_Id(project_id)[0]})
    is_true = check_is_empty(check_data, check_result, all_params)
    if not is_true:
        return check_result
    if scene != "migrate" and is_exist_same_ips(check_result, "eBackup_Datamover_nodes",
                                                check_data.get("eBackup_Datamover_nodes")):
        return check_result
    return ''


def check_datamover_web_login(check_data, check_result, all_params):
    hcp_password = check_data.get("eBackup_hcp_pwd")
    root_password = check_data.get("eBackup_root_pwd")
    admin_password = check_data.get("eBackup_admin_pwd")
    for datamover_groups in check_data.get("eBackup_Datamover_nodes").split("|"):
        datamover_hosts = datamover_groups.split(";")
        if not check_web_login(check_result, datamover_hosts, hcp_password, root_password, admin_password, 1):
            break
        if not is_in_same_cluster(check_result, 'eBackup_Datamover_nodes', datamover_hosts, hcp_password, root_password,
                                  admin_password, 1):
            break
        if not check_ebackup_role(check_result, 'eBackup_Datamover_nodes', datamover_hosts, hcp_password,
                                  root_password):
            break
    enable_other_params(check_result, all_params)
    return check_result


def check_workflow_web_login(check_data, check_result, all_params):
    hcp_password = check_data.get("eBackup_hcp_pwd")
    root_password = check_data.get("eBackup_root_pwd")
    admin_password = check_data.get("eBackup_admin_pwd")
    workflow_hosts = check_data.get("eBackup_Workflow_nodes").split(";")
    is_true = check_web_login(check_result, workflow_hosts, hcp_password,
                              root_password, admin_password, 2)
    if is_true:
        time.sleep(120)  # 登录太快会登录不上
        b_result = is_in_same_cluster(check_result, 'eBackup_Workflow_nodes',
                                      workflow_hosts, hcp_password,
                                      root_password, admin_password, 2)
        if b_result:
            check_ebackup_role(check_result, 'eBackup_Workflow_nodes',
                               workflow_hosts, hcp_password, root_password)
    enable_other_params(check_result, all_params)
    return check_result
