# -*- coding:utf-8 -*-

import time
from jinja2 import Environment, FileSystemLoader

from utils.business.dns_util import DNSApi, DNSNodeType
from utils.business.param_util import ParamUtil
from utils.business.project_condition_utils import get_project_condition_boolean
from utils.business.project_util import ProjectApi
from utils.common import log as logger
from utils.common.exception import HCCIException
from utils.common.fic_base import StepBaseInterface
from utils.common.message import Message
from utils.common.ssh_util import Ssh

from plugins.eBackup.common.api_adapter import API
from plugins.eBackup.common.constant import TIME_OUT_CODE
from plugins.eBackup.common.ebackup_rest import EbackupRest
from plugins.eBackup.common.ebackup_util import EBackupUtil
from plugins.eBackup.common.model import DmkTaskInfo
from plugins.eBackup.common.util import Utils
from plugins.eBackup.scripts.common import ebackup_util as ebackup_handle


class AdvancedConfig(StepBaseInterface):
    def __init__(self, project_id, pod_id):
        logger.init("eBackup")
        # self.params
        self.params = ParamUtil()
        self.project_id = project_id
        self.pod_id = pod_id
        self.project_api = ProjectApi()
        self.rest_api = EbackupRest()
        self.ssh_obj = Ssh()
        self.err_code = 0
        self.description = ''
        self.token = ''
        self.session_content = ''
        self.headers = ''
        self.base_url = ""
        self.if_ipv6 = 0
        self.datamover_iplist = None
        self.backup_storageunit_type = 'not_BSTU'
        self.__config_dict = \
            self.params.get_service_cloud_param(pod_id, "eBackup")

    def pre_check(self, project_id, pod_id):
        """
        插件内部接口：执行安装前的资源预检查，该接口由execute接口调用，工具框架不会直接调用此接口。
        :param project_id:
        :param pod_id:
        :return:
        """
        pass

    def main_handle(self):
        self.get_url_pre()
        self.login(self.__config_dict['eBackup_weblogin_password'])
        self.config_time_zone()
        self.check_storage_ip()
        self.config_storage_unit()
        self.config_ntp()
        if self.description == 'REST unauthorized.':  # session is invalid
            logger.info("the session is time out, try to login again.")
            self.err_code = 0
            self.login(self.__config_dict['eBackup_weblogin_password'])
            self.config_ntp()
        self.check_data_backup_status()
        # session is invalid
        if self.err_code == TIME_OUT_CODE or self.description == 'REST unauthorized.':
            logger.info("the session is invalid or time out, try to login again.")
            self.err_code = 0
            self.login(self.__config_dict['eBackup_weblogin_password'])
            self.check_data_backup_status()
        self.logout()
        self.set_policy_route()
        if self.err_code != 0:
            logger.error("advanced config failed.")
            if self.description == '':
                raise HCCIException(self.err_code)
            else:
                raise HCCIException(self.err_code, self.description)
        management_float_ip = Utils.find_admin_ip(self.datamover_iplist, self.__config_dict.get('hcp_ssh_password'),
                                                  self.__config_dict.get('eBackup_os_root_password'))
        EBackupUtil(self.project_id, self.pod_id).add_new_user(management_float_ip, "EBKUser")
        EBackupUtil(self.project_id, self.pod_id).add_new_user(management_float_ip, "NBIUser")

    def execute(self, project_id, pod_id):
        """
        标准调用接口：执行安装前预检查&安装&配置
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        if get_project_condition_boolean(project_id, 'eBackup_Proxy'):
            self.set_policy_route()
            if self.err_code != 0:
                logger.error("advanced config failed.")
                if self.description == '':
                    raise HCCIException(self.err_code)
                else:
                    raise HCCIException(self.err_code, self.description)
            return Message(200)
        try:
            self.main_handle()
        except HCCIException as e:
            return Message(500, e)
        except Exception as e:
            logger.error(
                "server&proxy advanced config failed ,the reason is %s" % e)
            return Message(500, e, e)
        return Message(200)

    def rollback(self, project_id, pod_id):
        """
        标准调用接口：执行回滚
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        return Message()

    def retry(self, project_id, pod_id):
        """
        标准调用接口：重试
        :return: Message类对象
        """
        return self.execute(project_id, pod_id)

    def check(self, project_id, pod_id):
        """
        标准调用接口：重试
        :param project_id:
        :param pod_id:
        :return:
        """
        pass

    def update_error_info(self, err, des):
        if self.err_code == 0:
            self.err_code = err
            self.description = des

    def get_url_pre(self):
        externalom_ip_lower = self.__config_dict['datamover_externalom_iplist'].lower()
        iplist = externalom_ip_lower.split(',')
        self.datamover_iplist = [ipStr.lower() for ipStr in iplist]
        if len(self.datamover_iplist) == 1:
            self.base_url = 'https://' + self.datamover_iplist[0] + ':8088/rest/dev'
        else:
            self.base_url = 'https://' + self.__config_dict['datamover_management_float_ip'] + ':8088/rest/dev'

    def check_storage_ip(self):
        ping_failed_iplist = []
        if not get_project_condition_boolean(self.project_id, 'eBackup_MSTU_OFF'):
            pathinfo = self.__config_dict['mdb_path'].split(':')
            if pathinfo[0] and not self.ping_storage_ip(pathinfo[0]):
                ping_failed_iplist.append(pathinfo[0])

        advanced_options_dict = self.project_api.get_project_info(self.project_id).get('advanced_options')
        self.backup_storageunit_type = advanced_options_dict.get('eBackup_BSTU', 'not_BSTU')
        if self.__config_dict['backup_storageunit_path'] != '' and self.backup_storageunit_type != 'not_BSTU':
            if self.backup_storageunit_type != 'eBackup_BSTU_CIFS':
                pathinfo = self.__config_dict['backup_storageunit_path'].split(':')
                ip = pathinfo[0]
            else:
                pathinfo = self.__config_dict['backup_storageunit_path']
                tmp_path = pathinfo[2:]
                pos = tmp_path.find('/')
                ip = tmp_path[0:pos]
            if not self.ping_storage_ip(ip):
                ping_failed_iplist.append(ip)

        if len(ping_failed_iplist) != 0:
            ping_failed_ipstr = ','.join(ping_failed_iplist)
            raise HCCIException(653043, ping_failed_ipstr)

    def ping_storage_ip(self, ip):
        if self.if_ipv6 == 1:
            cmd = "ping6 " + ip + " -c 3"
        else:
            cmd = "ping " + ip + " -c 3"
        if not ebackup_handle.check_ip_valid(ip, self.project_id):
            raise HCCIException(653103, ip)
        ssh_client = self.ssh_obj.ssh_create_client(
            self.datamover_iplist[0], "root",
            self.__config_dict['eBackup_os_root_password'])
        result = self.ssh_obj.ssh_exec_command_return(ssh_client, cmd)
        self.ssh_obj.ssh_close(ssh_client)
        if result:
            logger.info("excute succ.")
            return True
        else:
            logger.error("excute failed.")
            return False

    def login(self, pwd):
        logger.info("begin to login ebackup.")
        login_url = self.base_url + '/login'
        headers = {'Accept': 'application/json'}
        # login use pwd
        data = '{"scope":0,"username":"admin","password":"' + pwd + '"}'
        rsp = self.rest_api.post(login_url, headers, data)
        if rsp.status_code != 200:
            logger.error("rest req return not 200.")
            raise HCCIException(653020)
        errcode = rsp.json()["error"]["code"]
        if errcode is None:
            logger.error("errcode is empty.")
            raise HCCIException(653020)
        if errcode == 0:
            self.token = rsp.json()["data"]["iBaseToken"]
            cookie = rsp.headers.get('Set-Cookie')
            self.session_content = cookie.split(';')[0]
            self.headers = {
                'Accept': 'application/json;version=2.2;charset=UTF-8',
                'iBaseToken': self.token,
                'Cookie':
                    'language=en;' + self.session_content + '; '
                                                            'DEVICE_ID=dev; sessionIdleTime=60000; MACHINEROLE=0;'
                                                            ' CSRF_IBASE_TOKEN=' + self.token}
            logger.info("login ebackup succ.")
        else:
            description = rsp.json()["error"]["description"]
            logger.error("login failed,description:" + description)
            raise HCCIException(653022, description)

    def check_data_backup_status(self):
        logger.info("begin to config datamover manage data backup.")
        if not get_project_condition_boolean(
                self.project_id, 'eBackup_MSTU_OFF') and \
                self.__config_dict['mdb_path'] != "":
            url = self.base_url + '/vbackup_dbbackup_storage_conf/modify_' \
                                  'dbbackup_storage_conf'

            dir_name = self.get_dir_name()
            data = ''
            path = ''
            if get_project_condition_boolean(
                    self.project_id, 'eBackup_MSTU_FTPS'):
                path, data = self.get_ftps_data(dir_name)

            elif get_project_condition_boolean(
                    self.project_id, 'eBackup_MSTU_FTP'):
                path, data = self.get_ftp_data(dir_name)

            elif get_project_condition_boolean(
                    self.project_id, 'eBackup_MSTU_SFTP'):
                path, data = self.get_sftp_data()

            elif get_project_condition_boolean(
                    self.project_id, 'eBackup_MSTU_S3'):
                path = self.__config_dict['mdb_path']
                data = '{"FILESYSTEMTYPE":"7","PATH":"' + self.__config_dict['mdb_path'] \
                       + '","PROTOCOL":"0","SUBDIR":"' + dir_name + '","AK":"' \
                       + self.__config_dict['mdb_user_or_s3ak'] + '","SK":"' \
                       + self.__config_dict['mdb_password_or_s3sk'] + '","FORCEUSEDIR":1}'
            elif get_project_condition_boolean(
                    self.project_id, 'eBackup_MSTU_NFS'):
                path = self.__config_dict['mdb_path']
                data = '{"FILESYSTEMTYPE":"2","PATH":"' + \
                       self.__config_dict['mdb_path'] + \
                       '","PROTOCOL":"0","SUBDIR":"' + dir_name + \
                       '","FORCEUSEDIR":1}'

            if not self.config_data_backup(url, data):
                return
            self.config_manage_data_backup(path, dir_name)

    def get_sftp_data(self):
        if self.if_ipv6 == 1:
            pathinfo = self.__config_dict['mdb_path'].split('/')
            path_1 = pathinfo[0] + self.__config_dict['mdb_port']
            path = path_1 + '/' + pathinfo[1]
            data = '{"FILESYSTEMTYPE":"9","PATH":"' + path + \
                   '","USERNAME":"' + self.__config_dict[
                       'mdb_user_or_s3ak'] + '","PASSWORD":"' + \
                   self.__config_dict['mdb_password_or_s3sk'] + \
                   '","FORCEUSEDIR":1}'
        else:
            pathinfo = self.__config_dict['mdb_path'].split(':')
            path_1 = pathinfo[0] + ":" + self.__config_dict['mdb_port']
            path = path_1 + pathinfo[1]

            data = '{"FILESYSTEMTYPE":"9","PATH":"' + \
                   path + '","USERNAME":"' + \
                   self.__config_dict['mdb_user_or_s3ak'] + \
                   '","PASSWORD":"' + \
                   self.__config_dict['mdb_password_or_s3sk'] + \
                   '","FORCEUSEDIR":1}'
        return path, data

    def get_ftp_data(self, dir_name):
        if self.if_ipv6 == 1:
            pathinfo = self.__config_dict['mdb_path'].split('/')
            path_1 = pathinfo[0] + self.__config_dict['mdb_port']
            path = path_1 + '/' + pathinfo[1]
            data = '{"FILESYSTEMTYPE":"8","PATH":"' + \
                   path + '","PROTOCOL":"1","SUBDIR":"' + \
                   dir_name + '","USERNAME":"' + \
                   self.__config_dict['mdb_user_or_s3ak'] + \
                   '","PASSWORD":"' + \
                   self.__config_dict['mdb_password_or_s3sk'] + \
                   '","FORCEUSEDIR":1}'
        else:
            pathinfo = self.__config_dict['mdb_path'].split(':')
            path_1 = pathinfo[0] + ":" + self.__config_dict['mdb_port']
            path = path_1 + pathinfo[1]
            data = '{"FILESYSTEMTYPE":"8","PATH":"' + path + \
                   '","PROTOCOL":"1","SUBDIR":"' + dir_name + \
                   '","USERNAME":"' + \
                   self.__config_dict['mdb_user_or_s3ak'] + \
                   '","PASSWORD":"' + \
                   self.__config_dict['mdb_password_or_s3sk'] + \
                   '","FORCEUSEDIR":1}'
        return path, data

    def get_ftps_data(self, dir_name):
        if self.if_ipv6 == 1:
            pathinfo = self.__config_dict['mdb_path'].split('/')
            path = pathinfo[0] + self.__config_dict[
                'mdb_port'] + '/' + pathinfo[1]
            data = '{"FILESYSTEMTYPE":"8","PATH":"' + path + \
                   '","PROTOCOL":"0","SUBDIR":"' + dir_name + \
                   '","USERNAME":"' + \
                   self.__config_dict['mdb_user_or_s3ak'] + \
                   '","PASSWORD":"' + self.__config_dict[
                       'mdb_password_or_s3sk'] + '","FORCEUSEDIR":1}'
        else:
            pathinfo = self.__config_dict['mdb_path'].split(':')
            path_1 = pathinfo[0] + ":" + self.__config_dict['mdb_port']
            path = path_1 + pathinfo[1]
            data = '{"FILESYSTEMTYPE":"8","PATH":"' + path + \
                   '","PROTOCOL":"0","SUBDIR":"' + dir_name + \
                   '","USERNAME":"' + \
                   self.__config_dict['mdb_user_or_s3ak'] + \
                   '","PASSWORD":"' + self.__config_dict[
                       'mdb_password_or_s3sk'] + '","FORCEUSEDIR":1}'
        return path, data

    def get_dir_name(self):
        if len(self.datamover_iplist) == 1:
            dir_name = 'Datamover_' + \
                       self.datamover_iplist[0].replace('.', '_')
        else:
            management_float_ip_lower = \
                self.__config_dict['datamover_management_float_ip'].lower()
            dir_name = 'Datamover_' + \
                       management_float_ip_lower.replace('.', '_')
        dir_name = dir_name.replace(':', '_')
        return dir_name

    def config_data_backup(self, url, data):
        rsp = self.rest_api.put(url, self.headers, data)
        if rsp.status_code != 200:
            logger.error("rest req return not 200.")
            self.update_error_info(653020, '')
            return False
        errcode = rsp.json()["error"]["code"]
        description = rsp.json()["error"]["description"]
        if errcode is None or description is None:
            logger.error("errcode or description is empty.")
            self.update_error_info(653019, '')
            return False
        if errcode != 0:
            if errcode == TIME_OUT_CODE:  # session is invalid
                self.update_error_info(TIME_OUT_CODE, description)
                return False
            logger.error(
                "config manage data backup failed,"
                "description:" + description)
            self.update_error_info(653023, description)
            return False
        return True

    def config_manage_data_backup(self, path, dir_name):
        req_url = self.base_url + '/vbackup_dbbackup_storage_conf/' \
                                  'get_dbbackup_storage_conf'
        timeout = 0
        while timeout < 60:
            timeout += 1
            rsp = self.rest_api.get(req_url, self.headers)
            if rsp.status_code != 200:
                logger.error("rest req return not 200.")
                continue
            errcode = rsp.json()["error"]["code"]
            description = rsp.json()["error"]["description"]
            if errcode is None or description is None:
                logger.error("errcode or description is empty.")
                self.update_error_info(653019, '')
                return

            if errcode != 0:
                if errcode == TIME_OUT_CODE:  # session is invalid
                    self.update_error_info(TIME_OUT_CODE, description)
                    return
                logger.error("errcode is not 0,description:" + description)
                continue
            sub_dir = rsp.json()["data"]["SUBDIR"]
            file_path = rsp.json()["data"]["PATH"]
            if get_project_condition_boolean(
                    self.project_id, 'eBackup_MSTU_SFTP'):
                if file_path == path:
                    logger.info("config manage data backup succ.")
                    return
            else:
                if sub_dir == dir_name and file_path == path:
                    logger.info("config manage data backup succ.")
                    return
            time.sleep(5)
        logger.error("config manage data backup time out.")
        self.update_error_info(653024, '')

    def config_time_zone(self):
        logger.info("begin to config timezone.")
        url = self.base_url + '/system_timezone'
        data = '{"CMO_SYS_TIME_ZONE_NAME":"' + \
               self.__config_dict['timezone'] + '"}'
        rsp = self.rest_api.put(url, self.headers, data)
        if rsp.status_code != 200:
            logger.error("rest req return not 200.")
            self.update_error_info(653020, '')
            return
        errcode = rsp.json()["error"]["code"]
        description = rsp.json()["error"]["description"]
        if errcode is None or description is None:
            logger.error("errcode or description is empty.")
            self.update_error_info(653019, '')
            return
        if errcode != 0:
            logger.error("add timezone failed,description:" + description)
            self.update_error_info(653026, description)
            return
        logger.info("config time zone succ.")

    def config_ntp(self):
        logger.info("begin to config ntp.")
        url = self.base_url + '/ntp_client_config'
        dns = DNSApi()
        dns_type = DNSNodeType()
        dns_ips = dns.get_dns_ips(dns_type.OM, self.pod_id)
        if dns_ips and dns_ips.__len__() == 2:
            self.__config_dict['dns_ip'] = dns_ips[0] + "," + dns_ips[1]
            self.__config_dict['ntp_ip'] = dns_ips[0] + "," + dns_ips[1]
        else:
            self.update_error_info(653025, '')
            return

        data = '{"CMO_SYS_NTP_CLNT_CONF_SERVER_IP":"' + dns_ips[0] + \
               '","CMO_SYS_NTP_CLNT_CONF_SERVER_IP2":"' + dns_ips[
                   1] + '"}'
        rsp = self.rest_api.put(url, self.headers, data)
        if rsp.status_code != 200:
            logger.error("rest req return not 200.")
            self.update_error_info(653020, '')
            return
        errcode = rsp.json()["error"]["code"]
        description = rsp.json()["error"]["description"]
        if errcode is None or description is None:
            logger.error("errcode or description is empty.")
            self.update_error_info(653019, '')
            return
        if errcode != 0:
            logger.error("add ntp failed,description:" + description)
            self.update_error_info(653027, description)
            return
        logger.info("config ntp succ.")

    def construct_req_data(self, level, storage_unit_type, path):
        now = int(time.time())
        local_time = time.localtime(now)
        time_str = time.strftime("%Y%m%d%H%M%S", local_time)
        name = ''
        if level == '1':
            name = "BackupStorage_" + time_str
        if level == '2':
            name = "CopyStorage_" + time_str
        filesystem_type = ''
        if storage_unit_type == 'eBackup_BSTU_NFS':
            filesystem_type = '2'
        elif storage_unit_type == 'eBackup_BSTU_CIFS':
            filesystem_type = '1'
        elif storage_unit_type == 'eBackup_BSTU_S3':
            filesystem_type = '7'

        username = ''
        sk_passwd = ''
        if level == '1':
            if storage_unit_type == 'eBackup_BSTU_S3':
                username = self.__config_dict['s3_storageunit_ak']
                sk_passwd = self.__config_dict['s3_storageunit_sk']
            elif storage_unit_type == 'eBackup_BSTU_CIFS':
                username = self.__config_dict['cifs_storageunit_username']
                sk_passwd = self.__config_dict['cifs_storageunit_password']

        data = ''
        if storage_unit_type == 'eBackup_BSTU_NFS':
            data = '{"NAME":"' + name + \
                   '","DESCRIPTION":"","FILESYSTEMTYPE":"' + \
                   filesystem_type + '","LEVEL":"' + level + \
                   '","PATH":"' + path + '"}'
        elif storage_unit_type == 'eBackup_BSTU_S3':
            data = '{"NAME":"' + name + \
                   '","DESCRIPTION":"","FILESYSTEMTYPE":"' + \
                   filesystem_type + '","LEVEL":"' + level + \
                   '","PATH":"' + path + '","USERNAME":"' + username + \
                   '","PASSWORD":"' + sk_passwd + '","PROTOCOL":"0"}'
        elif storage_unit_type == 'eBackup_BSTU_CIFS':
            if username == '':
                data = '{"NAME":"' + name + \
                       '","DESCRIPTION":"","FILESYSTEMTYPE":"' + \
                       filesystem_type + '","LEVEL":"' + level + \
                       '","PATH":"' + path + '","USERNAME":""}'
            else:
                data = '{"NAME":"' + name + \
                       '","DESCRIPTION":"","FILESYSTEMTYPE":"' + \
                       filesystem_type + '","LEVEL":"' + level + \
                       '","PATH":"' + path + '","USERNAME":"' + \
                       username + '","PASSWORD":"' + sk_passwd + '"}'
        return data

    def is_existed_storage_unit(self, path):
        url = self.base_url + '/vbackup_storageunit'
        rsp = self.rest_api.get(url, self.headers)
        if rsp.status_code != 200:
            logger.error("rest req return not 200.")
            self.update_error_info(653020, '')
            return False, False
        errcode = rsp.json()["error"]["code"]
        description = rsp.json()["error"]["description"]
        if errcode is None or description is None:
            logger.error("errcode or description is empty.")
            self.update_error_info(653019, '')
            return False, False
        if errcode != 0:
            logger.error(
                "query storage unit status failed,description:" + description)
            self.update_error_info(653034, description)
            return False, False

        storage_unit_list = rsp.json()["data"]
        for sub in storage_unit_list:
            if sub['PATH'] == path:
                logger.info("storage unit path:%s, status:%s" % (
                    path, sub['RUNNINGSTATUS']))
                return True, True
        return True, False

    def add_storage_unit(self, data):
        url = self.base_url + '/vbackup_storageunit'
        rsp = self.rest_api.post(url, self.headers, data)
        if rsp.status_code != 200:
            logger.error("rest req return not 200.")
            self.update_error_info(653020, '')
            return
        errcode = rsp.json()["error"]["code"]
        description = rsp.json()["error"]["description"]
        if errcode is None or description is None:
            logger.error("errcode or description is empty.")
            self.update_error_info(653019, '')
            return
        if errcode != 0:
            logger.error("add storage unit failed,description:" + description)
            self.update_error_info(653042, description)
            return
        logger.info("exec add storage unit cmd succ.")
        return

    def config_one_storage_unit(self, level, storage_unit_type, path):
        is_succ, is_exist = self.is_existed_storage_unit(path)
        if is_succ and not is_exist:
            data = self.construct_req_data(level, storage_unit_type, path)
            self.add_storage_unit(data)

    def config_storage_unit(self):
        logger.info("begin to config storage unit.")
        backup_path = self.__config_dict['backup_storageunit_path']
        if backup_path != '' and self.backup_storageunit_type != 'not_BSTU':
            self.config_one_storage_unit('1', self.backup_storageunit_type, backup_path)

    def get_accounts(self):
        self.__config_dict['dmk_floatIp'] = self.params.get_param_value(
            self.pod_id, 'DMK', 'dmk_floatIp', 'dmk_floatIp')
        self.__config_dict['dmk_ui_password'] = self.params.get_param_value(
            self.pod_id, 'DMK', 'dmk_ui_password', 'dmk_ui_password')
        API.login_dmk(
            self.__config_dict['dmk_floatIp'],
            self.__config_dict['eBackup_dmk_user'],
            self.__config_dict['eBackup_dmk_password'])
        account1 = API.get_dmk_account_id(
            self.__config_dict['eBackup_dmk_user'],
            self.__config_dict['hcp_ssh_user']
        )
        return account1

    def init_config_file(self):
        logger.info("policy route: 1.enter get policy route config step.")
        # 获取安装时候的配置模板信息
        env = Environment(loader=FileSystemLoader(API.SCRIPT_PATH + "/conf"))
        env.trim_blocks = True
        env.lstrip_blocks = True
        policy_route_config_template = \
            env.get_template('eBackup_policy_route_config')
        policy_route_host_template = \
            env.get_template('eBackup_policy_route_host')

        # 装备config块
        config_val, host_val = self.assembly_dmk_conf()
        logger.info("policy route: 2.finish reflect the policy route params.")

        # 渲染config文件
        render_var_config = policy_route_config_template.render(config_val)
        render_host_config = policy_route_host_template.render(host_val)
        dmk_config_content = {}
        dmk_config_content.update(
            {'policy_route_var_config': render_var_config})
        dmk_config_content.update(
            {'policy_route_host_config': render_host_config})
        logger.info("policy route: 3.end  prepare the policy route params.")
        return dmk_config_content

    def assembly_dmk_conf(self):
        ebk_backupstorage_plane_gw = \
            self.__config_dict['datamover_backupstorage_gateway']
        config_val = {
            "ebk_backupstorage_plane_gw": ebk_backupstorage_plane_gw
        }
        host_info = self.__config_dict['datamover_externalom_iplist']
        tmp_host_dict = host_info.split(',')
        host_val = {
            "eBackup_host": tmp_host_dict
        }
        return config_val, host_val

    def set_policy_route(self):
        dmk_config_content = self.init_config_file()
        logger.info("init config of policy route succ.")
        dmk_hcp_account_id = self.get_accounts()
        _, ebackup_pkg_name = API.find_file(
            "OceanStor", "eBackup_DMK.zip")
        if ebackup_pkg_name == "":
            self.update_error_info(653004, ebackup_pkg_name)

        logger.info("query pkg name and path succ.")
        ebackup_package_version = API.get_package_version(ebackup_pkg_name)
        ebackup_host = dmk_config_content['policy_route_host_config']
        ebackup_config = dmk_config_content['policy_route_var_config']

        # do_action
        action = "[common]config BCManager"
        logger.info("begin to set policy route")
        dmk_task_info = DmkTaskInfo("eBackup", ebackup_package_version, action, [ebackup_host, ebackup_config])
        is_ok = API.executedmkdeployment(dmk_task_info=dmk_task_info, to_run="true", account_id=dmk_hcp_account_id)
        if is_ok:
            logger.info("set policy route succ.")
        else:
            logger.error("set policy route failed.")
            self.update_error_info(653118, action)

    def logout(self):
        if self.token == '' or self.session_content == '':
            return
        url = self.base_url + '/sessions'
        self.rest_api.delete(url, self.headers)
        logger.info("log out succ.")

# the code has been updated to python3.7
