# -*- coding:utf-8 -*-
from jinja2 import Environment, FileSystemLoader

from utils.PasswordManager.PasswordManager import FSPasswordManager
from utils.business.OpenStack.cps_util import HostsInfoByPod
from utils.business.param_util import ParamUtil
from utils.business.project_condition_utils import get_project_condition_boolean
from utils.business.project_condition_utils import get_project_conditions
from utils.common import log as logger
from utils.common.exception import HCCIException
from utils.common.fic_base import StepBaseInterface
from utils.common.message import Message
from utils.common.ssh_util import Ssh

from plugins.eBackup.common.api_adapter import API
from plugins.eBackup.common.model import DmkUserInfo, DmkTaskInfo
from plugins.eBackup.common.util import Utils


class DeployeBackupDriverWithDMK(StepBaseInterface):
    def __init__(self, project_id, pod_id):
        logger.init("eBackup")
        self.pod_id = pod_id
        self.project_id = project_id
        self.params = ParamUtil()
        self.__db_param_dict = None
        self.fsp_node_list = None
        self.dmk_config_content = {}
        self.manage_float_ip = ""
        self.externalom_ip_list = ""

    def pre_check(self, project_id, pod_id):
        """
        插件内部接口：执行安装前的资源预检查，该接口由execute接口调用，工具框架不会直接调用此接口。
        :param project_id:
        :param pod_id:
        :return:
        """
        pass

    def main_handle(self, project_id, pod_id):
        self.__db_param_dict = self.params.get_service_cloud_param(pod_id, "eBackup")
        if get_project_condition_boolean(project_id, 'eBackup_Proxy'):
            logger.info("no need to deploy driver")
        else:
            self.exec_dmk_operate()
        return Message(200)

    def execute(self, project_id, pod_id):
        """
        标准调用接口：执行安装前预检查&安装&配置
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        try:
            return self.main_handle(project_id, pod_id)
        except HCCIException as e:
            return Message(500, e)
        except Exception as e:
            return Message(500, e, e)

    def init_driver_config_file(self):
        logger.info("begin to init dmk config file for driver.")
        logger.info("get param info succ from db.")
        # 获取安装时候的配置模板信息
        env = Environment(loader=FileSystemLoader(API.SCRIPT_PATH + "/conf"))
        env.trim_blocks = True
        env.lstrip_blocks = True
        driver_config_template = env.get_template('eBackup_config')
        driver_host_template = env.get_template('eBackup_host')

        # 装备config块
        config_val, host_val = self.assembly_dmk_conf()

        # 渲染config文件
        render_var_config = driver_config_template.render(config_val)
        render_host_config = driver_host_template.render(host_val)
        dmk_config_content = {}
        dmk_config_content.update(
            {'vars_content': render_var_config})
        dmk_config_content.update(
            {'hosts_content': render_host_config})
        logger.info("init dmk config file for driver succ.")
        return dmk_config_content

    def assembly_dmk_conf(self):
        logger.info("begin to get fsp node info.")
        fsp_node_info = HostsInfoByPod(self.project_id, self.pod_id)
        self.fsp_node_list = fsp_node_info.get_ebk_driver_hosts_info()
        if len(self.fsp_node_list) == 0:
            logger.error("fsp node info is empty.")
            raise HCCIException(653045)
        logger.info("get fsp node info succ.")
        az_str = self.get_az()
        tmpip_list = self.externalom_ip_list.split(',')
        if len(tmpip_list) == 1:
            datamover_portal_ip = tmpip_list[0]
        else:
            datamover_portal_ip = self.manage_float_ip

        nfs_datalayout = ''
        cifs_datalayout = ''
        s3_datalayout = ''
        ebk_configmode = '0'
        # 默认配置数据布局为2，需要特殊不许安装后修改
        if get_project_condition_boolean(self.project_id, 'eBackup_BSTU_CIFS'):
            cifs_datalayout = '2'
        if get_project_condition_boolean(self.project_id, 'eBackup_BSTU_S3'):
            s3_datalayout = '2'
        if get_project_condition_boolean(self.project_id, 'eBackup_BSTU_NFS'):
            nfs_datalayout = '2'
        if get_project_condition_boolean(self.project_id, 'ExpansionServiceStorage') or \
                get_project_condition_boolean(self.project_id, 'ExpansionComputeRes_KVMNode'):
            ebk_configmode = '1'
        fsp_env_pwd = FSPasswordManager(self.pod_id).get_cps_admin_password()
        if get_project_condition_boolean(self.project_id, 'ExpansionServiceStorage') or get_project_condition_boolean(
                self.project_id, 'ExpansionComputeRes_KVMNode'):

            s3_datalayout = self.get_data_layout(az_str, fsp_env_pwd, "uds")
            nfs_datalayout = self.get_data_layout(az_str, fsp_env_pwd, "nfs")

        config_val = {
            "ebk_configMode": ebk_configmode,
            "ebk_IfConfigDriver": "1",
            "ebk_Ipaddr": datamover_portal_ip,
            "ebk_Azs": az_str,
            "ebk_CifsDataLayout": cifs_datalayout,
            "ebk_S3DataLayout": s3_datalayout,
            "ebk_NfsDataLayout": nfs_datalayout,
            "ebk_PasswordUsr": self.__db_param_dict['eBackup_azconfig_apipwd'],
            "ebk_PasswordEnv": fsp_env_pwd,
            "ebk_CertFilePath": self.get_cert_file_path()
        }
        host_val = {
            "eBackup_host": self.fsp_node_list
        }
        return config_val, host_val

    def get_cert_file_path(self):
        vision2 = '/usr/lib/python2.7/site-packages/cinder/backup'
        vision3 = '/usr/lib/python3.7/site-packages/cinder/backup'
        result = self.ssh_connect(
            ip=self.get_openstack_params('openstack_reverse_proxy_ip'),
            passwd=self.get_openstack_params('openstack_fsp_password'),
            cmd='ls %s' % vision3,
        )
        cert_file_path = vision3 if '0' in result[-2] else vision2
        return cert_file_path + '/cacert.pem'

    def get_openstack_params(self, params):

        if get_project_condition_boolean(self.project_id, "ProjectDeploy"):
            params = params.replace('openstack_', '')
            result = self.params.get_param_value(
                self.pod_id, 'Openstack', params)
        else:
            result = self.params.get_param_value(self.pod_id, '', params)
        return result

    @staticmethod
    def ssh_connect(ip, passwd, cmd, user='fsp'):
        ssh_obj = Ssh()
        ssh_client = ssh_obj.ssh_create_client(ip, user, passwd)
        return ssh_obj.ssh_exec_command_return(ssh_client, cmd)

    def get_az(self):
        condition_dic = get_project_conditions(self.project_id)
        is_bc_install = condition_dic.get('BCInstalled', '')

        self.externalom_ip_list = \
            self.__db_param_dict['datamover_externalom_iplist'].lower()
        self.manage_float_ip = \
            self.__db_param_dict['datamover_management_float_ip'].lower()

        if is_bc_install:
            self.manage_float_ip = \
                self.__db_param_dict['existed_datamover_mg_float_ip']
            self.externalom_ip_list = \
                self.__db_param_dict['existed_datamover_externalom_iplist'] \
                    .lower()
            if get_project_condition_boolean(
                    self.project_id, 'ExpansionComputeRes_KVMNode'):
                az_str = self.params.get_param_value(
                    self.pod_id, "OpenStack", "openstack_az", '')
                az_str = az_str.replace(' ', '')
                az_str = az_str.replace(',', '|')
            else:
                az_str = self.params.get_param_value(
                    self.pod_id, "OpenStack", "openstack_expansion_az", '')
                az_str = az_str.replace(' ', '')
                az_str = az_str.replace(',', '|')
        else:
            az_str = \
                self.__db_param_dict['az_related_to_datamover'].replace(' ',
                                                                        '')
            az_str = az_str.replace(',', '|')
        return az_str

    def rollback(self, project_id, pod_id):
        """
        标准调用接口：执行回滚
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        return Message()

    def retry(self, project_id, pod_id):
        return self.execute(project_id, pod_id)

    def check(self, project_id, pod_id):
        """
        标准调用接口：重试
        :param project_id:
        :param pod_id:
        :return:
        """
        pass

    def get_data_layout(self, az_name, fsp_env_pwd, layout_type):
        logger.info(f"try to get data layout for {az_name}")
        if layout_type not in ("nfs", "uds"):
            logger.warn(f"layout type is invalid.")
            return ""
        fsp_ip = self.get_openstack_params('openstack_reverse_proxy_ip')
        fsp_pwd = self.params.get_param_value(self.pod_id, "OpenStack", 'fsp_password', 'openstack_fsp_password')
        fsp_root_pwd = self.params.get_param_value(self.pod_id, "OpenStack", 'root_password', 'openstack_root_password')

        ssh_client = Utils.get_ssh_client(fsp_ip, 'fsp', fsp_pwd, fsp_root_pwd)
        Utils.login_openstack(fsp_env_pwd, ssh_client)
        az_list = az_name.split('|')
        cmd = "temp_list=(`cps template-list | grep cinder-backup- | awk '{print $4}'`);" \
              "for temp_name in \"${temp_list[@]}\"; do output=`cps template-ext-params-show " \
              "--service cinder ${temp_name}`;echo \"${output}\"  | grep -q %s && echo \"${output}\" " \
              "| grep --color=never %s_data_layout ;done"
        cmd_list = [cmd % (az_id, layout_type) for az_id in az_list]
        cmd_list.append('echo "result: $?"')
        result = Ssh.ssh_exec_command_return_list(ssh_client, " && ".join(cmd_list))
        if str(result).find("result: 0") == -1:
            logger.error(f"exec get data layout command failed. stdout: {str(result)}")
            return ""

        for line in result:
            if "_data_layout" not in line:
                continue
            stdout_list = line.split("|")
            if len(stdout_list) != 4:
                logger.warn(f"get data layout result from output failed. output: {line}.")
                continue
            return stdout_list[2].strip()
        return ""

    def get_fsp_account_info(self):
        self.__db_param_dict['fsp_root_pwd'] = self.params.get_param_value(
            self.pod_id, "OpenStack", 'root_password',
            'openstack_root_password')
        self.__db_param_dict['fsp_user_pwd'] = self.params.get_param_value(
            self.pod_id, "OpenStack", 'fsp_password', 'openstack_fsp_password')
        self.__db_param_dict['fsp_env_pwd'] = FSPasswordManager(self.pod_id). \
            get_cps_admin_password()

    def get_accounts(self):
        self.get_fsp_account_info()
        self.__db_param_dict['dmk_floatIp'] = self.params.get_param_value(
            self.pod_id, "DMK", 'dmk_floatIp', 'dmk_floatIp')
        API.login_dmk(
            self.__db_param_dict['dmk_floatIp'],
            self.__db_param_dict['eBackup_dmk_user'],
            self.__db_param_dict['eBackup_dmk_password'])
        team_id = API.get_dmk_team_id("eBackup")
        team_list = ["eBackup", "OTD"]
        try:
            account1 = API.get_dmk_account_id(
                self.__db_param_dict['eBackup_dmk_user'], "fsp")
        except Exception as e:
            logger.info("query account faild, the reason %s" % e)
            account1 = API.addaccounttodmk(
                "fsp", team_id, self.__db_param_dict['fsp_user_pwd'],
                self.__db_param_dict['fsp_root_pwd'])
            dmkinfo = DmkUserInfo(
                self.__db_param_dict['eBackup_dmk_user'], "fsp", team_list,
                [self.__db_param_dict['fsp_user_pwd'], self.__db_param_dict['fsp_root_pwd']]
            )
            bret = API.update_account_to_team(dmkinfo)
            if not bret:
                logger.error("eBackup,update fsp account to multi team failed.")
                raise HCCIException(653113, "fsp") from e
            return account1
        try:
            API.validate_account("fsp", account1, self.fsp_node_list[0], "su")
            return account1
        except HCCIException as msg:
            logger.error('method "validate_account" raise an error, the message is "%s".' % msg)
            msg_str = str(msg)
            checked_str = "ssh to " + self.fsp_node_list[0] + " by fsp failed"
            if checked_str in msg_str:
                API.delete_dmk_account(account1)
            else:
                raise msg
            team_list = ["eBackup", "OTD"]
            account1 = API.addaccounttodmk(
                "fsp", team_id, self.__db_param_dict['fsp_user_pwd'],
                self.__db_param_dict['fsp_root_pwd'])
            dmkinfo = DmkUserInfo(
                self.__db_param_dict['eBackup_dmk_user'], "fsp", team_list,
                [self.__db_param_dict['fsp_user_pwd'], self.__db_param_dict['fsp_root_pwd']]
            )
            bret = API.update_account_to_team(dmkinfo)
            if not bret:
                logger.error("update fsp account to multi team failed.")
                raise HCCIException(653113, "fsp") from msg
            return account1

    def exec_dmk_operate(self):
        dmk_config_content = self.init_driver_config_file()
        driver_config = dmk_config_content['vars_content']
        driver_host = dmk_config_content['hosts_content']

        dmk_fsp_account_id = self.get_accounts()
        logger.info("get fsp account info succ.")
        _, ebackup_pkg_name = API.find_file(
            "OceanStor", "eBackup_DMK.zip")
        if ebackup_pkg_name == "":
            raise HCCIException(653004, ebackup_pkg_name)
        ebackup_package_version = API.get_package_version(ebackup_pkg_name)
        logger.info("get ebackup package version info succ.")

        API.login_dmk(
            self.__db_param_dict['dmk_floatIp'],
            self.__db_param_dict['eBackup_dmk_user'],
            self.__db_param_dict['eBackup_dmk_password'])

        action = "[install]4.Install and configure eBackup Driver"
        logger.info("begin to deploy eBackup driver.")
        dmk_task_info = DmkTaskInfo("eBackup", ebackup_package_version, action, [driver_host, driver_config])
        is_ok = API.executedmkdeployment(dmk_task_info=dmk_task_info, to_run="true", account_id=dmk_fsp_account_id)
        if is_ok:
            logger.info("Install and configure eBackup driver success.")
        else:
            logger.error("Install and configure eBackup dirver failed.")
            raise HCCIException(653118, 'driver')

# the code has been updated to python3.7
