# -*- coding:utf-8 -*-
import time
import utils.common.log as logger
from utils.business.manageone_util2 import ManageOneUtil2
from utils.business.param_util import ParamUtil
from utils.common.exception import HCCIException
from utils.common.fic_base import StepBaseInterface
from utils.common.message import Message

from plugins.eBackup.common.ebackup_util import EBackupUtil
from plugins.eBackup.common.param_tool import ParamTool
from plugins.eBackup.scripts.common.ebackup_util import CustomParametersUtil
from plugins.eBackup.common.ebackup_rest import EbackupRest

logger.init("eBackup connect manageone for Datamover")


class ConfigOC(StepBaseInterface):
    def __init__(self, project_id, pod_id):
        super().__init__(project_id, pod_id)
        self.project_id = project_id
        self.pod_id = pod_id
        self.err_code = 0
        self.description = None
        self.param_tool = ParamTool(self.project_id, self.pod_id)
        self.ebackup = EBackupUtil(project_id, pod_id)
        self.param_util = ParamUtil()
        self.manageone = ManageOneUtil2()
        self.rest_api = EbackupRest()
        self.custom_pararms = CustomParametersUtil(project_id, pod_id)
        self.management_float_ip = self.ebackup.get_management_float_ip()
        self.externalom_iplist = self.ebackup.get_externalom_iplist()
        self.config_dict = self.param_util.get_service_cloud_param(pod_id, "eBackup")

    def pre_check(self, project_id, pod_id):
        """
        插件内部接口：执行安装前的资源预检查，该接口由execute接口调用，工具框架不会直接调用此接口。
        :param project_id:
        :param pod_id:
        :return:
        """
        pass

    def execute(self, project_id, pod_id):
        """
        标准调用接口：执行安装前预检查&安装&配置
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        try:
            base_url, headers = self.exec_config_oc()
        except HCCIException as e:
            return Message(500, e)
        except Exception as e:
            logger.error("Failed to connect ManageOne(datamover) ,the reason is %s" % e)
            return Message(500, e, e)

        if base_url == "" and headers == "":
            return Message(200)
        self.ebackup.logout_portal(base_url, headers)
        if self.err_code != 0:
            logger.error("Config OC for datamover failed.")
            if self.description == '':
                raise HCCIException(self.err_code)
            else:
                raise HCCIException(self.err_code, self.description)
        return Message(200)

    def exec_config_oc(self):
        if self.param_tool.is_expand_ebackup_proxy():
            return "", ""
        base_url = self.ebackup.get_server_url(self.management_float_ip)
        portal_account = self.ebackup.get_portal_account(self.management_float_ip)
        _, headers = self.ebackup.login_portal(base_url, portal_account)
        self.config_alarm_report(base_url, headers)
        return base_url, headers

    def rollback(self, project_id, pod_id):
        """
        标准调用接口：执行回滚
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        return Message()

    def retry(self, project_id, pod_id):
        """
        标准调用接口：重试
        :return: Message类对象
        """
        return self.execute(project_id, pod_id)

    def check(self, project_id, pod_id):
        """
        标准调用接口：重试
        :param project_id:
        :param pod_id:
        :return:
        """
        pass

    def update_error_info(self, err, des):
        if self.err_code == 0:
            self.err_code = err
            self.description = des

    def add_white_list(self):
        logger.info("Add XAAS whitelist(ebackup_server) start.")
        logic_region = self.config_dict.get('region_display_name')
        is_ok = self.manageone.addXaasWhiteList(pod_id=self.pod_id, sysName="eBackup", ipList=self.externalom_iplist,
                                                region="", groupType="", logicRegion=logic_region, restProtocol="1",
                                                snmpProtocol="", floatIp=self.management_float_ip)
        if not is_ok:
            logger.error("Add XAAS whitelist(ebackup_server) failed.")
            raise HCCIException(653010, '')
        logger.info("Add XAAS whitelist succeeded.")

    def config_alarm_report(self, base_url, headers):
        logger.info("Begin to config alarm report.")
        self.add_white_list()

        url = base_url + '/alarm_settings/default'
        alarm_user, alarm_pwd = self.manageone.getThirdpartyInfor(self.pod_id)[:2]
        if not alarm_user:
            logger.error("Alarm user is empty.")
            raise HCCIException(653039, "alarm user is empty")
        if not alarm_pwd:
            logger.error("Alarm pwd is empty.")
            raise HCCIException(653039, "alarm pwd is empty")

        oc_addrss = self.custom_pararms.get_oc_domain_name()
        logger.info("Alarm url:" + oc_addrss)
        data = '{"ISALARMON":1,"ALARMSERVICETYPE":"1","ALARMSERVERURL":"' + oc_addrss + '","ALARMUSERNAME":"' + \
               alarm_user + '","ALARMPASSWORD":"' + alarm_pwd + '"}'

        description = ''
        num = 0
        while num < 3:
            rsp = self.rest_api.put(url, headers, data)
            if rsp.status_code != 200:
                logger.error("Rest req return not 200.")
                self.update_error_info(653020, '')
                return
            errcode = rsp.json()["error"]["code"]
            description = rsp.json()["error"]["description"]
            if errcode is None or description is None:
                logger.error("Errcode or description is empty.")
                self.update_error_info(653019, '')
                return
            if errcode != 0:
                logger.error("Config alarm report failed,description:" + description)
                self.update_error_info(653039, description)
                num += 1
                time.sleep(30)
            else:
                logger.info("Config alarm report succeeded.")
                self.err_code = 0
                self.description = None
                return
        logger.error("config alarm report failed, description:" + description)
        self.update_error_info(653039, description)
