import time

import utils.common.log as logger
from utils.business.param_util import ParamUtil
from utils.common.code2Msg import code2Msg

from plugins.CSBS.common.param_check.os_param_check import check_pwd_will_expired
from plugins.CSBS.common.ssh_client import SshClient
from plugins.CSBS.common.upgrade import constant
from plugins.CSBS.common.upgrade.params import ParamsTools
from plugins.CSBS.common.util import to_str


class ParamsChecker(object):
    def __init__(self, project_id):
        self.project_id = project_id
        self.data_to_check = ParamUtil().get_need_check_cloud_params(
            self.project_id, constant.CSBS_SERVICE_NAME)
        self.ssh_client = SshClient()
        self.params_tools = ParamsTools(project_id)

    def check_karbor_login_ability(self, check_result):
        vm_ips_dict = self.params_tools.get_karbor_vm_ips()
        logger.info('Current vms info: {}.'.format(vm_ips_dict))
        karbor_count = 2 if self.params_tools.is_csha_scene else 3
        karbor_name_list = ["karbor_host{}".format(i)
                            for i in range(karbor_count)]
        karbor_ip_list = [vm_ips_dict[name] for name in karbor_name_list]

        check_result = self.check_karbor_djmanager_login_ability(
            check_result,
            karbor_ip_list)

        check_result = self.check_karbor_root_login_ability(
            check_result,
            karbor_ip_list
        )
        return check_result

    def check_karbor_djmanager_login_ability(self, check_result, ip_list):
        djmanager_pwd_err = []
        djmanager_pwd_will_expired = []
        djmanager_pwd_no_ssh_ips = []
        user = constant.KARBOR_USER
        pwd = self.data_to_check["csbs_karbor_djmanager_password"]
        check_list = ["csbs_karbor_djmanager_password"]

        def _get_djmanager_login_check_result():
            check_success_count = 0
            for _ip in ip_list:
                ssh_client = None
                pwd_will_expired_flag = None
                error_flag = False
                try:
                    ssh_client = self.ssh_client.create_ssh_client(_ip, user, pwd)
                except Exception as err:
                    logger.error(
                        f"Failed to login to Karbor node: IP {_ip}, User {constant.KARBOR_USER}, error:{str(err)}")
                    error_flag = True
                    if str(err).lower().find("authentication") != -1:
                        djmanager_pwd_err.append(_ip)
                    else:
                        djmanager_pwd_no_ssh_ips.append(_ip)
                finally:
                    if ssh_client:
                        pwd_will_expired_flag = check_pwd_will_expired(self.ssh_client, ssh_client,
                                                                       constant.KARBOR_USER)
                        self.ssh_client.ssh_close(ssh_client)
                if error_flag:
                    logger.error(f"The {user} account password has expired or password error, please check.")
                elif pwd_will_expired_flag:
                    logger.error(f"Node {_ip}, User {user} password will expire， please change and retry.")
                    djmanager_pwd_will_expired.append(_ip)
                else:
                    check_success_count += 1
            return check_success_count

        check_success_count = _get_djmanager_login_check_result()
        if check_success_count == len(ip_list):
            check_result.set_check_result(param_keys=check_list, status=200, error_msg='')
        else:
            if djmanager_pwd_err:
                logger.error(f"Failed to login to Karbor node: IP {', '.join(djmanager_pwd_err)}, "
                             f"User {constant.KARBOR_USER} passwd is expired or passwd error.")
                error_msg = code2Msg(642002)
            elif djmanager_pwd_will_expired:
                logger.error(f"Node IP {', '.join(djmanager_pwd_will_expired)}, "
                             f"User {constant.KARBOR_USER} password will expire.")
                error_msg = code2Msg(642003)
            else:
                logger.error(f"Failed to login to Karbor node: IP {', '.join(djmanager_pwd_no_ssh_ips)}, "
                             f"User {constant.KARBOR_USER}, please check ip.")
                error_msg = code2Msg(642001)
            check_result.set_check_result(param_keys=check_list,
                                          status=500,
                                          error_msg=error_msg)
        return check_result

    def check_karbor_root_login_ability(self, check_result, ip_list):
        djmanager_pwd = self.data_to_check["csbs_karbor_djmanager_password"]
        root_pwd = self.data_to_check["csbs_karbor_root_password"]
        check_list = ["csbs_karbor_root_password"]
        check_success_count = 0
        karbor_root_no_ssh_ips = []
        for _ip in ip_list:
            ssh_client = None
            try:
                ssh_client = self._check_root_login(djmanager_pwd, _ip, root_pwd)
            except Exception as err:
                logger.error(
                    "Failed to login to Karbor node: IP {}, User {}, "
                    "error: {}".format(_ip, 'root', str(err))
                )
                karbor_root_no_ssh_ips.append(_ip)
            finally:
                if ssh_client:
                    self.ssh_client.ssh_close(ssh_client)
            check_success_count += 1

        if check_success_count == len(ip_list):
            check_result.set_check_result(param_keys=check_list, status=200,
                                          error_msg='')
        else:
            logger.error(
                "Failed to login to Karbor node: IP {}, User {}".format(
                    ','.join(karbor_root_no_ssh_ips), "root")
            )
            check_result.set_check_result(param_keys=check_list,
                                          status=500,
                                          error_msg=code2Msg(642001))
        return check_result

    def _check_root_login(self, djmanager_pwd, _ip, root_pwd):
        ssh_client = self.ssh_client.create_ssh_client(_ip, constant.KARBOR_USER, djmanager_pwd)
        self.ssh_client.ssh_send_command_expect(
            ssh_client, 'sudo su root', 'password for root:', 50)
        self.ssh_client.ssh_send_command_expect(
            ssh_client, root_pwd, '#', 50)
        return ssh_client

    @staticmethod
    def _check_expires_str(ssh_client):
        expire_str = "has expired"
        ssh_client['channel'].send('\n')
        recv_str = ""
        num = 0
        while not (recv_str.endswith("$ ") or recv_str.endswith("# ")):
            num += 1
            time.sleep(1)
            if expire_str in recv_str or num > 10:
                logger.error("Get expired str, received shell prompt: "
                             "{}, in times:{}.".format(recv_str, num))
                return True
            if not ssh_client['channel'].recv_ready():
                continue
            recv_str = recv_str + to_str(ssh_client['channel'].recv(65535))
        logger.info("Test login success, no passwd expired.")
        return False
