import datetime
import os.path
import re

import utils.common.log as logger
from utils.common.exception import HCCIException
from utils.common.ssh_util import Ssh

# 命令返回的标准长度
SSH_RET_LENGTH = 7
EXCEPT_CODE = -3
ERROR_CODE = -2
COMMON_CODE = 0
# 最大天数
MAX_DAY_NUM = 999
EXPIRING_DAY = 7


def _is_upper_830_version(karbor_version):
    return karbor_version >= '8.3.0' and karbor_version != '8.3.RC1'


class SShCmd:
    """
    proxy
    """
    def __init__(self):
        self.ssh_obj = Ssh()

    def put_file_to_root(self, obj, file_path):
        """
        Uploading Files to All root Users
        :param obj:
        :param file_path:
        :return:
        """
        self.ssh_obj.put_file(
            obj.node_ip,
            obj.root_name,
            obj.root_pwd,
            file_path,
            obj.root_path,
            obj.port
        )

    def put_file_to_all_proxys(self, obj, file_path, root_path=''):
        """
        Uploading Files to All Proxy Users
        :param root_path:
        :param obj:
        :param file_path:
        :return:
        """
        if root_path:
            path = root_path
        else:
            path = obj.root_path

        for node in obj.get_proxy_node_list():
            self.ssh_obj.put_file(
                node.node_ip,
                node.user,
                node.user_pwd,
                file_path,
                path,
                obj.port
            )

    def put_file_to_all_roots(self, obj, file_path, root_path=''):
        """
        Uploading Files to All Proxy Users
        :param root_path:
        :param obj:
        :param file_path:
        :return:
        """
        if root_path:
            path = root_path
        else:
            path = obj.root_path

        for node_ip in obj.all_ip_list:
            self.ssh_obj.put_file(
                node_ip,
                obj.root_name,
                obj.root_pwd,
                file_path,
                path,
                obj.port
            )

    def ssh_cmds(self, obj, cmd, expect=None):
        cmd, expect_value = self.get_exe_cmd(cmd)
        result = self.ssh_obj.ssh_cmds(
            obj.node_ip,
            cmd,
            obj.user,
            obj.user_pwd,
            obj.root_pwd,
            "",
            ""
        )
        self.parsing_result(expect_value, result, expect)

    def ssh_exe_root_cmds(self, obj, cmd, expect=None):
        cmd, expect_value = self.get_exe_cmd(cmd)
        ssh_client = self.ssh_obj.ssh_create_client(
            obj.node_ip, "root", obj.root_pwd)

        ssh_result = self.ssh_obj.ssh_exec_command_return(
            ssh_client,
            cmd
        )
        self.ssh_obj.ssh_close(ssh_client)
        self.parsing_result(expect_value, ssh_result, expect)

    @staticmethod
    def get_exe_cmd(cmd):
        result_str = None
        try:
            if ';' in cmd:
                exe_result_cmd = cmd.split(';')[-1]
                match_obj = re.search(
                    r'echo(.*)\$\?', exe_result_cmd, re.M | re.I)
                if match_obj:
                    result_str = exe_result_cmd.replace('echo ', '')
                    result_str = result_str.replace('$?', '0')
        except Exception as err:
            logger.info('[KarborProxy]:' + str(err))
        if not result_str:
            cmd = cmd + '; echo exe cmd result: $?'
            result_str = 'exe cmd result: 0'
        return cmd, result_str

    def parsing_result(self, expect_value, result, expect=None):
        result = [str(x).replace('\n', '').strip() for x in result]
        if expect:
            if isinstance(expect, str):
                expect_value = expect

            if isinstance(expect, list):
                expect = [expect_value.replace('0', x).strip() for x in expect]
                if set(expect) & set(result):
                    logger.info(
                        '[KarborProxy]:' + str(set(expect) & set(result)))
                    self.success_result(expect_value)
                    return
                else:
                    self.fail_result(expect_value)

        if expect_value.strip() in result:
            self.success_result(expect_value)
        else:
            self.fail_result(expect_value)

    @staticmethod
    def success_result(expect_value):
        result_str = expect_value.replace('0', '')
        result_str = result_str + ' success'.strip()
        logger.info('[KarborProxy]:' + result_str)

    @staticmethod
    def fail_result(expect_value):
        result_str = expect_value.replace('0', '')
        result_str = result_str + ' fail'.strip()
        logger.error('[KarborProxy]:' + result_str)
        raise Exception(result_str)

    def ssh_cmds_to_all_proxys(self, obj, cmd, expect=None):
        cmd, expect_value = self.get_exe_cmd(cmd)
        for node in obj.get_proxy_node_list():
            result = self.ssh_obj.ssh_cmds(
                node.node_ip,
                cmd,
                node.user,
                node.user_pwd,
                node.root_pwd,
                "",
                ""
            )
            self.parsing_result(expect_value, result, expect)

    def ssh_cmds_to_proxy(self, node, cmd, expect=None):
        cmd, expect_value = self.get_exe_cmd(cmd)
        result = self.ssh_obj.ssh_cmds(
            node.node_ip,
            cmd,
            node.user,
            node.user_pwd,
            node.root_pwd,
            "",
            ""
        )
        self.parsing_result(expect_value, result, expect)

    def ssh_root_cmds(self, obj, cmd, expect=None):
        cmd, expect_value = self.get_exe_cmd(cmd)
        ssh_client = self.ssh_obj.ssh_create_client(
            obj.node_ip, "root", obj.root_pwd)

        ssh_result = self.ssh_obj.ssh_exec_command_return(
            ssh_client,
            cmd
        )
        self.ssh_obj.ssh_close(ssh_client)
        self.parsing_result(expect_value, ssh_result, expect)

    def ssh_root_cmds_to_all_nodes(self, obj, cmd, expect=None):
        """

        :param obj:
        :param cmd:
        :param expect:
        :return:
        """
        cmd, expect_value = self.get_exe_cmd(cmd)
        for node in obj.get_proxy_node_list():
            ssh_client = self.ssh_obj.ssh_create_client(
                node.node_ip, node.root_name, node.root_pwd)
            ssh_result = self.ssh_obj.ssh_exec_command_return(ssh_client, cmd)
            self.ssh_obj.ssh_close(ssh_client)
            self.parsing_result(expect_value, ssh_result, expect)

    @staticmethod
    def exe_user_to_root(node):
        ssh_client = Ssh.ssh_create_client(
            node.node_ip,
            node.user,
            node.user_pwd
        )
        Ssh.ssh_send_command(ssh_client, "su -", 'Password', 100)
        Ssh.ssh_send_command(ssh_client, node.root_pwd, '#', 100)
        return ssh_client

    def ssh_exe_config_opsvc_cmd(self, agentproxy, cmd):
        for proxy_node in agentproxy.get_proxy_node_list():
            logger.info('Starting to connect to %s .' % proxy_node.node_ip)
            ssh_client = self.exe_user_to_root(proxy_node)
            Ssh.ssh_send_command(ssh_client, cmd, 'password', 100)
            Ssh.ssh_send_command(ssh_client, agentproxy.op_svc_account.account_pwd, 'success', 100)
            Ssh.ssh_close(ssh_client)
            logger.info(
                'The %s OpSvc IAM account information is configured '
                'successfully.' % proxy_node.node_ip)

    def ssh_exe_config_ha_cmd(self, agentproxy, cmd, passwd):
        for proxy_node in [agentproxy.get_proxy_node_list()[0]]:
            logger.info('Starting to connect to %s .' % proxy_node.node_ip)
            ssh_client = self.exe_user_to_root(proxy_node)
            Ssh.ssh_send_command(ssh_client, cmd, 'password', 100)
            Ssh.ssh_send_command(ssh_client, passwd, 'again', 100)
            Ssh.ssh_send_command(ssh_client, passwd, 'success', 100)
            Ssh.ssh_close(ssh_client)
            logger.info(
                'The %s congfig ha account information is configured '
                'successfully.' % proxy_node.node_ip)

    def ssh_exe_config_iam_cmd(self, agentproxy, cmd):
        for proxy_node in agentproxy.get_proxy_node_list():
            logger.info('Starting to connect to %s .' % proxy_node.node_ip)
            ssh_client = self.exe_user_to_root(proxy_node)
            Ssh.ssh_send_command(ssh_client, cmd, 'password', 100)
            Ssh.ssh_send_command(
                ssh_client, agentproxy.op_service_account.account_pwd, 'success', 100)
            Ssh.ssh_close(ssh_client)
            logger.info(
                'The %s IAM account information is configured '
                'successfully.' % proxy_node.node_ip)

    def ssh_exe_config_alarm(self, agentproxy, cmd):
        for proxy_node in agentproxy.get_proxy_node_list():
            logger.info('Starting to connect to %s .' % proxy_node.node_ip)
            ssh_client = self.exe_user_to_root(proxy_node)
            Ssh.ssh_send_command(ssh_client, cmd, 'password', 100)
            Ssh.ssh_send_command(
                ssh_client, agentproxy.alarm_pwd, 'success', 100)
            Ssh.ssh_close(ssh_client)

    def ssh_su_root_cmds_to_karbor(self, karbor_node, cmd):
        ssh_result = self.ssh_obj.ssh_su_root_cmds(
            karbor_node.node_ip,
            cmd,
            karbor_node.user,
            karbor_node.user_pwd,
            karbor_node.root_pwd,
            22,
            ""
        )
        return ssh_result

    def ssh_cmd_to_karbor(self, karbor_node, cmd, expect=None):
        cmd, expect_value = self.get_exe_cmd(cmd)
        ssh_result = self.ssh_su_root_cmds_to_karbor(karbor_node, cmd)
        self.parsing_result(expect_value, ssh_result, expect)
        return ssh_result

    def put_file_to_kabor(self, karbor_node, file_path, root_path='.'):
        result = self.ssh_obj.put_file(
            karbor_node.node_ip,
            karbor_node.user,
            karbor_node.user_pwd,
            file_path,
            root_path,
            22
        )
        file_name = os.path.basename(file_path)
        if result:
            logger.info('File %s uploaded successfully.' % file_name)
        else:
            logger.error('Failed to upload the %s file.' % file_name)
            raise Exception('Failed to upload the %s file.' % file_name)

    def ssh_connect_karbor(self, karbor_node, karbor_version, cmd, col_info):
        try:
            if not karbor_version:
                raise HCCIException('can not get karbor version')
            self._ssh_connect_karbor(karbor_node, karbor_version, cmd, col_info)
        except Exception as err:
            logger.error(err)
            raise HCCIException(640098) from err

    @staticmethod
    def _ssh_connect_karbor(karbor_node, karbor_version, cmd, col_info):
        logger.info('Starting to connect to %s' % karbor_node.node_ip)
        ssh_client = Ssh.ssh_create_client(karbor_node.node_ip, karbor_node.user, karbor_node.user_pwd)
        Ssh.ssh_send_command(ssh_client, "sudo su", 'password', 100)
        Ssh.ssh_send_command(ssh_client, karbor_node.root_pwd, '#', 100)
        logger.info('Start to run the command for connecting to the DPA in Karbor.')
        Ssh.ssh_send_command(ssh_client, cmd, 'security', 100)
        logger.info('Start running the karbor command for connecting to the dpa.')
        Ssh.ssh_send_command(ssh_client, col_info[0], 'password', 100)
        Ssh.ssh_send_command(ssh_client, col_info[1], 'password', 100)
        if _is_upper_830_version(karbor_version):
            Ssh.ssh_send_command(ssh_client, col_info[1], 'ca cert', 100)
            logger.info('Download CA Cert')
            Ssh.ssh_send_command(ssh_client, 'y', 'successfully', 100)
        else:
            Ssh.ssh_send_command(ssh_client, col_info[1], 'general', 100)
        logger.info('Start running the Karbor command for connecting to the DPA.')
        Ssh.ssh_send_command(ssh_client, col_info[2], 'password', 100)
        Ssh.ssh_send_command(ssh_client, col_info[3], 'password', 100)
        if _is_upper_830_version(karbor_version):
            Ssh.ssh_send_command(ssh_client, col_info[3], 'ca cert', 100)
            logger.info('Download CA Cert')
            Ssh.ssh_send_command(ssh_client, 'y', 'successfully', 100)
        else:
            Ssh.ssh_send_command(ssh_client, col_info[3], 'system', 100)
        logger.info('Start running the Karbor command for connecting to the DPA.')
        Ssh.ssh_send_command(ssh_client, col_info[4], 'password', 100)
        Ssh.ssh_send_command(ssh_client, col_info[5], 'password', 100)
        if _is_upper_830_version(karbor_version):
            Ssh.ssh_send_command(ssh_client, col_info[5], 'ca cert', 100)
            logger.info('Download CA Cert')
            Ssh.ssh_send_command(ssh_client, 'y', 'dpa configuration successfully', 100)
        else:
            Ssh.ssh_send_command(ssh_client, col_info[5], 'success', 100)
        Ssh.ssh_close(ssh_client)

    def check_az_available(self, obj):
        for az_name in obj.get_az_related_to_dpa():
            logger.info("begin to check:" + az_name)
            cmd = 'nova availability-zone-list | grep ' + az_name + \
                  ';echo check az available result:$?'
            ssh_client = self.ssh_obj.ssh_create_client(
                obj.get_fsp_node(), "fsp", obj.fsp_user_pwd)
            Ssh.ssh_send_command(ssh_client, 'su', 'Password:', 100)
            Ssh.ssh_send_command(
                ssh_client, obj.fsp_root_pwd, '#', 100)
            Ssh.ssh_send_command(ssh_client, "TMOUT=0", '#', 100)
            Ssh.ssh_send_command(ssh_client, "source set_env", "1|2", 100)
            Ssh.ssh_send_command(ssh_client, '1', 'OS_PASSWORD', 100)
            Ssh.ssh_send_command(
                ssh_client, obj.fsp_env_pwd, '#', 100)
            cmd, expect_value = self.get_exe_cmd(cmd)
            result = Ssh.ssh_exec_command_return_list(
                ssh_client, cmd, timeout=30)
            self.ssh_obj.ssh_close(ssh_client)
            self.parsing_result(expect_value, result)

    def check_node_password_expired(self, node):
        try:
            self._is_node_alive(node)
        except Exception as err:
            logger.error(err)
            raise Exception('Failed to log in to the upper node. '
                            'Check whether the account has expired'
                            ' or the password is incorrect.', node.node_ip) from err
        ssh_client = self.get_server_ssh_client(
            node.node_ip, ssh_user=node.user, ssh_pwd=node.user_pwd,
            sudo_user=node.root_name, sudo_pwd=node.root_pwd)

        return_data = self.check_os_password_expired_day(
            ssh_client, [node.user, node.root_name])
        hcp_expire_day = return_data.get(node.user)
        root_expire_day = return_data.get(node.root_name)
        if hcp_expire_day <= 30 or root_expire_day <= 30:
            raise HCCIException('The node password is about to expire or has'
                                ' expired. Please change it.', node.node_ip)
        logger.info('Node %s Passes the Password Expiration Check ' % node.node_ip)

    def _is_node_alive(self, node):
        ssh_client = self.ssh_obj.ssh_create_client(node.node_ip, node.user, node.user_pwd)
        self.ssh_obj.ssh_close(ssh_client)

    @staticmethod
    def get_server_ssh_client(host, ssh_user=None, ssh_pwd=None,
                              sudo_user=None, sudo_pwd=None):
        ssh_ins = Ssh()
        ssh_client = None
        if ssh_user and ssh_pwd:
            ssh_client = ssh_ins.ssh_create_client(
                host, ssh_user, ssh_pwd, port=22, timeout=60)
            if sudo_user and sudo_pwd:
                ssh_ins.ssh_send_command(
                    ssh_client, "sudo su - {}".format(
                        sudo_user), "password", 20)
                ssh_ins.ssh_send_command(ssh_client, sudo_pwd, "#", 20)
                ssh_ins.ssh_send_command(ssh_client, "TMOUT=0", "#", 20)
            return ssh_client
        logger.error("The user name or password required for "
                     "login is missing.")
        return ssh_client

    def check_os_password_expired_day(self, ssh_client, user_lst):
        return_data = {}
        for user in user_lst:
            cmd = "chage -l {}".format(user)
            output = Ssh.ssh_send_command(ssh_client, cmd, "#", 20)
            output.pop(0)
            output.pop(-1)
            logger.info(
                "Execute command success, result: {}.".format(output))
            if not output or len(output) < SSH_RET_LENGTH:
                err_msg = "Execution command did not get the expected " \
                          "output, please check."
                logger.error(err_msg)
                raise Exception(err_msg)
            code, expires, inactive = self.get_password_expires(
                "".join(output))
            logger.info(
                "Ended to check {} user, result: {}, {}, {}.".format(
                    user, code, expires, inactive))
            if code == ERROR_CODE:
                err_msg = "Execution command did not get the expected " \
                          "output, please check."
                logger.error(err_msg)
                raise Exception(err_msg)
            elif code == EXCEPT_CODE:
                err_msg = \
                    "Analysis command output error, please try again."
                logger.error(err_msg)
                raise Exception(err_msg)
            elif code == 0:
                expire_days = (expires - datetime.datetime.utcnow()).days
                logger.info("The password of the {} user expires "
                            "after {} days.".format(user, expire_days))
                if not expire_days:
                    raise Exception(
                        "Get user: {} expire days failed.".format(user))
                return_data[user] = expire_days
        Ssh.ssh_close(ssh_client)
        return return_data

    @staticmethod
    def get_password_expires(output_list):
        """
        获取密码的过期时长和失效时间
        :param output_list: SSH命令返回结果
        :return:
        """
        # 对输出进行分行处理
        expired_time, inactive_time, expired_str = "", "", ""
        outputs = output_list.splitlines(False)
        if len(output_list) < SSH_RET_LENGTH:
            # 执行命令结果长度不够，错误
            raise Exception(f"Insufficient length of ssh cmd return val, error code:{ERROR_CODE}")
        try:
            for out_put_line in outputs:
                temp_name = out_put_line.split(":")[0].strip()
                temp_value = out_put_line.split(":")[1].strip()
                if temp_name == 'Password expires':
                    expired_str = temp_value
                    expired_time = datetime.datetime.utcnow() + datetime.timedelta(days=MAX_DAY_NUM) \
                        if expired_str == 'never' else datetime.datetime.strptime(expired_str, '%b %d, %Y')
                elif temp_name == 'Password inactive':
                    inactive_str = temp_value
                    inactive_time = datetime.datetime.utcnow() + datetime.timedelta(days=MAX_DAY_NUM) \
                        if inactive_str == 'never' else datetime.datetime.strptime(expired_str, '%b %d, %Y')
        except Exception as error:
            # 执行命令结果解析，错误
            logger.error(f"Failed to parse ssh command results. Error Message: {error}")
            raise Exception(
                f"Failed to parse ssh command results, except code: {EXCEPT_CODE}. Error Message: {error}") from error
        ret_code = COMMON_CODE
        if (not expired_time) or (not inactive_time):
            ret_code = ERROR_CODE
        return ret_code, expired_time, inactive_time
