import json
import time

import requests
import utils.common.log as logger
from utils.business.manageone_cmdb_util import ManageOneCmdbUtil
from utils.business.param_util import ParamUtil
from utils.business.vm_util import PublicServiceOM
from utils.common.message import Message
from utils.common.message import RESULT_CODE

from plugins.CSBS.common.constant import CSBS_SERVICE_NAME
from plugins.CSBS.common.iam_util import IamUtil
from plugins.CSBS.common.karbor import KarborUtil
from plugins.CSBS.common.params_tool import ParamTool
from plugins.CSBS.common.step_base import DeployBaseSubJob

logger.init("CSBS")

RETRY_TIME_LIMIT = 5


class DeleteKarbor3NodeAndInfo(DeployBaseSubJob):
    def __init__(self, project_id, pod_id, regionid_list=None):
        super(DeleteKarbor3NodeAndInfo, self).__init__(project_id, pod_id, regionid_list)
        self.karbor3_vm_name = "Service-CSBS3"
        self.om_util = PublicServiceOM()
        self.iam_util = IamUtil(self.project_id, self.pod_id)
        self.karbor_util = KarborUtil(self.project_id, self.pod_id)

        self.cmdb_util = ManageOneCmdbUtil(self.project_id, self.pod_id)
        self.param_util = ParamUtil()
        self.region_id = self.param_util.get_param_value(
            self.pod_id, "public", "region_id", "region0_id")
        self.param_tool = ParamTool(self.project_id, self.pod_id)

    def execute(self, project_id, pod_id, regionid_list=None):
        try:
            return self._dele_karbor_3node_and_info()
        except Exception as err:
            logger.info(f"Execute failed, err_msg: {str(err)}.")
            return Message(RESULT_CODE['ERROR'], err, err)

    def _dele_karbor_3node_and_info(self):
        retry_times = 0
        delete_vm_flag = False
        while retry_times < 3:
            try:
                # 删除karbor3虚拟机
                self.om_util.delete_vm_by_name(self.karbor3_vm_name, self.pod_id)
            except Exception as err:
                logger.warning(f"An error occurred while deleting the VM, err_msg:{str(err)}.")
                retry_times += 1
                time.sleep(60)
            else:
                delete_vm_flag = True
                break

        if not delete_vm_flag:
            raise Exception("Failed to delete the VM.")

        # 删除部署节点信息
        self.cmdb_util.remove_deploy_node_info(self.region_id, self.karbor3_vm_name)

        result = self._delete_karbor3_unify_pwd()
        if not result:
            raise Exception("Delete karbor3 unify passwd failed.")

        return Message(RESULT_CODE['SUCCESS'])

    def _delete_karbor3_unify_pwd(self):
        iam_token = self.iam_util.get_iam_token(self.iam_util.get_iam_account_info())
        if not iam_token:
            logger.error("Get iam token failed.")
            raise Exception("Get iam token failed.")
        header = {
            "Content-Type": "application/json",
            "X-Auth-Token": iam_token
        }
        oc_server_url = self.param_tool.get_oc_url()

        register_url = f"{oc_server_url}/rest/mounpwdservice/v1/account"
        register_info = self._build_request_data()
        if not register_info:
            logger.error("Build unify pwd info failed, so don't request.")
            return False

        register_retry_times = 0
        logger.info(f"Register unify pass_word url:{register_url}.")
        while True:
            try:
                rsp = requests.request("POST", register_url,
                                       data=json.dumps(register_info),
                                       verify=False, headers=header)
            except Exception as err:
                logger.error("Register karbor service to mo unify pass_word "
                             f"occur exc, info:{str(err)}.")
            else:
                logger.info(
                    f"Register unify pass_word result code:{rsp.status_code}.")
                if rsp.status_code <= 300 and rsp.content:
                    rsp_content = json.loads(rsp.content)
                else:
                    return False
                if rsp_content.get("code") == "00000000":
                    logger.info(
                        "Register karbor service to mo unify passwd "
                        f"success, response code:{rsp_content.get('code')}.")
                    return True
                logger.error("Register karbor service to mo unify pass_word "
                             f"failed, msg:{rsp.content}.")

            register_retry_times += 1
            if register_retry_times > RETRY_TIME_LIMIT:
                logger.error("Unify pass_word reach the maximum number of "
                             "retries, so do not try again")
                return False
            time.sleep(30)

    def _build_request_data(self):
        karbor3_node = self.karbor_util.get_karbor_node_list()[-1]
        account_list = [
            {
                "accountName": karbor3_node.user,
                "region": self.region_id,
                "ip": karbor3_node.node_ip,
                "accountType": 1,
                "usedScene": "karbor",
                "operationType": 2
            },
            {
                "accountName": "root",
                "region": self.region_id,
                "ip": karbor3_node.node_ip,
                "accountType": 1,
                "usedScene": "karbor",
                "operationType": 2
            }
        ]
        return {
            "componentName": CSBS_SERVICE_NAME,
            "subComponents": [{"subComponentName": CSBS_SERVICE_NAME,
                               "createdAccountList": account_list}]
        }
