import time
from datetime import datetime, timedelta, timezone

import utils.common.log as logger
from plugins.CSBS.common.upgrade import constant
from plugins.CSBS.scripts.upgrade.ab_test.tasks.quota import QuotaManager

POLICY_EXPIRED_TIME = 10


class FilesetTest(object):
    def __init__(self, context, http_client):
        self.context = context
        self.http_client = http_client
        self.console_host = self.context.basic_mo_test.get('console_host')
        self.project_id = self.context.user_info['project_id']
        self.region_id = self.context.engine_info['region_id']
        self.quota_manager = QuotaManager(self.context, self.http_client)

    def get_agents_info(self):
        headers = {"X-MicroVersion": "1.2"}
        self.http_client.headers_update(headers)
        url = f"https://{self.console_host}/cbs/rest/karbor/v1/{self.project_id}/hosts"
        return self.http_client.get(url)

    def fileset_apply_quota(self):
        self.quota_manager.apply_quota(quota_type="fileset_backup_capacity")

    def fileset_release_quota(self):
        self.quota_manager.release_quota(tar_quota_type="fileset_backup_capacity")

    def fileset_create_policy(self):
        policy_name = f"{constant.FSBS_POLICY_NAME_PRE}_{time.strftime('%d%H%M%S')}"
        self.apply_policy(constant.FILESET_PROVIDER_ID, policy_name, resource_type="OS::Application::FileSet",
                          is_enable_full_backup=True)

    def fileset_query_tasklist(self):
        self.get_plan_logs(provider_id=constant.FILESET_PROVIDER_ID, resource_type="OS::Application::FileSet")

    def fileset_query_backup_copy(self):
        self.get_backup_copies("OS::Application::FileSet")

    def fileset_delete_policy(self):
        self.delete_policy(constant.FILESET_PROVIDER_ID, resource_type="OS::Application::FileSet")

    def get_backup_copies(self, resource_type):
        url = f"https://{self.console_host}/cbs/rest/karbor/v1/{self.project_id}/checkpoint_items" \
              f"?resource_type={resource_type}"
        return self.http_client.get(url)

    def get_plan_logs(self, provider_id, resource_type, operation_type=None):
        url = f"https://{self.console_host}/cbs/rest/karbor/v1/{self.project_id}/operation_logs" \
              f"?resource_type={resource_type}&provider_id={provider_id}"
        if operation_type:
            if not isinstance(operation_type, (list, tuple)):
                raise Exception("The value of 'operation_type' must be a list or tuple.")
            if not set(operation_type) <= set(constant.OP_LOG_TYPE):
                raise Exception(f"The value of 'operation_type' must be in {constant.OP_LOG_TYPE}, "
                                f"the 'operation_type' is {operation_type}.")
            for _type in operation_type:
                url += f"&operation_type={_type}"
        return self.http_client.get(url)

    def get_policies(self, provider_id, resource_type=None):
        # 云服务器和云硬盘备份策略不能传入resource_type
        url = f"https://{self.console_host}/cbs/rest/karbor/v1/{self.project_id}/policies?" \
              f"provider_id={provider_id}"
        if resource_type:
            url += f"&resource_type={resource_type}"
        _, body = self.http_client.get(url)
        return body

    def _get_policy_detail_by_policy_id(self, policy_id):
        url = f"{self.console_host}/cbs/rest/karbor/v1/{self.project_id}/policies/{policy_id}"
        return self.http_client.get(url=url, verify=False)

    def apply_policy(self, provider_id, policy_name, resource_type,
                     resource_list=None,
                     is_enable_backup=False, is_enable_copy=False,
                     is_enable_full_backup=False):
        datetime.utcnow()
        expired_time = datetime.now(tz=timezone.utc) + timedelta(POLICY_EXPIRED_TIME)
        expired_time_str = expired_time.strftime("%Y-%m-%dT%H:%M:%S")
        if not resource_list:
            resource_list = []
        if not isinstance(resource_list, list):
            raise Exception("The value of 'resource_list' must be a list.")

        logger.info("Start to create a backup plan "
                    "that sets up a scheduling policy.")
        return self._post_create_policy_order(
            provider_id, policy_name,
            resource_type, resource_list,
            expired_time=expired_time_str,
            is_enabled_backup=is_enable_backup,
            is_enabled_copy=is_enable_copy,
            is_enabled_full_backup=is_enable_full_backup)

    def delete_policy(self, provider_id, resource_type=None):
        logger.info("Start to obtain policy info, "
                    f"provider_id: {provider_id}, "
                    f"resource_type: {resource_type}.")
        policy_info_list = self.get_policies(provider_id, resource_type).get("policies", None)
        logger.info(f"Succeed to obtain policy info, policy infos: {policy_info_list}.")
        policy_id_list = [policy_info["id"] for policy_info in policy_info_list]
        for policy_id in policy_id_list:
            self.delete_policy_by_policy_id(policy_id)

    def delete_policy_by_policy_id(self, policy_id):
        logger.info(f"Start to delete policy, policy id: {policy_id}.")
        url = f"https://{self.console_host}/cbs/rest/karbor/v1/{self.project_id}/policies/{policy_id}"
        self.http_client.delete(url)
        logger.info(f"Succeed to delete policy, policy id: {policy_id}.")

    def _post_create_policy_order(self, provider_id, policy_name,
                                  resource_type, resources,
                                  expired_time="", is_enabled_backup=False,
                                  is_enabled_copy=False,
                                  is_enabled_full_backup=False,
                                  encryption=False):
        if not is_enabled_backup and not is_enabled_full_backup and not is_enabled_copy:
            raise Exception("When creating a policy, specify the operation type of at least one policy. "
                            "The operation types of the policy are as follows:"
                            "full_backup, backup, copy.")
        url = f"https://{self.console_host}/cbs/rest/karbor/v1/{self.project_id}/policies"
        scheduling_name_backup = "scheduling_name_backup"
        scheduling_name_copy = "scheduling_name_copy"
        scheduling_name_full_backup = "scheduling_name_full_backup"

        scheduled_operations = []
        if is_enabled_full_backup:
            scheduling_full_backup = self._assemble_scheduled_operation(
                scheduling_name=scheduling_name_full_backup,
                is_enabled=is_enabled_full_backup,
                operation_type='full_backup')
            scheduled_operations.append(scheduling_full_backup)
        if is_enabled_backup:
            scheduling_backup = self._assemble_scheduled_operation(
                scheduling_name=scheduling_name_backup,
                is_enabled=is_enabled_backup,
                operation_type='backup')
            scheduled_operations.append(scheduling_backup)
        if is_enabled_copy:
            scheduling_copy = self._assemble_scheduled_operation(
                scheduling_name=scheduling_name_copy,
                is_enabled=is_enabled_copy,
                operation_type='copy')
            scheduled_operations.append(scheduling_copy)
        body_dict = {
            "policy": {
                "name": policy_name,
                "provider_id": provider_id,
                "scheduled_operations": scheduled_operations,
                "resources": resources,
                "parameters": {
                    "common": {
                        "az": self.context.az_id,
                        "expired_at": expired_time
                    },
                    "advanced_params": {
                        "encryption": encryption
                    }
                },
                "resource_type": resource_type
            }
        }
        return self.http_client.post(url, json=body_dict)

    @staticmethod
    def _assemble_scheduled_operation(scheduling_name, is_enabled,
                                      operation_type):
        max_backups = 20
        pattern_freq = "WEEKLY"
        pattern_by_day = "MO"
        pattern_by_hour = "10"
        pattern_by_minute = "20"

        trigger_pattern_str = "BEGIN:VCALENDAR\r\nBEGIN:VEVENT\r\n" \
                              "RRULE:FREQ={freq};BYDAY={by_day};" \
                              "BYHOUR={by_hour};BYMINUTE={by_minute}\r\n" \
                              "END:VEVENT\r\n" \
                              "END:VCALENDAR\r\n"
        trigger_pattern = trigger_pattern_str.format(freq=pattern_freq,
                                                     by_day=pattern_by_day,
                                                     by_hour=pattern_by_hour,
                                                     by_minute=pattern_by_minute)

        result_dict = {
            "name": scheduling_name,
            "description": "",
            "enabled": is_enabled,
            "operation_definition": {
                "max_backups": max_backups
            },
            "trigger": {
                "properties": {
                    "pattern": trigger_pattern
                }
            },
            "operation_type": operation_type
        }
        return result_dict
