import json

import utils.common.log as logger

from plugins.CSBS.common.upgrade import constant
from plugins.CSBS.scripts.upgrade.ab_test.tasks.product_manager import ProductManager

logger.init("CSBS-VBS")


class QuotaManager(object):
    def __init__(self, context, http_client):
        self.context = context
        self.http_client = http_client
        self.product_manager = ProductManager(self.context, self.http_client)
        self.console_host = self.context.basic_mo_test["console_host"]
        self.project_id = self.context.user_info['project_id']

    def csbs_check_and_apply_quota(self):
        self.apply_quota("backup_capacity")

    def vbs_check_and_apply_quota(self):
        self.apply_quota("volume_backup_capacity")

    def release_quota(self, tar_quota_type=None):
        logger.info("Start to get quota information.")
        quotas_info = self.get_quota_info()
        logger.info(f"Get quotas info successfully, quotas info:{quotas_info}.")
        for quota in quotas_info["quotas"]["resources"]:
            quota_type = quota["type"]
            if quota_type not in constant.QUOTA_TYPE_DICT:
                continue
            if tar_quota_type and quota_type != tar_quota_type:
                continue
            quota_num = quota.get("quota")
            if quota_num == 0:
                logger.info(f"The quota space is 0 and does not need to be released, quota type:{quota_type}.")
                continue
            logger.info(f"Start to release quota, quota type is {quota_type}.")
            quota_params = self.assemble_quota_params(quota_type, -quota_num)
            self.modify_quota(quota_type, quota_params)
            logger.info(f"Succeeded to release quota, quota type is {quota_type}.")

    def get_quota_info(self):
        url = f"https://{self.console_host}/cbs/rest/karbor/v1/{self.project_id}/quotas"
        _, quota_info = self.http_client.get(url)
        return quota_info

    def modify_quota(self, quota_type, quota_params):
        service_type = constant.QUOTA_TYPE_DICT.get(quota_type).get('service_type')
        product_id = self.product_manager.get_product_id(
            service_type=service_type,
            product_name_en=constant.QUOTA_TYPE_DICT.get(quota_type).get('product_name_en').replace(" ", ""))
        url = f"https://{self.console_host}/cbs/rest/karbor/v1/{self.project_id}/products/{product_id}/action"
        body_dict = {
            "apply": {
                "service_type": service_type,
                "region_id": self.context.engine_info.get('region_id'),
                "params": quota_params
            }
        }
        self.http_client.post(url, body=body_dict)

    def apply_quota(self, quota_type):
        if quota_type not in constant.QUOTA_TYPE_DICT:
            raise Exception(f"The quota_type should be in {constant.QUOTA_TYPE_DICT}.")
        logger.info(f"Start to get quota information, quota type:{quota_type}.")
        quota_info = self.get_quota_info()

        is_need_add_quota = False
        cur_backup_quota = 0
        for quota in quota_info["quotas"]["resources"]:
            if quota.get("type") == quota_type:
                cur_backup_quota = quota["quota"]
                if cur_backup_quota < constant.MINIMUM_QUOTA:
                    logger.info(f"The backup capacity is {cur_backup_quota}M, less than the established capacity.")
                    is_need_add_quota = True
                    break
                else:
                    logger.info(f"The backup capacity is {cur_backup_quota}M, which can be used normally.")
        if is_need_add_quota:
            logger.info(f"Start to apply backup space "
                        f"for {constant.QUOTA_TYPE_DICT.get(quota_type).get('product_name_en')}.")
            quota_num = constant.MINIMUM_QUOTA - cur_backup_quota
            self._add_quota(quota_type, quota_num)
            logger.info("Succeed in applying backup space "
                        f"for {constant.QUOTA_TYPE_DICT.get(quota_type).get('product_name_en')}.")

    def _add_quota(self, quota_type, quota_num):
        logger.info("Start to increase backup space.")
        params = self.assemble_quota_params(quota_type, quota_num)
        self.modify_quota(quota_type, params)
        logger.info("Succeed in increasing backup space.")

    @staticmethod
    def assemble_quota_params(quota_type, var_quota_num):
        if quota_type not in constant.QUOTA_TYPE_DICT:
            raise Exception(f"The quota_type should be in {constant.QUOTA_TYPE_DICT}.")
        display_num = var_quota_num // 1024
        params = {
            "count": 1,
            "param_items": [
                {
                    "type": quota_type,
                    "quota": var_quota_num
                }
            ],
            "display": {
                "en_US": [
                    {
                        "label": constant.QUOTA_TYPE_DICT[quota_type]['en_label'],
                        "type": "string",
                        "value": f"{display_num}GB"
                    }
                ],
                "zh_CN": [
                    {
                        "label": constant.QUOTA_TYPE_DICT[quota_type]['cn_label'],
                        "type": "string",
                        "value": f"{display_num}GB"
                    }
                ]
            }
        }
        return json.dumps(params)
