import os
import re
import tarfile
import zipfile

import utils.common.log as logger
from utils.business.manageone_cmdb_util import ManageOneCmdbUtil
from utils.business.manageone_util2 import ManageOneUtil2
from utils.business.param_util import ParamUtil
from utils.business.project_util import ProjectApi
from utils.common import param_check as paramCheckUnit
from utils.common.exception import HCCIException
from utils.common.ssh_util import Ssh
from utils.DBAdapter.DBConnector import BaseOps

from plugins.eBackup.common.algorithm_switching import get_cipher_type
from plugins.eBackup.common.api_adapter import API
from plugins.eBackup.common.constant import UNZIP_FILE_NUM_MAX, UNZIP_FILE_SIZE_MAX
from plugins.eBackup.common.ebackup_rest import EbackupRest
from plugins.eBackup.common.util import check_url_param


class eBackup:
    def __init__(self, project_id, pod_id):
        self.project_id = project_id
        self.pod_id = pod_id
        self.params = ParamUtil()
        self.region_id = self.params.get_param_value(pod_id, "public", "region_id", "region0_id")
        self.cmdb_util = ManageOneCmdbUtil(project_id, pod_id)
        self.ssh_obj = Ssh()
        self.service_name = "eBackup"
        self.data = self.params.get_service_cloud_param(pod_id, "eBackup")
        self.rest_api = EbackupRest()
        self.config_dict = self.params.get_service_cloud_param(pod_id, "eBackup")
        self.hcp_passwd = self.config_dict.get('hcp_ssh_password') or self.config_dict.get('eBackup_hcp_pwd')
        self.root_passwd = self.config_dict.get('eBackup_os_root_password') or self.config_dict.get('eBackup_root_pwd')

    def get_service_ip_list_from_cmdb(self):
        nodes_info = self.cmdb_util.get_deploy_node_info(region_code=self.region_id, index_name=self.service_name)
        return nodes_info

    def get_kmc_alg_mode(self):
        map_ = {
            'generalCipher': 0,
            'SMCompatible': 1,
            "SMOnly": 2
        }
        return map_[self.get_kmc_alg_mode_str()]

    def get_kmc_alg_mode_str(self):
        kmc_alg_mode = get_cipher_type(self.project_id)
        if not kmc_alg_mode:
            kmc_alg_mode = eBackup(self.project_id, self.pod_id) \
                .get_kmc_alg_mode_for_cmdb()
        return kmc_alg_mode

    def reencrypt_cert_cipher(self, ip_list, hcp_passwd, root_passwd):
        if self.get_kmc_alg_mode() == 0:
            logger.info("cipher mode is generalCipher,"
                        " no need to reencrypt cert.")
            return 0
        cmd = "sh /opt/huawei-data-protection/ebackup/bin/sudo_common_func.sh " \
              "cert_cipher_reencrypt && echo 'reencrypt cert successful.'"
        for ip in ip_list:
            result = self.ssh_obj.ssh_cmds(
                ip,
                cmd,
                "hcp",
                hcp_passwd,
                root_passwd,
                "",
                ""
            )
            if str(result).find("reencrypt cert successful.") == -1:
                logger.error("reencrypt cert failed on node %s. stdout: %s" % (ip, str(result)))
                continue
            logger.info("Reencrypt cert successful on node %s." % ip)
        return 0

    def get_kmc_alg_mode_for_cmdb(self):
        kmc_alg_mode = None
        nodes_info = self.get_service_ip_list_from_cmdb()
        for item in nodes_info:
            if item.get('name') and item.get('name').startswith('eBackup_service'):
                for extend_items in item.get('extendInfos'):
                    if extend_items.get('key') == 'CipherType':
                        kmc_alg_mode = extend_items.get('value')
        if not kmc_alg_mode:
            raise HCCIException(653096)
        return kmc_alg_mode

    def get_project_scales(self):
        service_scale = ParamUtil().get_project_scales_by_project_id(self.project_id).get("scale")
        return service_scale

    def login_ebk_iam(self, lb_listen_addr):
        try:
            logger.info(f"begin to login ebk_iam micro service. lb_listen_addr = {lb_listen_addr}")
            login_url = "https://%s:8090" % lb_listen_addr
            headers = {'Accept': 'application/json'}
            self.rest_api.get(login_url, headers)
            return True
        except Exception as e:
            logger.error("Test the connection to the IAM failed:%s" % str(e))
            return False

    def input_8090_iptables_rule(self):
        float_ip = self.get_datamover_internal_float_ip()
        ip_list = self.get_datamover_iplist()
        if not self.login_ebk_iam(float_ip):
            cmd = "iptables -I INPUT  -p tcp -s 0.0.0.0/0" \
                  " -d 0.0.0.0/0 --dport 8090 -j ACCEPT"
            for ebk_ip in ip_list[0:2]:
                self.ssh_obj.ssh_cmds(
                    ebk_ip, cmd, "hcp", self.hcp_passwd,
                    self.root_passwd, "", "")

    def delete_8090_iptables_rule(self):
        float_ip = self.get_datamover_internal_float_ip()
        ip_list = self.get_datamover_iplist()
        if self.login_ebk_iam(float_ip):
            cmd = "iptables -D INPUT  -p tcp -s 0.0.0.0/0 -d 0.0.0.0/0 --dport 8090 -j ACCEPT"
            for ebk_ip in ip_list:
                self.ssh_obj.ssh_cmds(
                    ebk_ip, cmd, "hcp", self.hcp_passwd,
                    self.root_passwd, "", "")

    def get_iam_tokens(self, iam_host, iam_port):
        logger.info("begin to login ebk_iam micro service.")
        url = "https://%s:%s/v3/auth/tokens" % (iam_host, iam_port)
        headers = {'Accept': 'application/json'}
        # login use pwd
        data = '{"auth":{"identity":{"methods":["password"],"password":' \
               '{"user":{"name":"admin","password":"%s","iam_type":0}}}}}' % \
               self.config_dict['eBackup_weblogin_password']
        rsp = self.rest_api.post(url, headers, data)
        if rsp.status_code != 200:
            logger.error("rest req return not 200.")
            raise HCCIException(653020)
        errcode = rsp.json()["Error"]["Code"]
        if errcode is None:
            logger.error("errcode is empty.")
            raise HCCIException(653020)
        if errcode == 0:
            token = rsp.json()["Data"]["token"]
            self.ebk_iam_headers = {"Accept": "application/json", "X-Auth-Token": token}
            logger.info("login ebk_iam micro service succ.")
            return self.ebk_iam_headers
        else:
            description = rsp.json()["Error"]["Description"]
            logger.error("login failed, description:%s" % description)
            raise HCCIException(653086, url, description)

    def logout_ebk_iam(self, iam_host, iam_port):
        if not self.ebk_iam_headers:
            return
        url = "https://%s:%s/v3/auth/tokens" % (iam_host, iam_port)
        self.rest_api.delete(url, self.ebk_iam_headers)
        logger.info("login out ebk_iam succ.")

    def get_datamover_iplist(self):
        ip_list = self.config_dict['datamover_externalom_iplist'].lower()
        datamover_iplist = ip_list.split(',')
        return datamover_iplist

    def get_datamover_internal_float_ip(self):
        return self.config_dict['datamover_internal_float_ip']

    def get_version_number(self):
        path, name = API.find_file("OceanStor", "eBackup_DMK.zip")
        if name == "":
            raise HCCIException(653004, "OceanStor......eBackup_DMK.zip")
        current_version = API.get_package_version(name)
        return current_version


class CustomParametersUtil:
    """
    CustomParametersUtil, Parameters handle for eBackup.
    """

    def __init__(self, project_id, pod_id):
        self.mo_util = ManageOneUtil2()
        self.params = ParamUtil()
        self.__config_dict = self.params.get_service_cloud_param(pod_id,
                                                                 "eBackup")
        self.db = BaseOps()
        self.project_id = project_id
        self.pod_id = pod_id
        self.mo_domain_list = self.mo_util.get_mo_domain_name_list(self.project_id, self.pod_id)

    def get_iam_domain_name(self):
        iam_domain = self.mo_domain_list[2] + ":26335"
        if not iam_domain:
            raise Exception("Failed to obtain iam domain.")
        logger.info("Succeed to obtain iam domain, "
                    f"the iam domain is: [{iam_domain}].")

        check_url_param(iam_domain)
        return iam_domain

    def get_oc_domain_name(self):
        oc_domain = self.mo_domain_list[0] + ":26335"
        if not oc_domain:
            raise Exception("Failed to obtain oc domain.")
        logger.info("Succeed to obtain oc domain, "
                    f"the oc domain is: [{oc_domain}].")

        check_url_param(oc_domain)
        return oc_domain


def check_compress_file(check_file_list):
    for file in check_file_list:
        if not os.path.exists(file):
            raise HCCIException(650057, file)
        if file.endswith(".zip"):
            zip_file = zipfile.ZipFile(file, "r")
            zip_file_name_list = zip_file.namelist()
            if len(zip_file_name_list) > UNZIP_FILE_NUM_MAX:
                raise HCCIException(653109, file)
            file_size = 0
            for zip_file_name in zip_file_name_list:
                file_info = zip_file.getinfo(zip_file_name)
                file_size += file_info.file_size
            if file_size > UNZIP_FILE_SIZE_MAX:
                raise HCCIException(653101, file)
        elif file.endswith(".tar.gz"):
            tar_files = tarfile.open(file, "r")
            tar_file_list = tar_files.getmembers()
            if len(tar_file_list) > UNZIP_FILE_NUM_MAX:
                raise HCCIException(653109, file)
            file_size = 0
            for tar_file in tar_file_list:
                file_size += tar_file.size
            if file_size > UNZIP_FILE_SIZE_MAX:
                raise HCCIException(653101, file)
        else:
            logger.error("compress file checking do not support this format.")
            raise HCCIException(653102, file)
    logger.info("Succeed to verify compress file")


def check_ip_valid(check_ip, project_id):
    if ProjectApi().is_ipv6_project(project_id):
        ip_version = 6
    else:
        ip_version = 4
    return paramCheckUnit.check_param_ip(check_ip, ip_version)


def check_base64_string(string):
    pattern = "^([A-Za-z0-9+/]{4})*([A-Za-z0-9+/]{3}=|[A-Za-z0-9+/]{2}==)?$"
    match_obj = re.match(pattern, string, re.I)
    return bool(match_obj)
