# -*- coding:utf-8 -*-
from utils.DBAdapter.DBConnector import BaseOps
from utils.business.param_util import ParamUtil
from utils.common import log as logger
from utils.common.exception import HCCIException
from utils.common.fic_base import StepBaseInterface
from utils.common.message import Message

from plugins.eBackup.common.model import DmkUserInfo
from plugins.eBackup.common.api_adapter import API


class CreateDmkAccount(StepBaseInterface):
    def __init__(self, project_id, pod_id):
        logger.init("eBackup")
        self.project_id = project_id
        self.pod_id = pod_id
        self.db = BaseOps()
        self.__db_param_dict = {}

    def pre_check(self, project_id, pod_id):
        """
        插件内部接口：执行安装前的资源预检查，该接口由execute接口调用，工具框架不会直接调用此接口。
        :param project_id:
        :param pod_id:
        :return:
        """
        pass

    def execute(self, project_id, pod_id):
        """
        标准调用接口：执行安装前预检查&安装&配置
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        try:
            self.create_dmk_user()
            return Message(200)
        except HCCIException as e:
            return Message(500, e)
        except Exception as e:
            logger.error(f"Failed to create DMK account for eBackup plugin, the reason is {e}")
            return Message(500, e, e)

    def rollback(self, project_id, pod_id):
        """
        标准调用接口：执行回滚
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        return Message()

    def retry(self, project_id, pod_id):
        """
        标准调用接口：重试
        :return: Message类对象
        """
        return self.execute(project_id, pod_id)

    def check(self, project_id, pod_id):
        """
        标准调用接口：重试
        :param project_id:
        :param pod_id:
        :return:
        """
        pass

    def get_all_params(self):
        self.__db_param_dict = API.get_service_param_list(
            self.db, self.pod_id, 'eBackup')
        params = ParamUtil()

        self.__db_param_dict['fsp_root_pwd'] = params.get_param_value(
            self.pod_id, "OpenStack", 'root_password',
            'openstack_root_password')
        self.__db_param_dict['fsp_user_pwd'] = params.get_param_value(
            self.pod_id, "OpenStack", 'fsp_password', 'openstack_fsp_password')
        self.__db_param_dict['fsp_env_pwd'] = params.get_param_value(
            self.pod_id, "OpenStack", 'env_os_password',
            'openstack_env_os_password')

        if not self.__db_param_dict['fsp_root_pwd']:
            raise HCCIException(653044, "fsp_root_pwd")
        if not self.__db_param_dict['fsp_user_pwd']:
            raise HCCIException(653044, "fsp_user_pwd")
        if not self.__db_param_dict['fsp_env_pwd']:
            raise HCCIException(653044, "fsp_env_pwd")
        dmk_param_key_list = [
            'dmk_ui_username',
            'dmk_os_business_username',
            'dmk_os_business_user_password']
        dmk_dict = API.get_dmk_info(self.pod_id)
        for key in dmk_param_key_list:
            self.__db_param_dict[key] = dmk_dict[key]
        self.__db_param_dict['dmk_floatIp'] = params.get_param_value(
            self.pod_id, "DMK", 'dmk_floatIp', 'dmk_floatIp')
        self.__db_param_dict['dmk_ui_password'] = params.get_param_value(
            self.pod_id, "DMK", 'dmk_ui_password', 'dmk_ui_password')

    def create_dmk_user(self):
        """
        创建DMK用户
        :return:
        """
        self.get_all_params()
        team_list = [self.__db_param_dict['eBackup_dmk_group_name'], "OTD"]
        team_id = self.create_dmk_login_user(team_list)
        self.create_root_ssh_account(team_id, team_list)
        self.create_fsp_ssh_account(team_id, team_list)
        self.create_hcp_ssh_account(team_id, team_list)

    def create_dmk_login_user(self, team_list):
        API.login_dmk(
            self.__db_param_dict['dmk_floatIp'],
            self.__db_param_dict['dmk_ui_username'],
            self.__db_param_dict['dmk_ui_password'])
        team_id, user_id = API.createdmkuser(
            self.__db_param_dict['eBackup_dmk_user'],
            self.__db_param_dict['eBackup_dmk_tmp_password'], '2',
            self.__db_param_dict['eBackup_dmk_group_name'])
        if team_id and user_id:
            bret = API.update_user_to_team(user_id, '2', team_list)
            if not bret:
                logger.error("eBackup,update user to multi team failed.")
                raise HCCIException(653112)

        API.login_dmk(
            self.__db_param_dict['dmk_floatIp'],
            self.__db_param_dict['eBackup_dmk_user'],
            self.__db_param_dict['eBackup_dmk_tmp_password'])
        API.changedmkuserpasswd(
            user_id,
            self.__db_param_dict['eBackup_dmk_tmp_password'],
            self.__db_param_dict['eBackup_dmk_password'])

        API.login_dmk(
            self.__db_param_dict['dmk_floatIp'],
            self.__db_param_dict['eBackup_dmk_user'],
            self.__db_param_dict['eBackup_dmk_password'])

        return team_id

    def create_root_ssh_account(self, team_id, team_list):
        try:
            API.get_dmk_account_id(
                self.__db_param_dict['eBackup_dmk_user'],
                self.__db_param_dict['eBackup_dmk_ssh_user'])
        except Exception as e:
            logger.info(f"step 1: query account faild, the reason {e}")
            API.addaccounttodmk(
                self.__db_param_dict['eBackup_dmk_ssh_user'], team_id,
                self.__db_param_dict['eBackup_os_root_password'],
                self.__db_param_dict['eBackup_os_root_password'])
            dmkinfo = DmkUserInfo(
                self.__db_param_dict['eBackup_dmk_user'], self.__db_param_dict['eBackup_dmk_ssh_user'], team_list,
                [self.__db_param_dict['eBackup_os_root_password'], self.__db_param_dict['eBackup_os_root_password']]
            )
            bret = API.update_account_to_team(dmkinfo)
            if not bret:
                logger.error(
                    "eBackup,update root account to multi team failed.")
                raise HCCIException(653113, self.__db_param_dict['eBackup_dmk_ssh_user']) from e

    def create_fsp_ssh_account(self, team_id, team_list):
        try:
            API.get_dmk_account_id(
                self.__db_param_dict['eBackup_dmk_user'], "fsp")
        except Exception as e:
            logger.info(f"step 2: query account faild, the reason {e}")
            API.addaccounttodmk(
                "fsp", team_id,
                self.__db_param_dict['fsp_user_pwd'],
                self.__db_param_dict['fsp_root_pwd'])
            dmkinfo = DmkUserInfo(
                self.__db_param_dict['eBackup_dmk_user'], "fsp", team_list,
                [self.__db_param_dict['fsp_user_pwd'], self.__db_param_dict['fsp_root_pwd']]
            )
            bret = API.update_account_to_team(dmkinfo)
            if not bret:
                logger.error("eBackup,update fsp account to multi team failed.")
                raise HCCIException(653113, "fsp") from e

    def create_hcp_ssh_account(self, team_id, team_list):
        try:
            API.get_dmk_account_id(
                self.__db_param_dict['eBackup_dmk_user'],
                self.__db_param_dict['hcp_ssh_user'])
        except Exception as e:
            logger.info(f"step 3: query account faild, the reason {e}")
            API.addaccounttodmk(
                self.__db_param_dict['hcp_ssh_user'], team_id,
                self.__db_param_dict['hcp_ssh_password'],
                self.__db_param_dict['eBackup_os_root_password'])
            dmkinfo = DmkUserInfo(
                self.__db_param_dict['eBackup_dmk_user'], self.__db_param_dict['eBackup_dmk_ssh_user'], team_list,
                [self.__db_param_dict['hcp_ssh_password'], self.__db_param_dict['eBackup_os_root_password']]
            )
            bret = API.update_account_to_team(dmkinfo)
            if not bret:
                logger.error("eBackup,update hcp account to multi team failed.")
                raise HCCIException(653113, self.__db_param_dict['hcp_ssh_user']) from e

# the code has been updated to python3.7
