# -*- coding:utf-8 -*-
import base64

from jinja2 import Environment, FileSystemLoader

from utils.business.dns_util import DNSApi, DNSNodeType
from utils.business.hardware_driver_util import DriverApi
from utils.business.manageone_cmdb_util import ManageOneCmdbUtil
from utils.business.param_util import ParamUtil
from utils.business.project_condition_utils import get_project_condition_boolean
from utils.business.project_condition_utils import get_project_conditions
from utils.common import log as logger
from utils.common.exception import HCCIException
from utils.common.fic_base import StepBaseInterface
from utils.common.message import Message
from utils.common.ssh_util import Ssh

from plugins.eBackup.common.ebackup_util import EBackupUtil
from plugins.eBackup.common.iam_util import IamUtil
from plugins.eBackup.common.param_tool import ParamTool
from plugins.eBackup.common.api_adapter import API
from plugins.eBackup.common.model import DmkTaskInfo
import plugins.eBackup.scripts.common.ebackup_util as ebackup_handle
from plugins.eBackup.scripts.common.ebackup_util import eBackup

INSTALL_ACTION_STR = "[install]2.Install and configure eBackup"
UNINSTALL_ACTION_STR = "[uninstall]1.Uninstall eBackup"


class BasicConfig(StepBaseInterface):
    def __init__(self, project_id, pod_id):
        logger.init("eBackup")
        # params
        self.params = ParamUtil()
        self.__db_param_dict = self.params.get_service_cloud_param(
            pod_id, "eBackup")
        self.project_id = project_id
        self.pod_id = pod_id
        self.ssh_obj = Ssh()
        self.primary_manage_ip = ""
        self.primary_inter_ip = ""
        self.primary_produce_manage_ip = ""
        self.primary_produce_storage_ip = ""
        self.primary_backup_storage_ip = ""

        self.standby_role = ""
        self.standby_manage_ip = ""
        self.standby_inter_ip = ""
        self.standby_produce_manage_ip = ""
        self.standby_produce_storage_ip = ""
        self.standby_backup_storage_ip = ""
        self.inter_float_ip = ""
        self.leader_public_key = ""
        self.primary_role = "0"
        self.network_type = "ipv4"
        self.firstip_list = None
        self.secondip_list = None
        self.thirdip_list = None
        # root_passwd hcp_user_pwd 默认密码
        self.root_passwd = self.__db_param_dict['eBackup_os_root_password']
        self.hcp_user_pwd = self.__db_param_dict['hcp_ssh_password']
        # server_*会被修改为MO上密码 MO上不存在时使用默认密码
        self.server_hcp_pwd = self.hcp_user_pwd
        self.server_root_pwd = self.root_passwd
        self.ebackup = eBackup(self.project_id, self.pod_id)
        self.ebackup_util = EBackupUtil(self.project_id, self.pod_id)
        self.cmdb_util = ManageOneCmdbUtil(project_id, pod_id)
        self.param_tool = ParamTool(self.project_id, self.pod_id)
        self.iam_util = IamUtil(self.project_id, self.pod_id)
        self.datamover_dns_ip = ""

    def pre_check(self, project_id, pod_id):
        """
        插件内部接口：执行安装前的资源预检查，该接口由execute接口调用，工具框架不会直接调用此接口。
        :param project_id:
        :param pod_id:
        :return:
        """
        pass

    def main_handle(self, project_id):
        node_ips = self.__db_param_dict.get('datamover_externalom_iplist').lower()
        node_ip_list = node_ips.split(',')
        if self.__db_param_dict.get('datamover_producestorage_iplist') == '':
            self.__db_param_dict['datamover_producestorage_iplist'] = node_ips
        self.set_ntp(node_ips)
        self.update_iscsi_random_number()
        self.exec_dmk_operate(INSTALL_ACTION_STR)

        if get_project_condition_boolean(project_id, 'eBackup_Proxy'):

            try:
                self.set_network_for_proxy(node_ip_list)
            except Exception as iptables_err:
                logger.warn(f"Set iptables for proxy failed. Reason:{iptables_err}")

            default_proxy_pub = self.query_backupnode_pub(
                node_ip_list[0], True, False)

            server_default_proxy_pub = self.query_backupnode_pub(
                self.primary_inter_ip, True, True)

            if default_proxy_pub != server_default_proxy_pub:
                proxy_pub = self.query_backupnode_pub(
                    node_ip_list[0], False, False)
                self.add_pub_to_server(proxy_pub)
        # 更新驱动
        self.update_driver(node_ip_list)

    def execute(self, project_id, pod_id):
        """
        标准调用接口：执行安装前预检查&安装&配置
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        try:
            self.main_handle(project_id)
        except HCCIException as e:
            return Message(500, e)
        except Exception as e:
            logger.error(
                "Failed to install and config eBackup(datamover) ,"
                "the reason is %s" % e)
            return Message(500, e, e)
        return Message(200)

    def set_network_for_proxy(self, proxy_nodes):
        logger.info("Begin to check network for proxy")
        active_standby_nodes = self.ebackup_util.get_active_stanby_node(self.primary_inter_ip)
        active_standby_net_mask = self.ebackup_util.get_node_ip_netmask(
            active_standby_nodes, "hcp", self.server_hcp_pwd)

        mask = self.ebackup_util.netmask_to_bit_length(self.__db_param_dict.get("eBackup_externalom_netmask"))
        proxy_net_mask = self.__db_param_dict.get("eBackup_externalom_gateway") + "/" + mask

        if active_standby_net_mask == proxy_net_mask:
            logger.info(f"netmask({active_standby_net_mask}) is same on proxy with active node. skip network config.")
            return

        logger.info(f"Netmask({active_standby_net_mask}|{proxy_net_mask}) "
                    f"is not same on proxy, need to config network for proxy.")
        # 放通 代理节点到主节点的防火墙
        active_standby_port_list = ["5569", "21000", "8090", "6432"]
        active_standby_cmd_list = ["iptables-save > /etc/sysconfig/iptables.bak.`date +%s`"]
        active_standby_base_cmd = \
            f"iptables -I INPUT -s {proxy_net_mask} -d {active_standby_net_mask} -p tcp --dport %s -j ACCEPT"
        active_standby_cmd_list = \
            active_standby_cmd_list + [active_standby_base_cmd % port for port in active_standby_port_list]
        active_standby_cmd_list.append("iptables-save > /etc/sysconfig/iptables")
        active_standby_cmd_list.append(f"echo 0 | sh /opt/huawei-data-protection/ebackup/bin/iptablesHelper.sh accept "
                                       f"HCPManagementPlane {proxy_net_mask} {active_standby_net_mask}")
        # 放通代理节点访问数据库
        active_standby_cmd_list.append(f"grep -q '{proxy_net_mask}' /opt/huawei-data-protection/ebackup/db/data/"
                                       f"pg_hba.conf || (echo 'hostssl   DATAMOVERDB    "
                                       f"GAUSSDB,EBKDBUSER,EBKDBUSERSEC,EBKDBADMIN {proxy_net_mask} sha256' >> "
                                       f"/opt/huawei-data-protection/ebackup/db/data/pg_hba.conf && "
                                       f"echo 'hostssl   ADMINDB    "
                                       f"GAUSSDB,EBKDBUSER,EBKDBUSERSEC,EBKDBADMIN {proxy_net_mask} sha256' >> "
                                       f"/opt/huawei-data-protection/ebackup/db/data/pg_hba.conf)")
        active_standby_cmd_list.append("/etc/init.d/gaussdb restart")
        active_standby_cmd_list.append("echo 'config network success'")

        # 放通 主节点到代理节点防火墙
        proxy_port_list = [str(port) for port in range(21000, 21130, 10)] + ["21150", "21371"]
        proxy_base_cmd = \
            f"iptables -I INPUT -s {active_standby_net_mask} -d {proxy_net_mask} -p tcp --dport %s -j ACCEPT"
        proxy_cmd_list = ["iptables-save > /etc/sysconfig/iptables.bak.`date +%s`"]
        proxy_cmd_list = proxy_cmd_list + [proxy_base_cmd % port for port in proxy_port_list]
        proxy_cmd_list.append("iptables-save > /etc/sysconfig/iptables")
        proxy_cmd_list.append(f"echo 1 | sh /opt/huawei-data-protection/ebackup/bin/iptablesHelper.sh accept "
                              f"HCPManagementPlane {active_standby_net_mask} {proxy_net_mask}")
        proxy_cmd_list.append("echo 'config network success'")

        for node_ip in active_standby_nodes:
            result = self.ssh_obj.ssh_cmds(
                node_ip, " && ".join(active_standby_cmd_list), "hcp", self.server_hcp_pwd, self.server_root_pwd, "", "")
            if "config network success" not in str(result):
                raise Exception(f"config network failed on node {node_ip}, stdout:{result}")

        for node_ip in proxy_nodes:
            result = self.ssh_obj.ssh_cmds(
                node_ip, " && ".join(proxy_cmd_list), "hcp", self.hcp_user_pwd, self.root_passwd, "", "")
            if "config network success" not in str(result):
                raise Exception(f"config network failed on node {node_ip}, stdout:{result}")
        logger.info("Success to config network for proxy")

    def query_ebackup_unroot_microservice_list(self):
        unroot_microservice_str = "BackupNode,ebk_copy,ebk_delete,ebk_restore,ebk_fsbackup,ebk_vmware,ebk_backup"
        if self.param_tool.is_install_csbs():
            return unroot_microservice_str

        region_id = self.params.get_param_value(self.pod_id, "public", "region_id", "region0_id")
        current_version = self.ebackup.get_version_number()
        node_info_list = self.cmdb_util.get_deploy_node_info_for_mo(region_id, "eBackup", current_version)
        for node_info in node_info_list:
            if node_info['name'].startswith('eBackup_service'):
                datamover_ip = node_info['mgmtIp']
                ret = self.query_ebackup_microservice_user(datamover_ip)
                if ret and ret[0] == 'root':
                    return ""
        return unroot_microservice_str

    def query_ebackup_microservice_user(self, datamover_ip):
        cmd = "ps -ef |grep ebk_backup |grep -vE 'grep|monitor|nginx' | head -n 1 | awk '{print $1}'"
        ssh_client = None
        ret = []
        account_info = API.get_account_info(self.project_id, self.pod_id, 'eBackupServer')
        account_dict = account_info.get(datamover_ip, dict())
        hcp_pwd = account_dict.get("hcp", "") if not account_dict.get("hcp", "") else self.hcp_user_pwd

        def do_query_ebackup_microservice_user():
            nonlocal ssh_client
            ssh_client = self.ssh_obj.ssh_create_client(datamover_ip, "hcp", hcp_pwd)
            return self.ssh_obj.ssh_exec_command_return_list(ssh_client, cmd)

        try:
            ret = do_query_ebackup_microservice_user()
        except Exception as e:
            logger.error(f"datamover: Obtaining {datamover_ip} Microservice Process Running User failed: {e}.")
        finally:
            if ssh_client:
                self.ssh_obj.ssh_close(ssh_client)
        logger.info(f"datamover: Obtaining the Microservice Process Running User. ret = {ret}")
        return ret

    def update_iscsi_random_number(self):
        cmd = '''echo "InitiatorName=`iscsi-iname`" > /etc/iscsi/initiatorname.iscsi && systemctl restart iscsid'''
        externalom_ip = self.__db_param_dict['datamover_externalom_iplist']
        ip_list = externalom_ip.split(',')
        for ip in ip_list:
            ssh_client = self.ssh_obj.ssh_create_client(
                ip,
                "root",
                self.__db_param_dict['eBackup_os_root_password']
            )
            self.ssh_obj.ssh_exec_command_return_list(ssh_client, cmd)
            self.ssh_obj.ssh_close(ssh_client)

    def init_ebackup_config_file(self):
        logger.info("datamover: 1.enter get datamover config step.")
        # 获取安装时候的配置模板信息
        env = Environment(loader=FileSystemLoader(API.SCRIPT_PATH + "/conf"))
        env.trim_blocks = True
        env.lstrip_blocks = True
        ebackup_config_template = \
            env.get_template('eBackup_config_eB')
        ebackup_host_template = env.get_template('eBackup_host_eB')

        # 装备config块
        logger.info("datamover: 2.init params object.")
        logger.info("datamover: 3.finish get datamover params.")
        self.assembly_datamover_plane_ip()
        logger.info("datamover: 4.finish compute datamover params.")
        config_val, host_val = self.assembly_dmk_conf()
        logger.info("datamover: 5.finish reflect the datamover params.")

        # 渲染config文件
        render_var_config = ebackup_config_template.render(config_val)
        render_host_config = ebackup_host_template.render(host_val)
        dmk_config_content = {}
        dmk_config_content.update(
            {'vars_content': render_var_config})
        dmk_config_content.update(
            {'hosts_content': render_host_config})
        logger.info("datamover: 6.end  prepare the datamover params.")
        return dmk_config_content

    def do_query_server_pwd(self, mg_node_ip):
        account_info = API.get_account_info(self.project_id, self.pod_id, 'eBackupServer')
        account_dict = account_info[mg_node_ip]
        if account_dict['hcp']:
            self.server_hcp_pwd = account_dict['hcp']
            self.server_root_pwd = account_dict['root']
        logger.info("query pwd from manageone succ.")

    def query_server_pwd(self, mg_node_ip):
        try:
            self.do_query_server_pwd(mg_node_ip)
        except Exception as e:
            logger.warning(f"query pwd from manageone failed:{str(e)}, use default server password.")

    def query_leader_public_key(self):
        cmd = 'cat /opt/huawei-data-protection/ebackup/conf/cert/' \
              'AdminNode.pub;echo "result: $?"'
        result = self.ssh_obj.ssh_cmds(
            self.primary_inter_ip, cmd, "hcp",
            self.server_hcp_pwd, self.server_root_pwd, "", "")
        if "result: 0" in result:
            self.leader_public_key = '"' + result[0] + '"'
            logger.info("get primary cert success.")
        else:
            logger.error("get primary certfailed, output:%s" % result)
            raise HCCIException(653044, 'eBackup_LeaderPublicKey')

    def assembly_server_and_proxy_plane_ip(self):
        self.primary_manage_ip = self.firstip_list[0]
        self.primary_inter_ip = self.firstip_list[0]
        self.primary_produce_manage_ip = self.firstip_list[0]
        self.primary_produce_storage_ip = self.secondip_list[0]
        self.primary_backup_storage_ip = self.thirdip_list[0]
        if len(self.firstip_list) > 1:
            self.standby_role = "1"
            firstip_str = ''
            secondip_str = ''
            thirdip_str = ''
            for index in range(1, len(self.firstip_list)):
                if firstip_str == '':
                    firstip_str = self.firstip_list[index]
                else:
                    firstip_str = \
                        firstip_str + ',' + self.firstip_list[index]
            for index in range(1, len(self.secondip_list)):
                if secondip_str == '':
                    secondip_str = self.secondip_list[index]
                    thirdip_str = self.thirdip_list[index]
                else:
                    secondip_str = \
                        secondip_str + ',' + self.secondip_list[index]
                    thirdip_str = \
                        thirdip_str + ',' + self.thirdip_list[index]

            self.standby_manage_ip = firstip_str
            self.standby_inter_ip = firstip_str
            self.standby_produce_manage_ip = firstip_str
            self.standby_produce_storage_ip = secondip_str
            self.standby_backup_storage_ip = thirdip_str

    def assembly_proxy_plane_ip(self):
        self.primary_inter_ip = \
            self.__db_param_dict['datamover_server_ip'].lower()
        self.query_server_pwd(self.primary_inter_ip)
        self.query_leader_public_key()
        self.primary_manage_ip = ""
        self.primary_produce_manage_ip = ""
        self.primary_produce_storage_ip = ""
        self.primary_backup_storage_ip = ""
        self.standby_manage_ip = \
            self.__db_param_dict['datamover_externalom_iplist'].lower()
        self.standby_inter_ip = \
            self.__db_param_dict['datamover_externalom_iplist'].lower()
        self.standby_produce_manage_ip = \
            self.__db_param_dict['datamover_externalom_iplist'].lower()
        self.standby_produce_storage_ip = \
            self.__db_param_dict['datamover_producestorage_iplist'].lower()
        if self.__db_param_dict['datamover_backupstorage_iplist'] == '':
            self.standby_backup_storage_ip = self.standby_produce_storage_ip
        else:
            self.standby_backup_storage_ip = \
                self.__db_param_dict['datamover_backupstorage_iplist'].lower()

    def assembly_datamover_plane_ip(self):
        pro_dic = get_project_conditions(self.project_id)
        if 'ManageIPV6' in pro_dic:
            if_ipv6 = pro_dic["ManageIPV6"]
            if if_ipv6 == 1:
                self.network_type = "ipv6"

        if get_project_condition_boolean(self.project_id, 'eBackup_Proxy'):
            self.primary_role = ""
            self.standby_role = "1"

        externalom_ip_lower = \
            self.__db_param_dict['datamover_externalom_iplist'].lower()
        self.firstip_list = externalom_ip_lower.split(',')
        producestorage_ip_lower = \
            self.__db_param_dict['datamover_producestorage_iplist'].lower()
        self.secondip_list = producestorage_ip_lower.split(',')
        if self.__db_param_dict['datamover_backupstorage_iplist'] == '':
            self.thirdip_list = self.secondip_list
        else:
            backupstorage_ip_lower = \
                self.__db_param_dict['datamover_backupstorage_iplist'].lower()
            self.thirdip_list = backupstorage_ip_lower.split(',')
        self.inter_float_ip = \
            self.__db_param_dict['datamover_internal_float_ip'].lower()

        if get_project_condition_boolean(self.project_id, 'eBackup_Proxy'):
            self.assembly_proxy_plane_ip()
        else:
            self.assembly_server_and_proxy_plane_ip()

    def assembly_dmk_conf(self):
        kmc_alg_mode = self.ebackup.get_kmc_alg_mode()
        unroot_list = self.query_ebackup_unroot_microservice_list()
        region_id = self.params.get_param_value(self.pod_id, "public", "region_id", "region0_id")
        global_domain_name = self.__db_param_dict['global_domain_name']
        msuser_password = self.__db_param_dict['eBackup_msuser_password']
        iam_account = self.iam_util.get_iam_account_info("op_svc_csbs")
        config_val = {'msuser_password': msuser_password,
                      "eBackup_NetworkType": self.network_type,
                      'ebackup_unroot_microservice_list': unroot_list,
                      'KmcAlgMode': kmc_alg_mode,
                      'region_id': region_id,
                      'global_domain_name': global_domain_name,
                      'iam_account': iam_account.account_name,
                      'iam_account_password': iam_account.account_pwd,
                      "eBackup_InstallIBDriver": "0",
                      "eBackup_VMType": "1",
                      "eBackup_Template_BackupManagementPlane": "eth0",
                      "eBackup_Template_InternalCommunicationPlane": "eth1",
                      "eBackup_Template_ProductionManagementPlane": "eth0",
                      "eBackup_Template_ProductionStoragePlane": "eth0",
                      "BackupStoragePlane": "eth0",
                      "eBackup_PrimaryNode_EbackupRole": self.primary_role,
                      "eBackup_PrimaryNode_BackupManagementIP": self.primary_manage_ip,
                      "eBackup_PrimaryNode_InternalCommunicationIP": self.primary_inter_ip,
                      "eBackup_PrimaryNode_ProductionManagementIP": self.primary_produce_manage_ip,
                      "eBackup_PrimaryNode_ProductionStorageIP": self.primary_produce_storage_ip,
                      "eBackup_PrimaryNode_BackupStorageIP": self.primary_backup_storage_ip,
                      "eBackup_PrimaryNode_FloatingIP": self.inter_float_ip,
                      "eBackup_PrimaryNode_HcpPassword": self.server_hcp_pwd,
                      "eBackup_PrimaryNode_RootPassword": self.server_root_pwd,
                      "eBackup_StandbyNode_EbackupRole": self.standby_role,
                      "eBackup_StandbyNode_BackupManagementIP": self.standby_manage_ip,
                      "eBackup_StandbyNode_InternalCommunicationIP": self.standby_inter_ip,
                      "eBackup_StandbyNode_ProductionManagementIP": self.standby_produce_manage_ip,
                      "eBackup_StandbyNode_ProductionStorageIP": self.standby_produce_storage_ip,
                      "eBackup_StandbyNode_BackupStorageIP": self.standby_backup_storage_ip,
                      "eBackup_LeaderPublicKey": self.leader_public_key,
                      "eBackup_Region_Id": self.__db_param_dict['region_display_name'],
                      "eBackup_DNSIp": self.datamover_dns_ip}
        host_val = {"eBackup_host": self.firstip_list}
        return config_val, host_val

    def rollback(self, project_id, pod_id):
        """
        标准调用接口：执行回滚
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        try:
            self.exec_dmk_operate(UNINSTALL_ACTION_STR)
            return Message(200)
        except HCCIException as e:
            return Message(500, e)
        except Exception as e:
            logger.error(
                "Failed to rollback:[install]2.Install and "
                "configure eBackup(datamover) ,the reason is %s" % str(e))
            return Message(500, e, e)

    def retry(self, project_id, pod_id):
        """
        标准调用接口：重试
        :return: Message类对象
        """
        return self.execute(project_id, pod_id)

    def check(self, project_id, pod_id):
        """
        标准调用接口：重试
        :param project_id:
        :param pod_id:
        :return:
        """
        pass

    def query_backupnode_pub(self, ebk_ip, is_default_pub, is_server):
        if is_default_pub:
            cmd = "cat /opt/huawei-data-protection/ebackup/conf/cert/" \
                  "defaultBackupNode.pub"
        else:
            cmd = "cat /opt/huawei-data-protection/ebackup/conf/cert/" \
                  "BackupNode.pub"
        if is_server:
            hcp_pwd = self.server_hcp_pwd
            root_pwd = self.server_root_pwd
        else:
            hcp_pwd = self.hcp_user_pwd
            root_pwd = self.root_passwd
        result = self.ssh_obj.ssh_cmds(
            ebk_ip, cmd, "hcp",
            hcp_pwd, root_pwd, "", "")
        if result:
            return result[0]
        else:
            logger.error("query backupnode.pub failed.")
            raise HCCIException(653044, 'backupnode.pub')

    def add_pub_to_server(self, proxy_pub):
        # 使用base64转义，防止命令注入
        proxy_pub_base64 = base64.b64encode(
            proxy_pub.encode("utf-8")).decode(encoding="utf-8")
        if not ebackup_handle.check_base64_string(proxy_pub_base64):
            raise HCCIException(653103, 'proxy_pub')

        cmd = "proxy_uuid=`uuidgen`;cd /opt/huawei-data-protection" \
              "/ebackup/conf/cert/;echo \"$(echo '%s' | base64 -d)\" > $proxy_uuid.pub;" \
              "chown hcpprocess:hcpmgr $proxy_uuid.pub;" \
              "echo \"result: $?\"" % proxy_pub_base64

        result = self.ssh_obj.ssh_cmds(
            self.primary_inter_ip, cmd, "hcp",
            self.server_hcp_pwd, self.server_root_pwd, "", "")
        if "result: 0" in result:
            logger.info("add pub to server success.")
        else:
            logger.error("add pub to server failed, output:%s" % result)
            raise HCCIException(653044, 'eBackup_LeaderPublicKey')

    def get_all_params(self):
        dmk_param_key_list = [
            'dmk_ui_username',
            'dmk_os_business_username',
            'dmk_os_business_user_password']
        dmk_dict = API.get_dmk_info(self.pod_id)
        for key in dmk_param_key_list:
            self.__db_param_dict[key] = dmk_dict[key]
        self.__db_param_dict['dmk_floatIp'] = \
            self.params.get_param_value(
                self.pod_id, 'DMK', 'dmk_floatIp', 'dmk_floatIp')
        self.__db_param_dict['dmk_ui_password'] = self.params.get_param_value(
            self.pod_id, 'DMK', 'dmk_ui_password', 'dmk_ui_password')

        dns = DNSApi()
        dns_type = DNSNodeType()
        dns_ips = dns.get_dns_ips(dns_type.OM, self.pod_id)
        if len(dns_ips) == 2:
            self.datamover_dns_ip = ",".join(dns_ips)
            backup_storageunit_dns_ip = \
                self.__db_param_dict.get('backup_storageunit_dns_ip')
            if backup_storageunit_dns_ip:
                self.datamover_dns_ip = \
                    backup_storageunit_dns_ip + "," + self.datamover_dns_ip
        else:
            raise HCCIException("653025")

    def get_business_user_id(self):
        API.login_dmk(
            self.__db_param_dict['dmk_floatIp'],
            self.__db_param_dict['eBackup_dmk_user'],
            self.__db_param_dict['eBackup_dmk_password'])
        business_user_id = API.get_dmk_account_id(
            self.__db_param_dict['eBackup_dmk_user'],
            self.__db_param_dict['eBackup_dmk_ssh_user']
        )
        return business_user_id

    def exec_dmk_operate(self, action):
        logger.info("datamover: begin the install and config datamover step")
        self.get_all_params()
        dmk_root_account_id = self.get_business_user_id()
        _, ebackup_package_name = API.find_file(
            "OceanStor", "eBackup_DMK.zip")
        if ebackup_package_name == "":
            raise HCCIException(653004, ebackup_package_name)
        ebackup_package_version = API.get_package_version(ebackup_package_name)

        dmk_config_content = self.init_ebackup_config_file()
        ebackup_host = dmk_config_content['hosts_content']
        ebackup_config = dmk_config_content['vars_content']

        # do_action
        logger.info("begin to %s (datamover)" % action)
        dmk_task_info = DmkTaskInfo("eBackup", ebackup_package_version, action, [ebackup_host, ebackup_config])
        is_ok = API.executedmkdeployment(dmk_task_info=dmk_task_info, to_run="true", account_id=dmk_root_account_id)
        if is_ok:
            logger.info("%s (datamover) success." % action)
        else:
            logger.error("%s (datamover)  failed.")
            raise HCCIException(653118, 'datamover')

    def set_ntp(self, service_om_ip):
        dns = DNSApi()
        dns_type = DNSNodeType()
        dns_ips = dns.get_dns_ips(dns_type.OM, self.pod_id)
        if dns_ips:
            if not ebackup_handle.check_ip_valid(dns_ips[0], self.project_id):
                raise HCCIException(653103, 'dns_ips')
            cmd = 'service ntpd stop;ntpdate' + ' ' + dns_ips[0]
            om_ip_list = service_om_ip.split(',')
            for om_ip in om_ip_list:
                ssh_client = self.ssh_obj.ssh_create_client(
                    om_ip, "root", self.root_passwd)
                result = self.ssh_obj.ssh_exec_command_return(ssh_client, cmd)
                self.ssh_obj.ssh_close(ssh_client)
                if "last cmd result: 0\n" in result:
                    logger.info("set ntp successful")
                else:
                    logger.error("set ntp failed on %s" % om_ip)
                    raise HCCIException(653069, om_ip)
            return
        else:
            raise HCCIException(653025, '')

    def update_driver(self, ip_lists: list):
        """
        更新节点驱动
        Returns
        -------

        """
        all_drivers = list()
        driver_infos = dict()
        for node_ip in ip_lists:
            if get_project_condition_boolean(self.project_id,
                                             'eBackupServerARM'):
                cpu_arch = "aarch64"
            else:
                cpu_arch = "x86_64"
            driver_obj = DriverApi(self.project_id,
                                   os_ip_list=[node_ip],
                                   os_username='hcp',
                                   os_password=self.hcp_user_pwd,
                                   root_password=self.root_passwd)
            driver_info = driver_obj.get_nodes_driver_list_by_os(
                component="ebackup", node_type="ebackup",
                arch=cpu_arch).get(node_ip)
            # 驱动列表
            driver_list = list(driver_info.keys())
            logger.info("get node %s driver info: %s" % (node_ip, driver_list))
            if not driver_list:
                logger.info(
                    "There is no driver to need to install in current node[%s]." % node_ip)
                continue
            all_drivers.extend(driver_list)
            driver_infos[node_ip] = driver_list

        logger.info("Install all package on storage nodes")
        all_drivers = list(set(all_drivers))
        driver_and_ip_list = self.sorted_drivers(driver_infos, all_drivers)
        logger.info(
            "Installing All Drivers and Nodes is : %s" % driver_and_ip_list)
        for driver_and_ip in driver_and_ip_list:
            driver, ip_list = driver_and_ip
            driver_obj = DriverApi(self.project_id,
                                   os_ip_list=ip_list,
                                   os_username="hcp",
                                   os_password=self.hcp_user_pwd,
                                   root_password=self.root_passwd)
            driver_obj.install_driver(driver, timeout=600)

    @staticmethod
    def sorted_drivers(driver_infos, all_drivers):
        """
        以驱动为依据，对驱动列表进行分批次
        """
        driver_list = list()
        for driver in all_drivers:
            ip_list = []
            for node in driver_infos:
                if driver in driver_infos.get(node):
                    ip_list.append(node)
            if ip_list:
                data = ([driver], ip_list)
                driver_list.append(data)
        return driver_list

# the code has been updated to python3.7
