# -*- coding:utf-8 -*-
from utils.business.param_util import ParamUtil
from utils.business.project_condition_utils import get_project_condition_boolean
from utils.common import log as logger
from utils.common.exception import HCCIException
from utils.common.fic_base import StepBaseInterface
from utils.common.message import Message

from plugins.eBackup.common.iam_util import IamUtil
from plugins.eBackup.scripts.common.ebackup_util import CustomParametersUtil
from plugins.eBackup.scripts.common.ebackup_util import eBackup
from plugins.eBackup.common.ebackup_rest import EbackupRest


class ConfigIAMAccount(StepBaseInterface):
    def __init__(self, project_id, pod_id):
        logger.init("eBackup")
        # self.params
        self.params = ParamUtil()
        self.project_id = project_id
        self.pod_id = pod_id
        self.rest_api = EbackupRest()
        self.__config_dict = self.params.get_service_cloud_param(pod_id, "eBackup")
        self.hcp_passwd = self.__config_dict['hcp_ssh_password']
        self.root_passwd = self.__config_dict['eBackup_os_root_password']
        self.ebackup = eBackup(self.project_id, self.pod_id)
        self.float_ip = self.ebackup.get_datamover_internal_float_ip()
        self.pararms_util = CustomParametersUtil(project_id, pod_id)
        self.iam_util = IamUtil(self.project_id, self.pod_id)
        self.ebk_iam_headers = {}

    def pre_check(self, project_id, pod_id):
        """
        插件内部接口：执行安装前的资源预检查，该接口由execute接口调用，工具框架不会直接调用此接口。
        :param project_id:
        :param pod_id:
        :return:
        """
        pass

    def main_handle(self):
        self.ebackup.input_8090_iptables_rule()
        self.ebk_iam_headers = self.ebackup.get_iam_tokens(self.float_ip, "8090")
        iam_account = self.iam_util.get_iam_account_info("op_service")
        iam_address = self.pararms_util.get_iam_domain_name()
        self.config_iam_to_datamover(iam_account, iam_address)
        self.ebackup.logout_ebk_iam(self.float_ip, "8090")
        self.ebackup.delete_8090_iptables_rule()

    def execute(self, project_id, pod_id):
        """
        标准调用接口：执行安装前预检查&安装&配置
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        if get_project_condition_boolean(project_id, 'eBackup_Proxy'):
            logger.info("In eBackup_Proxy scenario, no need to config IAM account")
            return Message(200)
        logger.info("begin to config openstack.")
        try:
            self.main_handle()
        except HCCIException as e:
            return Message(500, e)
        except Exception as e:
            logger.error("IAM account config failed ,the reason is %s" % e)
            return Message(500, e, e)
        return Message(200)

    def rollback(self, project_id, pod_id):
        """
        标准调用接口：执行回滚
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        return Message()

    def retry(self, project_id, pod_id):
        """
        标准调用接口：重试
        :return: Message类对象
        """
        return self.execute(project_id, pod_id)

    def check(self, project_id, pod_id):
        """
        标准调用接口：重试
        :param project_id:
        :param pod_id:
        :return:
        """
        pass

    def config_iam_to_datamover(self, account, iam_address):
        data = '{ "ServiceName":["ebk_alarm", "ebk_iam"], "Settings":[{"ConfDomain" : ' \
               '"OpenstackConfig","ConfItem": "OpenstackConfigValue",' \
               '"ConfValue" : "{\\"AuthType\\": \\"IAM\\",' \
               '\\"AuthUserName\\": \\"%s\\",\\"AuthPassword\\": \\"%s\\",' \
               '\\"AuthURL\\": \\"https://%s\\"}"}]}' % \
               (account.account_name, account.account_pwd, iam_address)
        url = "https://%s:8090/v1/srv_governance/service_settings" % self.float_ip

        rsp = self.rest_api.patch(url, self.ebk_iam_headers, data)
        if rsp.status_code != 200:
            logger.error(f"rest req return not 200. status_code = {rsp.status_code}")
            raise HCCIException(653020)
        errcode = rsp.json()["Error"]["Code"]
        if errcode is None:
            logger.error("errcode is empty.")
            raise HCCIException(653020)
        if errcode != 0:
            description = rsp.json()["Error"]["Description"]
            logger.error("config openstack failed, description:%s" % description)
            raise HCCIException(653028, description)
        logger.info("config iam account to datamover succ.")
# the code has been updated to python3.7
