# -*- coding:utf-8 -*-
import os
import traceback

from IPy import IP

from utils.DBAdapter.DBConnector import BaseOps
from utils.business.manageone_cmdb_util import ManageOneCmdbUtil
from utils.business.param_util import ParamUtil
from utils.business.project_condition_utils import get_project_condition_boolean
from utils.business.project_condition_utils import get_project_conditions
from utils.business.project_util import ProjectApi
from utils.business.storage.storage_tool import JoinDistributedStorageInf
from utils.business.storage.storage_tool import JoinParams
from utils.business.storage.storage_tool import StorageTool
from utils.common import log as logger
from utils.common.exception import HCCIException
from utils.common.fic_base import StepBaseInterface
from utils.common.message import Message
from utils.common.ssh_util import Ssh

try:
    from plugins.DistributedStorage.utils.interface.DistributedStorage import DistributedStorageArgs
except ImportError as err:
    logger.error(str(err))

from plugins.eBackup.common.api_adapter import API
import plugins.eBackup.scripts.common.ebackup_util as ebackup_handle


def cmp_ip(start_ip, end_ip):
    if start_ip > end_ip:
        return 1
    elif start_ip == end_ip:
        return 0
    else:
        return -1


class JoinIntoFusionStorageBase(StepBaseInterface):
    def __init__(self, project_id, pod_id):
        super().__init__(project_id, pod_id)
        self.project_id = project_id
        self.pod_id = pod_id
        self.project_api = ProjectApi()
        self.params = ParamUtil()
        self.db = BaseOps()
        self.ssh_obj = Ssh()
        self._config_dict = self.params.get_service_cloud_param(pod_id, "eBackup")
        self.hcp_passwd = self._config_dict['hcp_ssh_password']
        self.root_passwd = self._config_dict['eBackup_os_root_password']
        self.datamover_server_ip = self._config_dict.get('datamover_server_ip')
        self.cmdb_util = ManageOneCmdbUtil(project_id, pod_id)
        self.region_id = self._config_dict['region_id']
        self.fsm_ip = ''
        self.fsm_admin_pwd = ''
        self.index = None
        self.datamover_name = None
        self.join = None
        self.architecture = ""
        self.storage_tool = StorageTool(self.project_id, self.pod_id, self.db)

        project_dic = get_project_conditions(project_id)
        self.is_ipv6 = True if project_dic.get("ManageIPV6") else False
        self.externalom_ip_list = self._config_dict['datamover_externalom_iplist'].lower()
        self.externalom_ip_arr = self.externalom_ip_list.split(',')
        p_storage_ip_list = self._config_dict['datamover_producestorage_iplist'].lower()
        self.p_storage_ip_arr = p_storage_ip_list.split(',')
        self.p_storage_netmask = self._config_dict['datamover_producestorage_netmask'].lower()
        self.p_storage_gateway = self._config_dict['datamover_producestorage_gateway'].lower()
        # 是否是扩容AZ场景
        self.is_bc_installed = project_dic.get('BCInstalled', '')
        if self.is_bc_installed:
            self.externalom_ip_list = self._config_dict['existed_datamover_externalom_iplist'].lower()
            self.externalom_ip_arr = self.externalom_ip_list.split(',')
            p_storage_ip_list = self._config_dict['existed_datamover_producestorage_iplist'].lower()
            self.p_storage_ip_arr = p_storage_ip_list.split(',')
            self.p_storage_netmask = self._config_dict['existed_datamover_producestorage_netmask'].lower()
            self.p_storage_gateway = self._config_dict['existed_datamover_producestorage_gateway'].lower()
        self.fsm_fsdmin_pwd = ''
        self.netcard_name = self.query_netcard_name(self.externalom_ip_arr, self.p_storage_ip_arr)
        self.init_for_join_fs()

    def pre_check(self, project_id, pod_id):
        """
        插件内部接口：执行安装前的资源预检查，该接口由execute接口调用，工具框架不会直接调用此接口。
        :param project_id:
        :param pod_id:
        :return:
        """
        return Message(200)

    def execute(self, project_id, pod_id, *args, **kwargs):
        """
        标准调用接口：执行安装前预检查&安装&配置
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        # 交由子类实现 下同
        return Message(200)

    def rollback(self, project_id, pod_id):
        """
        标准调用接口：执行回滚
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        return Message(200)

    def retry(self, project_id, pod_id):
        """
        标准调用接口：重试
        :return: Message类对象
        """
        return Message(200)

    @staticmethod
    def is_exist_network(network_info_list, start_ip, end_ip):
        if not network_info_list:
            return False

        for network_ip_info in network_info_list:
            net_start_ip = network_ip_info.get("ip_segment", dict()).get('begin_ip', '')
            net_end_ip = network_ip_info.get("ip_segment", dict()).get('end_ip', '')
            if not net_start_ip or not net_end_ip:
                continue
            if start_ip == net_start_ip and net_end_ip == end_ip:
                return True

            or_start_ip_num = IP(start_ip).int()
            or_end_ip_num = IP(end_ip).int()
            start_ip_num = IP(net_start_ip).int()
            end_ip_num = IP(net_end_ip).int()

            if start_ip_num <= or_start_ip_num <= or_end_ip_num <= end_ip_num:
                return True
        return False

    def get_network_info_list(self, ebackup_start_ip, ebackup_end_ip, pacific_ip_segment_list):
        is_exist = self.is_exist_network(pacific_ip_segment_list, ebackup_start_ip, ebackup_end_ip)
        if not is_exist:
            p_storage_network = {'port_name': self.netcard_name,
                                 'ip_segment': {
                                     'begin_ip': ebackup_start_ip,
                                     'end_ip': ebackup_end_ip},
                                 'subnet_prefix': self.p_storage_netmask,
                                 'default_gateway': self.p_storage_gateway}
            pacific_ip_segment_list.append(p_storage_network)
            logger.info("get frontend storage network successful.")
        return pacific_ip_segment_list

    def init_for_join_fs(self):
        self.get_fsm_info()
        node_deploy_info = self.get_nodes(self.externalom_ip_arr)
        ebackup_start_ip, ebackup_end_ip = self.get_net_range(self.p_storage_ip_arr)

        input_args = {"float_ip": self.fsm_ip,
                      "portal_pwd": self.fsm_admin_pwd,
                      "net_type": "storage_frontend",
                      "scenario": "extend",
                      "node_type": 1
                      }
        route_info = {"nodeMgrIps": self.externalom_ip_arr,
                      "portName1": self.netcard_name,
                      "gateway1": self.p_storage_gateway
                      }
        init_network_info = [{'port_name': self.netcard_name,
                              'ip_segment': {'begin_ip': ebackup_start_ip, 'end_ip': ebackup_end_ip},
                              'subnet_prefix': self.p_storage_netmask,
                              'default_gateway': self.p_storage_gateway
                              }]
        join_params = JoinParams(input_args, node_deploy_info, init_network_info, route_info)
        ip_segment_join = JoinDistributedStorageInf(join_params)
        try:
            rsp_result, rsp_data = ip_segment_join.join_ds.query_storage_network(ip_segment_join.join_ds.opr)
        except Exception as error:
            logger.error(f"get ip_segment of storage_network failed:{error}")
            raise HCCIException(653006, str(error)) from error
        if rsp_result.get('code') != 0:
            raise HCCIException(653006, str(rsp_result))

        ip_segment_list = rsp_data.get('ip_list', [{}])
        network_info_list = self.get_network_info_list(ebackup_start_ip, ebackup_end_ip, ip_segment_list)
        join_params = JoinParams(input_args, node_deploy_info, network_info_list, route_info)
        self.join = JoinDistributedStorageInf(join_params)

    def get_fsm_ip(self, project_id, pod_id, project_info_dict):
        if get_project_condition_boolean(project_id,
                                         '(TenantStorFB80|TenantStorHCIFS800)&(ExpansionAZ_KVM|'
                                         'ExpansionPOD_KVM|ExpansionAZ_BMS|ExpansionPOD_BMS)'):
            self.fsm_ip = self.db.get_user_input_cloud_param_by_key(project_id, "expansion_new_fsm_float_ip")
        elif get_project_condition_boolean(project_id, 'TenantStorFBReuse80&ExpansionAZ_KVM'):
            self.fsm_ip = self.db.get_user_input_cloud_param_by_key(project_id, "expansion_az_fsm_float_ip")
        elif get_project_condition_boolean(project_id, 'TenantStorFBReuse80&ExpansionAZ_BMS'):
            self.fsm_ip = self.db.get_user_input_cloud_param_by_key(project_id, "reuse_fsm_float_ip")
        elif get_project_condition_boolean(project_id, 'ExpansionServiceStorage'):
            self.fsm_ip = self.db.get_user_input_cloud_param_by_key(project_id, "expansion_fusionstorage_float_ip")
        else:
            # 获取分布式存储参数FSMOMFloatingIP02
            try:
                self.fsm_ip = DistributedStorageArgs(project_id, pod_id).get_business_storage_float_ip()
            except Exception as error:
                logger.error(str(error))

        if not self.fsm_ip:
            self.fsm_ip = self.db.get_user_input_cloud_param_by_key(project_id, 'FSMOMFloatingIP')
        # 如果是扩容AZ-KVM、扩容AZ-BMS，且新建FusionStorageBlock场景下，业务存储FusionStorageManage浮动IP地址
        if 'expansion_type' in project_info_dict \
                and project_info_dict['expansion_type'] in ['ExpansionAZ_KVM', 'ExpansionAZ_BMS'] \
                and project_info_dict['service_storage_backend_type'] in ['TenantStorFB', 'TenantStorFB80']:
            self.fsm_ip = self.db.get_user_input_cloud_param_by_key(project_id, 'expansion_new_fsm_float_ip')
        if not self.fsm_ip:
            self.fsm_ip = self.storage_tool.get_storage_float_ip(az_mode='manager')
        if not self.fsm_ip:
            logger.error("Can not get fsm_ip.")
            raise HCCIException(653051, "fsm_ip")

    def get_fsm_info(self):
        project_info_dict = self.project_api.get_project_info(self.project_id)
        # 获取fsm的ip、用户名、密码
        self.get_fsm_ip(self.project_id, self.pod_id, project_info_dict)
        self.fsm_admin_pwd = self.db.get_user_input_cloud_param_by_key(self.project_id, 'FsmAdminPasswd')
        if not self.fsm_admin_pwd:
            # 获取分布式存储参数FSMPortalPassword
            try:
                self.fsm_admin_pwd = DistributedStorageArgs(self.project_id, self.pod_id).get_default_fsm_portal_pwd()
            except Exception as error:
                logger.error(f"Can not get fsm_admin_pwd.{error}")
                raise HCCIException(653051, "fsm_admin_pwd") from error
        logger.info("Get fsm info succ.")

    @staticmethod
    def get_net_range(storage_ip_list):

        start_ip = storage_ip_list[0]
        end_ip = storage_ip_list[0]
        for tmp_ip in storage_ip_list:
            if cmp_ip(tmp_ip, start_ip) < 0:
                start_ip = tmp_ip

        for tmp_ip in storage_ip_list:
            if cmp_ip(tmp_ip, end_ip) > 0:
                end_ip = tmp_ip
        return start_ip, end_ip

    def do_query_pwd(self, project_id, pod_id, mg_node_ip):
        account_info = API.get_account_info(project_id, pod_id, 'eBackupServer')
        account_dict = account_info[mg_node_ip]
        if account_dict['hcp']:
            self.hcp_passwd = account_dict['hcp']
            self.root_passwd = account_dict['root']
        logger.info("Query pwd from manageone succ.")

    def query_pwd(self, project_id, pod_id, mg_node_ip):
        try:
            self.do_query_pwd(project_id, pod_id, mg_node_ip)
        except Exception as e:
            logger.error("Query pwd from manageone failed:%s", str(e))

    def check(self, project_id, pod_id):
        """
        标准调用接口：重试
        :param project_id:
        :param pod_id:
        :return:
        """
        return Message(200)

    def query_netcard_name(self, externalom_ip_arr, p_storage_ip_arr):
        ebk_storage_ip = p_storage_ip_arr[0]

        if not ebk_storage_ip:
            logger.error("Can not get business storage ip of ebackup.")
            raise HCCIException(653052, "datamover_producestorage_iplist")
        first_node_ip = externalom_ip_arr[0]
        if not ebackup_handle.check_ip_valid(ebk_storage_ip, self.project_id):
            raise HCCIException(653103, "datamover_producestorage_iplist")
        if self.is_bc_installed:
            self.query_pwd(self.project_id, self.pod_id, first_node_ip)
        if self.is_ipv6:
            self.ssh_obj.ssh_cmds(
                first_node_ip, "rm -rf /home/hcp/QueryNetworkCardName.sh", "hcp",
                self.hcp_passwd, self.root_passwd, "", "")
            query_netcard_name_cmd = "chmod 550 QueryNetworkCardName.sh 1>/dev/null 2>/dev/null;" \
                                     "dos2unix QueryNetworkCardName.sh 1>/dev/null 2>/dev/null;" \
                                     "/bin/bash QueryNetworkCardName.sh " + ebk_storage_ip
            self.ssh_obj.put_file(first_node_ip, "hcp", self.hcp_passwd,
                                  "/opt/cloud/hcci/src/HCCI/plugins/eBackup/shell_tool/QueryNetworkCardName.sh")
        else:
            query_netcard_name_cmd = "ip route | grep '" + ebk_storage_ip + "'| awk  '{print $3}'"
        ssh_client = self.ssh_obj.ssh_create_client(first_node_ip, "hcp", self.hcp_passwd)
        result = self.ssh_obj.ssh_exec_command_return_list(ssh_client, query_netcard_name_cmd)
        self.ssh_obj.ssh_close(ssh_client)
        if result:
            logger.info("Execute query netcard name successful")
            return result[0]
        else:
            logger.error("Execute query netcard name failed")
            raise HCCIException(653013, query_netcard_name_cmd)

    def install_api_package(self, ip_list):
        dsware_api_cert_file = "/home/pkg/distribute_storage_cert_file"
        path, name = API.find_file("OceanStor-Pacific_", "_API.tar.gz")
        file_path = os.path.join(path, name)
        if not os.path.exists(file_path):
            raise HCCIException(653004, "OceanStor-Pacific_*.tar.gz")
        ebackup_handle.check_compress_file([file_path])
        self.fsm_fsdmin_pwd = self.db.get_user_input_cloud_param_by_key(self.project_id, 'FSMfsdminPasswd')
        if not self.fsm_fsdmin_pwd:
            # 获取分布式存储参数FSMfsdminPassword
            try:
                self.fsm_fsdmin_pwd = DistributedStorageArgs(self.project_id, self.pod_id).get_default_fsm_fsadmin_pwd()
            except Exception as error:
                logger.error(f"Can not get fsm_fsdmin_pwd.{error}")
                raise HCCIException(653051, "fsm_fsdmin_pwd") from error
        storage_info = {'fsadmin_pwd': self.fsm_fsdmin_pwd, 'portal_pwd': self.fsm_admin_pwd}
        cert_info = self.storage_tool.get_fsc_cli_cert_file(dsware_api_cert_file, self.fsm_ip, storage_info)
        password = cert_info.get('password')
        for datamover_ip in ip_list.split(','):
            if self.is_bc_installed:
                self.query_pwd(self.project_id, self.pod_id, datamover_ip)
            self.ssh_obj.put_file(datamover_ip, "hcp", self.hcp_passwd, file_path, destination='.', port=22)
            self.update_cert_file(datamover_ip, cert_info)
            self.install_api(file_path, datamover_ip)
            self.replacing_cert(password, datamover_ip)

    def update_cert_file(self, datamover_ip, cert_info):
        ca_file = cert_info.get('ca_file')
        cert_file = cert_info.get('cert_file')
        key_file = cert_info.get('key_file')
        self.ssh_obj.put_file(datamover_ip, "hcp", self.hcp_passwd, ca_file, destination='.', port=22)
        self.ssh_obj.put_file(datamover_ip, "hcp", self.hcp_passwd, cert_file, destination='.', port=22)
        self.ssh_obj.put_file(datamover_ip, "hcp", self.hcp_passwd, key_file, destination='.', port=22)

    def install_api(self, file_path, datamover_ip):
        name = file_path.replace("/home/pkg/", "")
        if file_path.endswith("zip"):
            uncompress_cmd = "unzip -o '" + name + "'"
        else:
            uncompress_cmd = "tar -zxvf '" + name + "'"

        check_archi_cmd = "lscpu |grep  aarch64; echo execute check result: $?"
        result = self.ssh_obj.ssh_cmds(datamover_ip, check_archi_cmd, "hcp", self.hcp_passwd, self.root_passwd, "", "")
        if "execute check result: 0" in result:
            self.architecture = "aarch64"
        else:
            self.architecture = "x86-64"

        if not ebackup_handle.check_ip_valid(self.fsm_ip, self.project_id):
            raise HCCIException(653103, "fsm_ip")

        cmd = "chattr -i /opt/huawei-data-protection/ebackup/vbstool/;chattr -i -R /opt/huawei-data-protection/" \
              "ebackup/vbstool/lib; chattr -i -R /opt/huawei-data-protection/ebackup/vbstool/conf;" + uncompress_cmd + \
              "; mkdir -p /opt/huawei-data-protection/ebackup/vbstool/lib/scripts/;" \
              "mkdir -p /opt/huawei-data-protection/ebackup/tmp/javarunenv/;" \
              "tar -zmxvf /opt/dsware/agent/jre-*.tar.gz -C /opt/huawei-data-protection/ebackup/tmp/javarunenv/;" \
              "/bin/cp -r ca_file.ca cert_file.cert key_file.ca " \
              "/opt/huawei-data-protection/ebackup/vbstool/lib/scripts/ ;cd OceanStor-Pacific_*_*/;" \
              "rm -rf lib/log4j-* ;sed -i 's#export LD_LIBRARY_PATH=.*#export LD_LIBRARY_PATH" \
              "=$LD_LIBRARY_PATH:/opt/huawei-data-protection/ebackup/vbstool/lib#g' /opt/huawei-data-protection/" \
              "ebackup/vbstool/vrmVBSTool.sh;" \
              "/bin/cp  client_self.keystore client_trust.keystore dsware-api.properties fsa_server.key manager-ssl." \
              "properties primary_ks.key standby_ks.key /opt/huawei-data-protection/ebackup/vbstool/conf/;" \
              "/bin/cp dr_cli.xml readme.txt version zk-client.jks /opt/huawei-data-protection/ebackup/vbstool/;" \
              "/bin/cp dsware-api-*.jar /opt/huawei-data-protection/ebackup/vbstool/lib/;" \
              "/bin/cp  primary_ks.key  standby_ks.key /opt/huawei-data-protection/ebackup/vbstool/lib/;" \
              "/bin/cp -r scripts/* /opt/huawei-data-protection/ebackup/vbstool/lib/scripts/;" \
              "/bin/cp -r lib/* /opt/huawei-data-protection/ebackup/vbstool/lib/;" \
              "touch /opt/huawei-data-protection/ebackup/vbstool/storage_port.ini;" \
              "/bin/cp -r /opt/huawei-data-protection/ebackup/vbstool/lib/linux-%s/* " \
              "/opt/huawei-data-protection/ebackup/vbstool/lib/" % self.architecture
        self.ssh_obj.ssh_cmds(datamover_ip, cmd, "hcp", self.hcp_passwd, self.root_passwd, "", "")
        logger.info(f"Execute install api package succ: {datamover_ip}")

    def replacing_cert(self, password, datamover_ip):
        if not ebackup_handle.check_ip_valid(self.fsm_ip, self.project_id):
            raise HCCIException(653103, "fsm_float_ip")
        cmd = "export LD_LIBRARY_PATH=/usr/lib64:/opt/huawei-data-protection/ebackup/vbstool/lib/linux-%s;" \
              "sh /opt/huawei-data-protection/ebackup/vbstool/lib/scripts/replace_cert.sh " \
              "-s /opt/huawei-data-protection/ebackup/vbstool/lib/scripts/cert_file.cert " \
              "-c /opt/huawei-data-protection/ebackup/vbstool/lib/scripts/ca_file.ca " \
              "-k /opt/huawei-data-protection/ebackup/vbstool/lib/scripts/key_file.ca " \
              "-f %s -j /opt/huawei-data-protection/ebackup/tmp/javarunenv/*jre* " % (self.architecture, self.fsm_ip)
        check_cmd = "echo last cmd res:$?"
        restore_cmd = "chattr +i /opt/huawei-data-protection/ebackup/vbstool/ ;chattr +i -R " \
                      "/opt/huawei-data-protection/ebackup/vbstool/lib; chattr +i -R " \
                      "/opt/huawei-data-protection/ebackup/vbstool/conf"
        sshclient = self.ssh_obj.ssh_create_client(datamover_ip, "hcp", self.hcp_passwd)
        self.ssh_obj.ssh_send_command(sshclient, 'su', 'assword:', 100)
        self.ssh_obj.ssh_send_command(sshclient, self.root_passwd, '#', 100)
        self.ssh_obj.ssh_send_command(sshclient, cmd, 'password', 100)
        self.ssh_obj.ssh_send_command(sshclient, password, '#', 100)
        result = self.ssh_obj.ssh_exec_command_return_list(sshclient, check_cmd, timeout=30)
        if str(result).find('last cmd res:0') == -1:
            self.ssh_obj.ssh_send_command(sshclient, restore_cmd, '#', 100)
            raise HCCIException(653104, datamover_ip)
        copy_cmd = "grep -E '8.1.3|8.1.5' /opt/huawei-data-protection/ebackup/vbstool/version && " \
                   f"/bin/cp /opt/huawei-data-protection/ebackup/vbstool/conf/cert/{self.fsm_ip}/client_* " \
                   "/opt/huawei-data-protection/ebackup/vbstool/conf"
        self.ssh_obj.ssh_send_command(sshclient, copy_cmd, '#', 100)
        self.ssh_obj.ssh_send_command(sshclient, restore_cmd, '#', 100)
        logger.info(f"Execute replace api cert succ: {datamover_ip}")

    @staticmethod
    def get_ebackup_deploy_scene(project_id):
        is_ebackup_driver = get_project_condition_boolean(project_id, 'eBackup_ServerProxy_Driver')
        is_proxy = get_project_condition_boolean(project_id, 'eBackup_Proxy')
        return is_ebackup_driver, is_proxy

    @staticmethod
    def get_ebackup_index_on_cmdb(node_info_list):
        index = 1
        node_name_list = []
        if node_info_list:
            for node_info in node_info_list:
                name_part_list = node_info['name'].split('_')
                if len(name_part_list) == 3:
                    node_name_list.append(name_part_list[1])
        if node_name_list:
            new_node_name_list = list(set(node_name_list))
            index = len(new_node_name_list) + 1
        datamover_name = "eBackup_service" + str(index)
        return datamover_name

    def get_existed_ebackup_index_on_cmdb(self, node_info_list):
        datamover_name = ''
        for node_info in node_info_list:
            node_info_addresses = node_info['ipAddresses']
            if node_info_addresses:
                ips = list(map(lambda x: x.get('ip'), node_info_addresses))
                if self.datamover_server_ip in ips:
                    datamover_name = node_info['name']
                if datamover_name:
                    break
        return datamover_name

    def get_proxy_cmdb_name_prefix(self, node_info_list):
        datamover_name = self.get_existed_ebackup_index_on_cmdb(node_info_list)
        datamover_name_list = datamover_name.split('_')
        sub_name = "%s_%s" % (datamover_name_list[0], datamover_name_list[1])
        node_name_list = []
        for node_info in node_info_list:
            if node_info['name'].find(sub_name) >= 0:
                node_name_list.append(node_info)
        index = len(node_name_list)
        datamover_name = sub_name
        return index, datamover_name

    def get_ebackup_cmdb_name_prefix(self, node_info_list):
        is_ebackup_driver, is_proxy = self.get_ebackup_deploy_scene(self.project_id)
        try:
            if is_ebackup_driver:
                self.index = 0
                self.datamover_name = self.get_ebackup_index_on_cmdb(node_info_list)
            elif is_proxy:
                self.index, self.datamover_name = self.get_proxy_cmdb_name_prefix(node_info_list)
        except Exception as e:
            logger.warn(f"Failed to get ebackup cmdb name prefix, error:{str(e)} traceback:\n {traceback.format_exc()}")

    def get_ebackup_cmdb_name(self, ip, node_info_list):
        is_ebackup_driver, is_proxy = self.get_ebackup_deploy_scene(self.project_id)
        if is_ebackup_driver or is_proxy:
            self.index += 1
            return self.datamover_name + "_node" + str(self.index)
        for node_info in node_info_list:
            node_info_addresses = node_info.get('ipAddresses', '')
            if not node_info_addresses:
                continue
            ips = list(map(lambda x: x.get('ip'), node_info_addresses))
            if ip in ips:
                datamover_name = node_info['name']
                return datamover_name
        return ""

    def get_nodes(self, ip_list):
        node_list = []
        node_deploy_list = []
        node_info_list = []
        try:
            node_info_list = self.cmdb_util.get_deploy_node_info_for_mo(self.region_id, "eBackup")
        except Exception as e:
            logger.warn(f"Failed to get ebackup node info, error:{str(e)} traceback:\n {traceback.format_exc()}")
        self.get_ebackup_cmdb_name_prefix(node_info_list)
        for ip in ip_list:
            ebackup_host_name = "eBackup" + ip
            ebackup_cmdb_name = ""
            try:
                ebackup_cmdb_name = self.get_ebackup_cmdb_name(ip, node_info_list)
            except Exception as e:
                logger.warn(f"Failed to get ebackup cmdb name, error:{str(e)} traceback:\n {traceback.format_exc()}")
            if ebackup_cmdb_name:
                ebackup_host_name = ebackup_cmdb_name
            if self.is_bc_installed:
                self.query_pwd(self.project_id, self.pod_id, ip)
            node_ebackup = {"serial_number": "",
                            "name": ebackup_host_name,
                            "management_internal_ip": ip,
                            "role": "compute",
                            "cabinet": 2,
                            "subrack": "",
                            "slot_number": "",
                            "authentication_mode": "password",
                            "user_name": "hcp",
                            "password": self.hcp_passwd,
                            "root_password": self.root_passwd}
            node_list.append(node_ebackup)
            node_deploy_list.append(ip)
        return node_list


class JoinIntoFusionStorageCluster(JoinIntoFusionStorageBase):
    def execute(self, project_id, pod_id, *args, **kwargs):
        try:
            self.join.deploy
        except HCCIException as e:
            return Message(500, e)
        except Exception as e:
            logger.error(f"Join into Huawei Distributed Block Storage cluster failed: {str(e)}")
            return Message(500, e, e)
        logger.info("Join in fs succ.")
        return Message(200)

    def rollback(self, project_id, pod_id):
        try:
            self.join.rollback
        except HCCIException as e:
            return Message(500, e)
        except Exception as e:
            logger.error(f"Join into Huawei Distributed Block Storage cluster rollback failed: {str(e)}")
            return Message(500, e, e)
        logger.info("Rollback succ.")
        return Message(200)

    def retry(self, project_id, pod_id):
        try:
            self.join.retry
        except HCCIException as e:
            return Message(500, e)
        except Exception as e:
            logger.error(f"Join into Huawei Distributed Block Storage cluster retry failed: {str(e)}")
            return Message(500, e, e)
        logger.info("Join in fs succ.")
        return Message(200)


class InstallAPIPackage(JoinIntoFusionStorageBase):

    def execute(self, project_id, pod_id, *args, **kwargs):
        try:
            self.install_api_package(self.externalom_ip_list)
        except HCCIException as e:
            return Message(500, e)
        except Exception as e:
            logger.error(f"install api package failed: {str(e)}")
            return Message(500, e, e)
        logger.info(f"Install api package succ.")
        return Message(200)

    def retry(self, project_id, pod_id):
        return self.execute(project_id, pod_id)

# the code has been updated to python3.7
