from utils.business.install_csp_agent import CSPAgent
from utils.business.manageone_cmdb_util import ManageOneCmdbUtil
from utils.business.param_util import ParamUtil
from utils.common import log as logger
from utils.common.exception import HCCIException
from utils.common.fic_base import StepBaseInterface
from utils.common.message import Message

logger.init("eBackup")


class InstallCspAgent(StepBaseInterface):
    def __init__(self, project_id, pod_id):
        super(InstallCspAgent, self).__init__(project_id, pod_id)
        self.project_id = project_id
        self.pod_id = pod_id
        self.cmdb_util = ManageOneCmdbUtil(project_id, pod_id)
        self.service_name = "eBackup"
        self.params = ParamUtil()
        self.region_id = \
            self.params.get_param_value(self.pod_id, "public",
                                        "region_id", "region0_id")
        self.user_name = "hcp"
        self.root_pwd = self.params.get_param_value(
            pod_id,
            "eBackup",
            "eBackup_os_root_password",
            "eBackup_server_node_root_pwd")
        self.hcp_pwd = self.params.get_param_value(
            pod_id,
            "eBackup",
            "hcp_ssh_password",
            "eBackup_server_node_hcp_pwd")

    def pre_check(self, project_id, pod_id):
        """Plug-in internal interface.

        Perform resource pre-check before installation,
        this interface is called by the execute interface,
        The tool framework does not directly call this interface.
        :param project_id:
        :param pod_id:
        :return:
        """
        return Message(200)

    def main_handle(self):
        node_list = self._get_node_list()
        logger.info("Start to install csp agent for {} service.".format(self.service_name))
        ret_dic = CSPAgent(self.pod_id).install_csp_agent(service_name=self.service_name, node_list=node_list)

        if ret_dic['result_code'] != '0':
            raise Exception("Install csp agent failed.")
        logger.info("Succeed to install csp agent for {} service.".format(self.service_name))
        return Message(200)

    def execute(self, project_id, pod_id, *args, **kwargs):
        """Plug-in internal interface.

        Perform installation & configuration.
        :param project_id:
        :param pod_id:
        :return:Message object
        """
        try:
            return self.main_handle()
        except HCCIException as e:
            logger.error("Failed to install csp agent, err_msg:{}.".format(str(e)))
            return Message(500, e)
        except Exception as e:
            logger.error("Failed to execute InstallCSPAgent, err_msg:{}.".format(str(e)))
            return Message(500, HCCIException(653082))

    def rollback(self, project_id, pod_id):
        """Plug-in internal interface.

        Perform job job failed rollback.
        :param project_id:
        :param pod_id:
        :return:Message object
        """
        return self.execute(project_id, pod_id)

    def retry(self, project_id, pod_id):
        """Plug-in internal interface.

        Perform job failed retry.
        :return: Message object
        """
        return self.execute(project_id, pod_id)

    def check(self, project_id, pod_id):
        """Plug-in internal interface.

        Check before task execution.
        :param project_id:
        :param pod_id:
        :return:
        """
        pass

    def _get_node_list(self):
        ebackup_ip_list = self.get_service_ip_list_from_cmdb(self.region_id,
                                                             self.service_name)
        logger.info("Succeed to obtain  ips, "
                    "ip list is:{}.".format(ebackup_ip_list))
        node_list = []
        for ip in ebackup_ip_list:
            node_info = {
                "node_ip": ip,
                "process_username": self.user_name,
                "user_password": self.hcp_pwd,
                "sudo_cmd": "su",
                "root_password": self.root_pwd
            }
            node_list.append(node_info)
        return node_list

    def get_service_ip_list_from_cmdb(self, region_id, service_name):
        nodes_info = \
            self.cmdb_util.get_deploy_node_info(region_code=region_id,
                                                index_name=service_name)
        logger.info("Get {} service nodes info: {}.".format(service_name,
                                                            nodes_info))
        return [node_info.get("mgmtIp")
                for node_info in nodes_info if node_info.get("mgmtIp")]
