# -*- coding:utf-8 -*-
import os
import stat
import time
import traceback
import requests

from utils.common.fic_base import StepBaseInterface
from utils.business.manageone_util2 import ManageOneUtil2
from utils.business.param_util import ParamUtil
from utils.common.check_result import CheckResult
from utils.common.message import Message
import utils.common.log as logger
from utils.common.exception import HCCIException

from plugins.eBackup.common.util import Utils


class MigrateBase(StepBaseInterface):
    def __init__(self, project_id, pod_id, regionid_list=None):
        super(MigrateBase, self).__init__(project_id, pod_id, regionid_list)
        self.project_id = project_id
        self.pod_id = pod_id
        self.regionid_list = regionid_list
        self.base_url = ""
        self.headers = ""
        self.token = ""
        self.session = ""
        self.manageone = ManageOneUtil2()
        self.workflow_role = 2
        util = ParamUtil()
        self.__db_param_dict = util.get_service_cloud_param(project_id, "eBackup", self.regionid_list[0])
        self.workflow_ips = [
            self.__db_param_dict["eBackup_Workflow_nodes"], self.__db_param_dict["eBackup_Manager_nodes"]]
        mo_public_params = ParamUtil().get_service_cloud_param(
            self.project_id, "ManageOne", self.regionid_list[0])
        global_domain_name = mo_public_params.get("ManageOne_global_domain_name")
        self.hcp_passwd = self.__db_param_dict['eBackup_hcp_pwd']
        self.root_passwd = self.__db_param_dict['eBackup_root_pwd']
        self.admin_passwd = self.__db_param_dict['eBackup_admin_pwd']
        self.alarm_url = "oc." + self.regionid_list[0] + "." + global_domain_name + ":26335"
        self.config_file = os.path.realpath(__file__ + '/../config.ini')

    def login_portal(self, float_ip):
        ip_version = Utils.check_ip_version(float_ip)
        self.token, self.session = Utils.login(float_ip, self.admin_passwd, ip_version)
        self.headers = {
            'Accept': 'application/json;version=2.2;charset=UTF-8',
            'iBaseToken': self.token,
            'Cookie':
                'language=en;' + self.session + '; DEVICE_ID=dev; sessionIdleTime=60000; '
                                                'MACHINEROLE=0; CSRF_IBASE_TOKEN=' + self.token
        }

    def set_alarm_domain(self, oc_domain):
        oc_username = "thirdparty"
        oc_password = self.__db_param_dict['eBackup_alarm_pwd']
        url = self.base_url + 'alarm_settings/default'
        if not oc_username or not oc_password:
            raise Exception("get alarm info failed")
        logger.info("alarm url:" + oc_domain)
        data = '{"ISALARMON":1,"ALARMSERVICETYPE":"1","ALARMSERVERURL":"' + \
               oc_domain + '","ALARMUSERNAME":"' + oc_username + \
               '","ALARMPASSWORD":"' + oc_password + '"}'

        num = 0
        # 重试12次
        while num < 12:
            rsp = requests.put(url=url, headers=self.headers, data=data, verify=False)
            if rsp.status_code != 200:
                logger.error("rest req return not 200.")
                return
            errcode = rsp.json()["error"]["code"]
            description = rsp.json()["error"]["description"]
            if errcode is None or description is None:
                logger.error("errcode or description is empty.")
                return
            if errcode != 0:
                logger.error("config alarm report failed,description:" + description)
                time.sleep(60)
                num = num + 1
            else:
                logger.info("config alarm report succ.")
                return
            logger.error("config alarm report failed,description:" + description)

    def rollback(self, project_id, pod_id, regionid_list=None):
        return Message(200)

    def retry(self, project_id, pod_id, regionid_list=None):
        return self.execute(project_id, pod_id, regionid_list)

    def execute(self, project_id, pod_id, regionid_list=None):
        return Message(200)


class CheckPortalAccount(MigrateBase):
    def execute(self, project_id, pod_id, regionid_list=None):
        token = ''
        session = ''
        ip_version = None
        result = CheckResult(
            itemname_ch="检查admin账户是否可登录",
            itemname_en="Check whether the admin account can be used for login.",
            status="success"
        )
        try:
            float_ip = Utils.find_float_ip(self.workflow_ips, self.hcp_passwd, self.root_passwd)
        except Exception as e:
            logger.error(traceback.format_exc() + "\n" + str(e))
            return Message(500, HCCIException(650017, self.workflow_ips))
        ip_version = Utils.check_ip_version(float_ip)
        try:
            token, session = Utils.login(float_ip, self.admin_passwd, ip_version)
        except Exception as e:
            logger.error(traceback.format_exc() + "\n" + str(e))
            return Message(500, HCCIException(650009, float_ip))
        finally:
            Utils.logout(float_ip, token, session, self.workflow_role, ip_version)
        return Message(200, check_results=[result])


class AlarmDomainChange(MigrateBase):
    def save_domain_name_to_file(self, domain_name):
        def do_save_domain_name_to_file():
            flags = os.O_WRONLY | os.O_CREAT
            modes = stat.S_IWUSR | stat.S_IRUSR
            with os.fdopen(os.open(self.config_file, flags, modes), 'w') as fp:
                fp.truncate()
            with open(self.config_file, mode='a') as fp:
                fp.write(domain_name)

        try:
            do_save_domain_name_to_file()
        except Exception as e:
            logger.error(traceback.format_exc() + "\n" + str(e))
            logger.error("Failed to save domain name to file. domain name:" + domain_name)

    def execute(self, project_id, pod_id, regionid_list=None):
        def do_execute():
            float_ip = Utils.find_float_ip(self.workflow_ips, self.hcp_passwd, self.root_passwd)
            ip_version = Utils.check_ip_version(float_ip)
            if ip_version == 4:
                self.base_url = f'https://{float_ip}:8088/rest/dev/'
            else:
                self.base_url = f'https://[{float_ip}]:8088/rest/dev/'
            cur_domain_name = Utils.query_alarm_info(float_ip, self.admin_passwd)
            if not cur_domain_name:
                logger.warn("current domain name is none.")
            else:
                cur_domain_name = cur_domain_name.replace("https://", "") + ":26335"
                logger.info("current domain name is " + cur_domain_name)
                self.save_domain_name_to_file(cur_domain_name)

            self.login_portal(float_ip)
            self.set_alarm_domain(self.alarm_url)
            return Message(200)
        try:
            return do_execute()
        except Exception as e:
            logger.error(traceback.format_exc() + "\n" + str(e))
            return Message(500, HCCIException(650045))


class AlarmDomainRollback(MigrateBase):
    def get_domain_name_from_file(self):
        domain_name = ""
        flags = os.O_WRONLY | os.O_CREAT
        modes = stat.S_IWUSR | stat.S_IRUSR
        try:
            with os.fdopen(os.open(self.config_file, flags, modes), 'r') as fp:
                domain_name = fp.read().strip()
        except Exception as e:
            logger.error(traceback.format_exc() + "\n" + str(e))
            logger.error("Failed to save domain name to file. domain name:" + domain_name)
        return domain_name

    def execute(self, project_id, pod_id, regionid_list=None):
        def do_rollback():
            float_ip = Utils.find_float_ip(self.workflow_ips, self.hcp_passwd, self.root_passwd)
            ip_version = Utils.check_ip_version(float_ip)
            if ip_version == 4:
                self.base_url = f'https://{float_ip}:8088/rest/dev/'
            else:
                self.base_url = f'https://[{float_ip}]:8088/rest/dev/'
            domain_name = self.get_domain_name_from_file()
            if not domain_name:
                logger.info("domain name is null. no need rollback.")
                return Message(200)

            self.login_portal(float_ip)
            self.set_alarm_domain(domain_name)
            return Message(200)
        try:
            return do_rollback()
        except Exception as e:
            logger.error(traceback.format_exc() + "\n" + str(e))
            return Message(500, HCCIException(650045))
