# -*- coding:utf-8 -*-
import utils.common.log as logger
from utils.DBAdapter.DBConnector import BaseOps
from utils.PasswordManager.PasswordManager import FSPasswordManager
from utils.business.OpenStack.cps_util import HostsInfoByPod
from utils.business.param_util import ParamUtil
from utils.business.project_condition_utils import get_project_condition_boolean
from utils.client.FSPAuthClient import FSPAuthClient
from utils.common.exception import HCCIException
from utils.common.fic_base import StepBaseInterface
from utils.common.message import Message
from utils.common.ssh_util import Ssh

import plugins.eBackup.scripts.common.ebackup_util as ebackup_handle
from plugins.eBackup.scene.deploy.params.params_check import check_openstack_region_sub
from plugins.eBackup.scene.deploy.params.params_check import check_param_string
from plugins.eBackup.scripts.upgrade.util import Utils
from plugins.eBackup.common.model import SshInfo

logger.init("eBackup CheckDriverStatus")


class CheckDriverStatus(StepBaseInterface):
    def __init__(self, project_id, pod_id):
        self.params = ParamUtil()
        self.__config_dict = self.params.get_service_cloud_param(pod_id, "eBackup")
        self.region_id = self.params.get_param_value(pod_id, "public", "region_id", "region0_id")
        self.project_id = project_id
        self.pod_id = pod_id
        self.ssh_obj = Ssh()
        self.rest_client = FSPAuthClient.get_cps_rest_client(BaseOps(), self.project_id, self.pod_id)
        self.cmd = "service hcp status"
        self.clear_history_cmd = ">~/.bash_history&&history -c&&rm -rf /home/fsp/CheckDriverStatus.sh"
        self.root_passwd = self.__config_dict['eBackup_os_root_password']
        self.fsp_user_pwd = self.__config_dict.get('fsp_user_pwd')
        self.fsp_root_pwd = self.__config_dict.get('fsp_root_pwd')
        self.fsp_env_pwd = self.__config_dict.get('fsp_env_pwd')

    def pre_check(self, project_id, pod_id):
        """
        插件内部接口：执行安装前的资源预检查，该接口由execute接口调用，工具框架不会直接调用此接口。
        :param project_id:
        :param pod_id:
        :return:
        """
        pass

    def execute(self, project_id, pod_id, *args, **kwargs):
        """
        标准调用接口：执行安装前预检查&安装&配置
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        try:
            self.exec_check_driver_status(project_id)
        except HCCIException as err:
            return Message(500, err)
        except Exception as err:
            logger.error("check driver status error :%s" % str(err))
            return Message(500, err, err)
        return Message(200)

    def exec_check_driver_status(self, project_id):
        if get_project_condition_boolean(project_id, 'eBackup_Proxy'):
            return
        self.get_account_info()
        server_ip = self.get_server_ip()
        fsp_node_info = HostsInfoByPod(self.project_id, self.pod_id)
        fsp_node_list = fsp_node_info.get_ebk_driver_hosts_info()

        self.check_driver_status(fsp_node_list, server_ip)

    def get_cinder_backup_templates(self, az_ids):
        templates = []
        template_list = self.rest_client.get_template_list("cinder")
        for template in template_list:
            if not template.startswith("cinder-backup"):
                continue
            if not check_param_string(template):
                continue
            template_detail = self.rest_client.get_template_params("cinder", template)
            driver_datas = template_detail.get("driver_data", dict())
            if not driver_datas:
                logger.info(f"Current template {template} has no driver data")
                continue
            current_azs = []
            for data in driver_datas.values():
                if data:
                    current_azs.append(data.get('availability_zone', ''))
            for az_id in az_ids:
                if az_id in current_azs and template not in templates:
                    templates.append(template)
        if not templates:
            raise Exception("Get cinder-backup template return None.")
        logger.info(f"Get cinder-backup template return {templates}.")
        return templates

    def login_openstack(self, ssh_client):
        Ssh.ssh_send_command(ssh_client, "TMOUT=0", "#", 300)
        Ssh.ssh_send_command(ssh_client, "source set_env", "1|2", 300)
        Ssh.ssh_send_command(ssh_client, "2", "CPS_USERNAME", 300)
        Ssh.ssh_send_command(ssh_client, "cps_admin", "CPS_PASSWORD", 300)
        resp = Ssh.ssh_send_command(ssh_client, self.fsp_env_pwd, "#", 300)
        if [x for x in resp if x.find("Authenticate Failed") >= 0]:
            raise HCCIException(650051)

    def clean_up_env(self, ssh_client):
        cmd = "sed -i '/CPS_PASSWORD/d' ~/.bash_history;sed -i '/CPS_PASSWORD/d' /var/log/messages"
        Utils.ssh_send_comand_return(self.ssh_obj, cmd)
        Ssh.ssh_close(ssh_client)
        logger.info("OpenStack SSH client is closed")

    def check_driver_status(self, fsp_node_list, server_ip):
        for sub_value in self.__config_dict['az_related_to_datamover'].split(','):
            if not check_openstack_region_sub(sub_value):
                raise HCCIException(653103, 'az_related_to_datamover')
        if not ebackup_handle.check_ip_valid(server_ip, self.project_id):
            raise HCCIException(653103, server_ip)
        templates = self.get_cinder_backup_templates(self.__config_dict['az_related_to_datamover'].split(','))
        templates_str = ','.join(templates)
        fsp_ssh_info = SshInfo(fsp_node_list.pop(), 'fsp', self.fsp_user_pwd, self.fsp_root_pwd)
        ssh_client = Utils.get_ssh_client(fsp_ssh_info)
        self.login_openstack(ssh_client)
        Ssh.ssh_send_command(ssh_client, "rm -rf /home/fsp/CheckDriverStatus.sh 2>/dev/null 1>&2", "#", 300)
        self.ssh_obj.put_file(fsp_ssh_info.node_ips, "fsp", self.fsp_user_pwd,
                              "/opt/cloud/hcci/src/HCCI/plugins/eBackup/shell_tool/CheckDriverStatus.sh",
                              "/home/fsp", 22)
        logger.info(f"put file to {fsp_ssh_info.node_ips} succ.")
        Ssh.ssh_send_command(ssh_client, "chmod 550 /home/fsp/CheckDriverStatus.sh", "#", 300)
        Ssh.ssh_send_command(ssh_client, "dos2unix /home/fsp/CheckDriverStatus.sh", "dos2unix", 300)
        result = Ssh.ssh_send_command(ssh_client,
                                      f"/bin/bash /home/fsp/CheckDriverStatus.sh '{templates_str}' {server_ip}",
                                      "status is ok", 300)
        self.clean_up_env(ssh_client)
        logger.info(f"clear history succ in {fsp_ssh_info.node_ips}")
        if "status is ok.\n" in result:
            logger.info("driver status is ok in " + fsp_ssh_info.node_ips)
        else:
            logger.error("driver status is not ok in " + fsp_ssh_info.node_ips)
            raise HCCIException(653057, fsp_ssh_info.node_ips)

    def get_server_ip(self):
        if 'datamover_externalom_iplist' in self.__config_dict:
            externalom_ip_lower = self.__config_dict['datamover_externalom_iplist'].lower()
            tmpip_list = externalom_ip_lower.split(',')
            if len(tmpip_list) == 1:
                server_ip = tmpip_list[0]
            else:
                server_ip = self.__config_dict['datamover_management_float_ip'].lower()
        else:
            server_ip = self.__config_dict['datamover_management_float_ip'].lower()
        return server_ip

    def get_account_info(self):
        # 去级联
        self.fsp_root_pwd = self.params.get_param_value(self.pod_id, "OpenStack", 'root_password',
                                                        'openstack_root_password')
        self.fsp_user_pwd = self.params.get_param_value(self.pod_id, "OpenStack", 'fsp_password',
                                                        'openstack_fsp_password')
        self.fsp_env_pwd = FSPasswordManager(self.pod_id).get_cps_admin_password()

        error_info = ''
        if not self.fsp_user_pwd:
            error_info += 'fsp_user_pwd'
        if not self.fsp_root_pwd:
            error_info += ', fsp_root_pwd'
        if not self.fsp_env_pwd:
            error_info += ', fsp_env_pwd'
        if error_info:
            raise HCCIException(653044, error_info)

    def rollback(self, project_id, pod_id):
        """
        标准调用接口：执行回滚
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        pass

    def retry(self, project_id, pod_id):
        """
        标准调用接口：重试
        :return: Message类对象
        """
        return self.execute(project_id, pod_id)

    def check(self, project_id, pod_id):
        """
        标准调用接口：重试
        :param project_id:
        :param pod_id:
        :return:
        """
        pass

# the code has been updated to python3.7
