# -*- coding: utf-8 -*-
from utils.business.project_condition_utils import get_project_condition_boolean
from utils.common import log as logger
from utils.common.exception import HCCIException
from utils.common.fic_base import StepBaseInterface
from utils.common.fic_base import TestCase
from utils.common.message import Message
from utils.common.ssh_util import Ssh

from plugins.ResourceCheck.common.libs import common_libs
from plugins.ResourceCheck.common.libs import network_libs
from plugins.ResourceCheck.common.libs.BMCCmd import BMCCmdExc
from plugins.ResourceCheck.common.libs.ibmc_cmd_libs import IBMCOperate


class DoEbackupCheck(TestCase):
    def __init__(self, project_id, pod_id):
        super(DoEbackupCheck, self).__init__(project_id, pod_id)
        self.ssh_obj = Ssh()
        self.cmd_exe = BMCCmdExc()
        self.bmc_info = None

    def pre_condition(self):
        pass

    def procedure(self):
        self.bmc_info = self.db.get_ebackup_install_info(self.pod_id)
        bmc_ip_list = []
        for sub_bmc_info in self.bmc_info:
            bmc_ip_lower = sub_bmc_info['bmc_ip'].lower()
            self.check_disk(bmc_ip_lower, sub_bmc_info['bmc_name'], sub_bmc_info['bmc_passwd'])
            bmc_ip_list.append(bmc_ip_lower)
        common_libs.get_ssh_ip(bmc_ip_list)

        for value in bmc_ip_list:
            if not common_libs.check_nic_info(self.pod_id, value, 4, 0):
                logger.error("check nic info failed.")
                raise HCCIException(103291)
            self.check_cpu(value)
            self.check_mem(value)
        pci_obj = network_libs.NicPciCls(self.bmc_info)
        for sub_bmc_info in self.bmc_info:
            bmc_ip_lower = sub_bmc_info['bmc_ip'].lower()
            management_check = ''
            management_plane_slot = sub_bmc_info['management_plane'].replace(' ', '').split(',')
            for slot in management_plane_slot:
                management_check, port_id = pci_obj.get_nic_pci_and_pid_by_slot(bmc_ip_lower, slot)
            if not management_check:
                logger.error(f"Nic slot map is wrong. Bmc ip is {bmc_ip_lower}, "
                             f"nic slot map is {sub_bmc_info['management_plane']}")
                raise HCCIException(104000, bmc_ip_lower, str(sub_bmc_info['management_plane']))

            if sub_bmc_info['production_storage_plane'] != '':
                business_check = ''
                business_plane_slot = sub_bmc_info['management_plane'].replace(' ', '').split(',')
                for slot in business_plane_slot:
                    business_check, port_id = pci_obj.get_nic_pci_and_pid_by_slot(bmc_ip_lower, slot)
                if not business_check:
                    logger.error(f"Nic slot map is wrong. Bmc ip is {bmc_ip_lower}, "
                                 f"nic slot map is {sub_bmc_info['production_storage_plane']}")
                    raise HCCIException(104000, bmc_ip_lower, str(sub_bmc_info['production_storage_plane']))

            if sub_bmc_info['backup_storage_plane'] != '':
                storage_check = ''
                storage_plane_slot = sub_bmc_info['management_plane'].replace(' ', '').split(',')
                for slot in storage_plane_slot:
                    storage_check, port_id = pci_obj.get_nic_pci_and_pid_by_slot(bmc_ip_lower, slot)

                if not storage_check:
                    logger.error(f"Nic slot map is wrong. Bmc ip is {bmc_ip_lower}, "
                                 f"nic slot map is sub_bmc_info['backup_storage_plane']")
                    raise HCCIException(104000, bmc_ip_lower, str(sub_bmc_info['backup_storage_plane']))

    def check_disk(self, ssh_ip, user_name, passwd):
        logger.info(f"bmc_ip:{ssh_ip} Disk check")
        if get_project_condition_boolean(self.project_id, 'eBackupServerARM'):
            if not IBMCOperate(ssh_ip, user_name, passwd).get_raid_controller_ids():
                cmd = "depmod > /dev/null 2>&1;modprobe hiraid > /dev/null 2>&1;" \
                      "fdisk -l | grep Disk | grep -wE 'GB|GiB|TiB|TB' | " \
                      "awk -F '/' '{print $3}' | awk -F ':' '{print $1}' | grep -v mapper | grep -vi nvme " \
                      "| grep -v loop | grep -v md | wc -l"
                result = self.cmd_exe.run([ssh_ip], cmd)
                disk_num = result.get(ssh_ip).get('stdout', '').strip()
                logger.info(f'disk IP:{ssh_ip}, num:{disk_num}')
                if disk_num != '2':
                    logger.error(f'Disk check failed, disk IP:{ssh_ip}, num:{disk_num}')
                    raise HCCIException('653068', ssh_ip)

    def check_cpu(self, bmc_ip):

        # 执行命令查询CPU内核大小
        if get_project_condition_boolean(self.project_id, 'eBackupServerARM'):
            cmd = "grep -o 'processor' /proc/cpuinfo | wc -l"
            size = self.cmd_exe.run([bmc_ip], cmd).get(bmc_ip, {}).get('stdout', '')
            size = int(size.strip())
        else:
            cmd = "awk '/cpu cores/{print $NF;exit}' /proc/cpuinfo"
            cpu_cores = self.cmd_exe.run([bmc_ip], cmd).get(bmc_ip, {}).get('stdout', '')
            cmd = "grep 'physical id' /proc/cpuinfo | sort -u | wc -l"
            cpu_slots = self.cmd_exe.run([bmc_ip], cmd).get(bmc_ip, {}).get('stdout', '')
            size = int(cpu_cores) * int(cpu_slots)
        logger.info(f"size type:{size}")
        if int(size) >= 8:
            logger.info("eBackup_CPU check is success!")
        else:
            logger.error("eBackup_CPU check is Failed!")
            raise HCCIException("103261", 8, size, bmc_ip)

    def check_mem(self, bmc_ip):
        # 执行命令查询CPU内核大小
        cmd = "awk '/MemTotal:/{print $2}' /proc/meminfo"
        memory_all = self.cmd_exe.run([bmc_ip], cmd).get(bmc_ip, {}).get('stdout', '')
        mem_size = int(memory_all) / 1024 / 1024
        logger.info(f"Mem_size is {mem_size}")
        if int(mem_size) >= 16:
            logger.info("eBackup_Mem_Size check is ok !")
        else:
            logger.error("eBackup_Mem_Size check is Failed! ")
            raise HCCIException("103271", 16, mem_size, bmc_ip)

    def post_condition(self):
        pass

    def failure(self):
        super(DoEbackupCheck, self).failure()


class EBackupCheck(StepBaseInterface):
    def __init__(self, project_id, pod_id):
        super(EBackupCheck, self).__init__(project_id, pod_id)
        self.project_id = project_id
        self.pod_id = pod_id
        self.implement = None

    def pre_check(self, project_id, pod_id):
        """
        插件内部接口：执行安装前的资源预检查，该接口由execute接口调用，工具框架不会直接调用此接口。
        :param project_id:
        :param pod_id:
        :return:
        """
        pass

    def main_handle(self, project_id, pod_id):
        self.implement = DoEbackupCheck(project_id, pod_id)
        self.implement.procedure()
        return Message(200)

    def execute(self, project_id, pod_id):
        try:
            return self.main_handle(project_id, pod_id)
        except HCCIException as err:
            return Message(500, err)
        except Exception as err:
            return Message(500, HCCIException('653012', err))

    def rollback(self, project_id, pod_id):
        """
        标准调用接口：执行回滚
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        return Message(200)

    def retry(self, project_id, pod_id):
        """
        标准调用接口：重试
        :return: Message类对象
        """
        return self.execute(project_id, pod_id)

    def check(self, project_id, pod_id):
        """
        标准调用接口：重试
        :param project_id:
        :param pod_id:
        :return:
        """
        pass

# the code has been updated to python3.7
