# -*- coding:utf-8 -*-
from utils.DBAdapter.DBConnector import BaseOps
from utils.business.param_util import ParamUtil
from utils.common import log as logger
from utils.common.exception import HCCIException
from utils.common.fic_base import StepBaseInterface
from utils.common.message import Message

from plugins.eBackup.common.model import DmkUserInfo
from plugins.eBackup.common.api_adapter import API


class CreateDmkAccount(StepBaseInterface):
    def __init__(self, project_id, pod_id):
        logger.init("eBackup")
        self.project_id = project_id
        self.pod_id = pod_id
        self.param_db = BaseOps()
        self.team_id = ""
        self.team_list = []
        self.__db_param_dict = {}
        self.dmk_login_user = ""
        self.root_user = ""
        self.root_password = ""
        self.fsp_password = ""
        self.fsp_root_password = ""
        self.hcp_user = ""
        self.hcp_password = ""

    def pre_check(self, project_id, pod_id):
        """
        插件内部接口：执行安装前的资源预检查，该接口由execute接口调用，工具框架不会直接调用此接口。
        :param project_id:
        :param pod_id:
        :return:
        """
        pass

    def execute(self, project_id, pod_id):
        """
        标准调用接口：执行安装前预检查&安装&配置
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        try:
            self.create_dmk_user()
            return Message(200)
        except HCCIException as err:
            return Message(500, err)
        except Exception as err:
            logger.error(f"Failed to create DMK account for eBackup plugin, the reason is {err}")
            return Message(500, err, err)

    def rollback(self, project_id, pod_id):
        """
        标准调用接口：执行回滚
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        return Message()

    def retry(self, project_id, pod_id):
        """
        标准调用接口：重试
        :return: Message类对象
        """
        return self.execute(project_id, pod_id)

    def check(self, project_id, pod_id):
        """
        标准调用接口：重试
        :param project_id:
        :param pod_id:
        :return:
        """
        pass

    def get_all_params(self):
        self.__db_param_dict = API.get_service_param_list(
            self.param_db, self.pod_id, 'eBackup')
        params = ParamUtil()

        self.fsp_root_password = params.get_param_value(
            self.pod_id, "OpenStack", 'root_password',
            'openstack_root_password')
        self.fsp_password = params.get_param_value(
            self.pod_id, "OpenStack", 'fsp_password', 'openstack_fsp_password')

        if not self.fsp_root_password:
            raise HCCIException(653044, "fsp_root_pwd")
        if not self.fsp_password:
            raise HCCIException(653044, "fsp_user_pwd")
        dmk_param_key_list = [
            'dmk_ui_username',
            'dmk_os_business_username',
            'dmk_os_business_user_password']
        dmk_dict = API.get_dmk_info(self.pod_id)
        for key in dmk_param_key_list:
            self.__db_param_dict[key] = dmk_dict[key]
        self.__db_param_dict['dmk_floatIp'] = params.get_param_value(
            self.pod_id, "DMK", 'dmk_floatIp', 'dmk_floatIp')
        self.__db_param_dict['dmk_ui_password'] = params.get_param_value(
            self.pod_id, "DMK", 'dmk_ui_password', 'dmk_ui_password')
        self.dmk_login_user = self.__db_param_dict.get('eBackup_dmk_user')
        self.root_user = self.__db_param_dict.get('eBackup_dmk_ssh_user')
        self.root_password = self.__db_param_dict.get('eBackup_os_root_password')
        self.hcp_user = self.__db_param_dict.get('hcp_ssh_user')
        self.hcp_password = self.__db_param_dict.get('hcp_ssh_password')

    def create_dmk_user(self):
        """
        创建DMK用户
        :return:
        """
        self.get_all_params()
        self.team_list = [self.__db_param_dict.get('eBackup_dmk_group_name'), "OTD"]
        self.team_id = self.create_dmk_login_user()

        root_dmkinfo = DmkUserInfo(
            self.dmk_login_user, self.root_user, self.team_list, [self.root_password, self.root_password])
        fsp_dmkinfo = DmkUserInfo(
            self.dmk_login_user, "fsp", self.team_list, [self.fsp_password, self.fsp_root_password])
        hcp_dmkinfo = DmkUserInfo(
            self.dmk_login_user, self.hcp_user, self.team_list, [self.hcp_password, self.root_password])

        self.create_ssh_account_to_dmk(root_dmkinfo)
        self.create_ssh_account_to_dmk(fsp_dmkinfo)
        self.create_ssh_account_to_dmk(hcp_dmkinfo)

    def create_dmk_login_user(self):
        API.login_dmk(
            self.__db_param_dict.get('dmk_floatIp'),
            self.__db_param_dict.get('dmk_ui_username'),
            self.__db_param_dict.get('dmk_ui_password'))
        team_id, user_id = API.createdmkuser(
            self.__db_param_dict.get('eBackup_dmk_user'),
            self.__db_param_dict.get('eBackup_dmk_tmp_password'), '2',
            self.__db_param_dict.get('eBackup_dmk_group_name'))
        if team_id and user_id:
            bret = API.update_user_to_team(user_id, '2', self.team_list)
            if not bret:
                logger.error("eBackup,update user to multi team failed.")
                raise HCCIException(653112)

        API.login_dmk(
            self.__db_param_dict.get('dmk_floatIp'),
            self.__db_param_dict.get('eBackup_dmk_user'),
            self.__db_param_dict.get('eBackup_dmk_tmp_password'))
        API.changedmkuserpasswd(
            user_id,
            self.__db_param_dict.get('eBackup_dmk_tmp_password'),
            self.__db_param_dict.get('eBackup_dmk_password'))

        API.login_dmk(
            self.__db_param_dict.get('dmk_floatIp'),
            self.__db_param_dict.get('eBackup_dmk_user'),
            self.__db_param_dict.get('eBackup_dmk_password'))

        return team_id

    def create_ssh_account_to_dmk(self, dmkinfo: DmkUserInfo):
        account_id = ""
        try:
            account_id = API.get_dmk_account_id(dmkinfo.user_name, dmkinfo.account_name)
        except Exception as no_such_account_err:
            logger.info(f"query account faild, the reason {no_such_account_err}")

        if account_id:
            logger.info(f"Query account({dmkinfo.account_name}) successfully,Now begin to delete it.")
            API.delete_dmk_account(account_id)
            logger.info(f"Delete account({dmkinfo.account_name}) success,it will be create again.")

        API.addaccounttodmk(dmkinfo.account_name, self.team_id, dmkinfo.passwd, dmkinfo.sudo_passwd)
        bret = API.update_account_to_team(dmkinfo)
        if not bret:
            logger.error("eBackup,update root account to multi team failed.")
            raise HCCIException(653113, dmkinfo.account_name)

# the code has been updated to python3.7
