# -*- coding:utf-8 -*-
import utils.common.log as logger
from utils.business.manageone_cmdb_util import ManageOneCmdbUtil
from utils.business.param_util import ParamUtil
from utils.business.project_condition_utils import get_project_condition_boolean
from utils.business.project_util import ProjectApi
from utils.common.exception import HCCIException
from utils.common.fic_base import StepBaseInterface
from utils.common.message import Message

from plugins.eBackup.common.api_adapter import API
from plugins.eBackup.scripts.common.ebackup_util import CommonTools

logger.init("eBackup RegisterCMDB")
ERROR_INFO_2 = "ebackup is not registerd on cmdb."


class RegisterCMDB(StepBaseInterface):
    def __init__(self, project_id, pod_id):
        self.pod_id = pod_id
        self.project_id = project_id
        self.params = ParamUtil()
        self.project_api = ProjectApi()
        self.cmdb_util = ManageOneCmdbUtil(project_id, pod_id)
        self.__config_dict = self.params.get_service_cloud_param(pod_id, "eBackup")
        self.datamover_om_ips = self.__config_dict.get('datamover_externalom_iplist')
        self.region_id = self.__config_dict['region_id']
        self.datamover_server_ip = self.__config_dict.get('datamover_server_ip')
        self.manage_float_ip = self.__config_dict.get('datamover_management_float_ip')
        self.kmc_alg_mode = CommonTools(project_id, pod_id).get_kmc_alg_mode_str()

    def execute(self, project_id, pod_id, *args, **kwargs):
        """
        标准调用接口：执行安装前预检查&安装&配置
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        try:
            if not self.exec_register_cmdb(project_id, pod_id, *args, **kwargs):
                raise HCCIException("653067", ERROR_INFO_2)
        except HCCIException as err:
            logger.error("Failed executing registerCMDB, ErrMsg: %s" % str(err))
            return Message(500, err)
        except Exception as err:
            logger.error("Failed executing registerCMDB, ErrMsg: %s " % err)
            return Message(500, HCCIException("653067", str(err)))
        logger.info('Succeed executing RegisterCMDB')
        return Message(200)

    def exec_register_cmdb(self, project_id, pod_id, *args, **kwargs):
        current_version, service_scale = RegisterCMDB(project_id, pod_id).get_version_scale(project_id)
        datamover_om_iplist = self.datamover_om_ips.split(',')
        # 判断信息是否已经添加
        node_info_list = self.cmdb_util.get_deploy_node_info_for_mo(self.region_id, "eBackup", current_version)
        if node_info_list and self.is_register_cmdb(node_info_list, datamover_om_iplist):
            return True

        is_driver, is_proxy, is_expand_az = RegisterCMDB(project_id, pod_id).get_ebackup_deploy_scene(project_id)
        if is_driver:
            datamover_name = self.get_ebackup_index_on_cmdb(node_info_list)
            self.set_az_list(datamover_name)
            node_index = 1
            deploy_node_infos = []
            self.register_ebackup_to_cmdb(datamover_om_iplist, datamover_name, current_version,
                                          node_index, deploy_node_infos)
        elif is_proxy:
            if not node_info_list:
                return False
            datamover_name = self.get_existed_ebackup_index_on_cmdb(node_info_list)
            if not datamover_name:
                return False
            sub_name, node_index, node_name_list = self.get_register_data(datamover_name, node_info_list)
            self.register_ebackup_to_cmdb(datamover_om_iplist, sub_name, current_version, node_index,
                                          node_name_list)
        elif is_expand_az:
            service_name = "eBackup_service1_" + self.region_id
            data_dicts = self.cmdb_util.get_cloud_service_info(self.region_id, "eBackup", name=service_name)
            az_str = self.get_az_list()
            for data_dict in data_dicts:
                extend_info_list = data_dict.get("extendInfos", [])
                az_list = ""
                for extend_info in extend_info_list:
                    if extend_info.get("key") == "az_list":
                        az_list = extend_info.get("value", "")
                        break
                if not az_list:
                    az_list = az_str
                elif az_str not in az_list:
                    az_list += "," + az_str
                else:
                    logger.info(f"az info {az_str} is already in {service_name} service.")
                    continue
                self.cmdb_util.update_cloud_service_extend_infos(
                    self.region_id, "eBackup",
                    [{"key": "az_list", "value": az_list}], False, name=service_name)
            logger.info(f"successful register az info {az_str} in eBackup_service1 service.")
        else:
            logger.info("no need to register CMDB information.")
        return True

    @staticmethod
    def is_register_cmdb(node_info_list, datamover_om_iplist):
        for node_info in node_info_list:
            ip_address_list = node_info['ipAddresses']
            for ip_address in ip_address_list:
                if ip_address['ip'] in datamover_om_iplist:
                    logger.info(
                        'this info has been registered, '
                        'no need to register again.')
                    return True
        return False

    def set_az_list(self, service_name):
        if service_name.startswith("eBackup_service1_"):
            logger.info("First eBackup cloud service, no need update az_list")
            return
        service_name = "eBackup_service1_" + self.region_id
        node_infos = self.cmdb_util.get_cloud_service_info(self.region_id, "eBackup", name=service_name)
        current_az = self.get_az_list()
        for node_info in node_infos:
            cmdb_az_list = ""
            extend_infos = node_info.get("extendInfos")
            update_extend_info = []
            for extend_info in extend_infos:
                if extend_info.get('key') != 'az_list':
                    update_extend_info.append(extend_info)
                else:
                    cmdb_az_list = extend_info.get('value')

            if current_az not in cmdb_az_list:
                cmdb_az_list += "," + current_az
            update_extend_info.append({"key": "az_list", "value": cmdb_az_list})
            node_info['extendInfos'] = update_extend_info
            logger.info(f"update {service_name} extendinfo az_list to {cmdb_az_list}")
            self.cmdb_util.set_cloud_service_info(self.region_id, node_info)


    def get_az_list(self):
        az_str = ''
        if get_project_condition_boolean(self.project_id, "eBackup&ExpansionAdCloudService"):
            az_str = self.__config_dict.get('az_related_to_datamover').strip()
        elif get_project_condition_boolean(self.project_id, "eBackup&ProjectDeploy"):
            az_str = self.params.get_param_value(self.pod_id, "OpenStack", "openstack_region", '').strip()
        elif get_project_condition_boolean(self.project_id, "ExpansionAZ&BCInstalled"):
            az_str = self.params.get_param_value(self.pod_id, "OpenStack", "openstack_expansion_az", '').strip()
        if not az_str:
            raise HCCIException("653131")
        return az_str

    def register_ebackup_to_cmdb(self, datamover_om_iplist, node_name_prefix,
                                 current_version, node_index,
                                 deploy_node_infos):
        for datamover_om_ip in datamover_om_iplist:
            node_name = node_name_prefix + "_node" + str(node_index)
            deploy_node_infos.append({
                "name": node_name,
                "mgmtIp": datamover_om_ip,
                "ipAddresses": [{"ip": datamover_om_ip}],
            })
            node_index = node_index + 1

        ebackup_service_info = {
            "indexName": "eBackup",
            "version": current_version,
            "name": node_name_prefix + '_' + self.region_id,
            "deployNodeInfos": deploy_node_infos,
            "extendInfos": [
                {
                    "key": "CipherType",
                    "value": self.kmc_alg_mode
                },
                {
                    "key": "az_list",
                    "value": self.get_az_list()
                }
            ]
        }
        logger.info('start to register ebackup to CMDB '
                    'info: {}'.format(ebackup_service_info))
        self.cmdb_util.set_cloud_service_info(
            self.region_id, ebackup_service_info)

    @staticmethod
    def get_register_data(datamover_name, node_info_list):
        datamover_name_list = datamover_name.split('_')
        sub_name = "%s_%s" % (datamover_name_list[0], datamover_name_list[1])
        node_name_list = []
        for node_info in node_info_list:
            if node_info['name'].find(sub_name) >= 0:
                node_name_list.append(node_info)
        node_index = len(node_name_list) + 1
        return sub_name, node_index, node_name_list

    @staticmethod
    def get_ebackup_index_on_cmdb(node_info_list):
        index = 1
        node_name_list = []
        if node_info_list:
            for node_info in node_info_list:
                name_part_list = node_info['name'].split('_')
                if len(name_part_list) == 3:
                    node_name_list.append(name_part_list[1])
        if node_name_list:
            new_node_name_list = list(set(node_name_list))
            index = len(new_node_name_list) + 1
        datamover_name = "eBackup_service" + str(index)
        return datamover_name

    @staticmethod
    def get_version_scale(project_id):
        path, name = API.find_file("OceanStor", "eBackup_DMK.zip")
        if name == "":
            raise HCCIException(653004, "OceanStor......eBackup_DMK.zip")
        current_version = API.get_package_version(name)
        service_scale = ParamUtil().get_project_scales_by_project_id(
            project_id).get("scale")
        return current_version, service_scale

    def get_existed_ebackup_index_on_cmdb(self, node_info_list):
        datamover_name = ''
        for node_info in node_info_list:
            node_info_addresses = node_info['ipAddresses']
            if node_info_addresses:
                ips = list(map(lambda x: x.get('ip'), node_info_addresses))
                if self.datamover_server_ip in ips:
                    datamover_name = node_info['name']
                if datamover_name:
                    break
        return datamover_name

    @staticmethod
    def get_ebackup_deploy_scene(project_id):
        is_driver = get_project_condition_boolean(project_id, 'eBackup_ServerProxy_Driver')
        is_proxy = get_project_condition_boolean(project_id, 'eBackup_Proxy')
        is_expand_az = get_project_condition_boolean(project_id, 'ExpansionAZ&BCInstalled')
        return is_driver, is_proxy, is_expand_az

    def pre_check(self, project_id, pod_id):
        """
        插件内部接口：执行安装前的资源预检查，该接口由execute接口调用，
        工具框架不会直接调用此接口。
        :param project_id:
        :param pod_id:
        :return:
        """
        return Message(200)

    def rollback(self, project_id, pod_id):
        """
        标准调用接口：执行回滚
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        return self.execute(project_id, pod_id)

    def retry(self, project_id, pod_id):
        """
        标准调用接口：重试
        :return: Message类对象
        """
        return self.execute(project_id, pod_id)

# the code has been updated to python3.7
