# -*- coding:utf-8 -*-
import os
import stat

import utils.common.log as logger
from utils.common.check_result import CheckResult
from utils.common.exception import HCCIException
from utils.common.fic_base import StepBaseInterface
from utils.common.message import Message
from utils.common.ssh_util import Ssh

from plugins.eBackup.common.params_check_tool import check_ssh_login
from plugins.eBackup.common.util import Utils
from plugins.eBackup.common.model import SshInfo
from plugins.eBackup.scripts.upgrade.util import get_ebackup_driver_nodes

eBackup_Driver_status_check_SUCCESS = [
    CheckResult(itemname_ch="校验eBackup Driver节点fsp用户登录密码",
                itemname_en="Check the fsp user password of the eBackup "
                            "Driver nodes", status="success"),
    CheckResult(itemname_ch="检查即将过期的fsp用户登录密码",
                itemname_en="Check the fsp user password which is about"
                            " to expire", status="success"),
    CheckResult(itemname_ch="校验eBackup Driver节点root用户登录密码",
                itemname_en="Check the root user password of the eBackup "
                            "Driver nodes", status="success")
]


class DriverPrecheck(StepBaseInterface):
    def __init__(self, project_id, pod_id, regionid_list=None):
        super(DriverPrecheck, self).__init__(project_id, pod_id, regionid_list)
        self.project_id = project_id
        self.pod_id = pod_id
        self.regionid_list = regionid_list
        self.region_id = regionid_list[0]
        self.params_dict = Utils.init_system_params(project_id,
                                                    regionid_list[0])
        self.__ssh = Ssh()
        self.check_result_flag = True

    def write_driver_node_to_file(self, driver_host_dict):
        def do_write_driver_node_to_file():
            logger.info("Begin to write driver nodes to file.")
            reverse_proxy_ip = self.params_dict.get("openstack_reverse_proxy_ip")
            file_postfix = '.txt'
            file_path_name = os.path.realpath(
                __file__ + '/../' + '%s' + '_' + '%s' + '%s') % \
                             (self.region_id, reverse_proxy_ip, file_postfix)
            flags = os.O_WRONLY | os.O_CREAT
            modes = stat.S_IWUSR | stat.S_IRUSR
            with os.fdopen(os.open(file_path_name, flags, modes), 'w') as file_pointer:
                file_pointer.truncate()
                # write cps node to file
                file_pointer.write(','.join(driver_host_dict.get("cpsnode")) + '\n')
                # write driver node to file
                file_pointer.write(','.join(driver_host_dict.get("eBackupDriver")) + '\n')
            logger.info("Wite driver nodes to [%s] successfully.", file_path_name)

        try:
            do_write_driver_node_to_file()
        except Exception as ex:
            logger.error(str(ex))
            raise ex

    def execute(self, project_id, pod_id, regionid_list=None):
        def do_execute():
            openstack_reverse_proxy_ip = self.params_dict.get("openstack_reverse_proxy_ip")
            logger.info(f"Begin to query eBackup driver:{openstack_reverse_proxy_ip}")
            all_driver_nodes = get_ebackup_driver_nodes(openstack_reverse_proxy_ip, self.params_dict)
            all_need_check_ips = all_driver_nodes.get("eBackupDriver")
            for cps_ip in all_driver_nodes.get("cpsnode"):
                if cps_ip not in all_need_check_ips:
                    all_need_check_ips.append(cps_ip)

            check_result = []
            ssh_login_result = self.get_ssh_login_result(all_need_check_ips)

            self.handle_expire_msg(ssh_login_result.get("password_goto_expire_msg"), check_result)
            self.handle_unpassed_fsp_nodes(ssh_login_result.get("unpassed_fsp_nodes"), check_result)
            self.handle_unpassed_root_nodes(ssh_login_result.get("unpassed_root_nodes"), check_result)

            if self.check_result_flag:
                self.write_driver_node_to_file(all_driver_nodes)
                logger.info("Check password successfully and write nodes to file successfully.")
            else:
                logger.error("Check password failed,it is no need to write nodes to file.")
            return Message(200, check_results=check_result)

        try:
            return do_execute()
        except HCCIException as err:
            logger.error("Precheck driver failed: " + str(err))
            return Message(500, err)
        except Exception as err:
            logger.error("Failed to precheck eBackup Driver,the reason is %s" % err)
            return Message(500, error_msg_cn=f"升级前检查出现异常，请查看日志并重试，异常信息:{str(err)}",
                           error_msg_en=f"Exception occurs while prechecking, please check the log and try again,"
                                        f" error info:{str(err)}.")

    def get_ssh_login_result(self, all_need_check_ips):
        ssh_login_result = {
            "unpassed_fsp_nodes": [],
            "unpassed_root_nodes": [],
            "password_goto_expire_msg": {}
        }
        for node_ip in all_need_check_ips:
            node_ssh_info = SshInfo(
                node_ip, "fsp", self.params_dict.get("openstack_fsp_pwd"), self.params_dict.get("openstack_root_pwd"))
            ret_code, msg = check_ssh_login(node_ssh_info, 3)
            if 1 == ret_code or 2 == ret_code:
                ssh_login_result.get("unpassed_fsp_nodes").append(node_ip)
            elif 3 == ret_code:
                ssh_login_result.get("password_goto_expire_msg").update(
                    {node_ip: "The fsp password will expire in %s days." % msg})
            elif 4 == ret_code:
                ssh_login_result.get("unpassed_root_nodes").append(node_ip)
        return ssh_login_result

    def handle_expire_msg(self, password_goto_expire_msg, check_result):
        if 0 != len(password_goto_expire_msg):
            unpassed_ips = list(password_goto_expire_msg.keys())
            result = CheckResult(
                itemname_ch="检查即将过期的fsp用户登录密码",
                itemname_en="Check the fsp user password "
                            "which is about to expire",
                status="failure",
                error_msg_cn=HCCIException(
                    650037, ','.join(unpassed_ips), 'fsp',
                    str(password_goto_expire_msg)))
            self.check_result_flag = False
        else:
            result = CheckResult(
                itemname_ch="检查即将过期的fsp用户登录密码",
                itemname_en="Check the fsp user "
                            "password which is about to expire",
                status="success")
        check_result.append(result)

    def handle_unpassed_fsp_nodes(self, unpassed_fsp_nodes, check_result):
        if 0 != len(unpassed_fsp_nodes):
            result = CheckResult(
                itemname_ch="校验eBackup Driver节点fsp用户登录密码",
                itemname_en="Check the fsp user password of the"
                            " eBackup Driver nodes",
                status="failure",
                error_msg_cn=HCCIException(
                    650005, ",".join(unpassed_fsp_nodes)))
            self.check_result_flag = False
        else:
            result = CheckResult(
                itemname_ch="校验eBackup Driver节点fsp用户登录密码",
                itemname_en="Check the fsp user password of "
                            "the eBackup Driver nodes",
                status="success")
        check_result.append(result)

    def handle_unpassed_root_nodes(self, unpassed_root_nodes, check_result):
        if 0 != len(unpassed_root_nodes):
            result = CheckResult(
                itemname_ch="校验eBackup Driver节点root用户登录密码",
                itemname_en="Check the root user password "
                            "of the eBackup Driver nodes",
                status="failure",
                error_msg_cn=HCCIException(650006,
                                           ','.join(unpassed_root_nodes)))
            self.check_result_flag = False
        else:
            result = CheckResult(
                itemname_ch="校验eBackup Driver节点root用户登录密码",
                itemname_en="Check the root user password"
                            " of the eBackup Driver nodes",
                status="success")
        check_result.append(result)

    def rollback(self, project_id, pod_id, regionid_list=None):
        return Message(200)

    def retry(self, project_id, pod_id, regionid_list=None):
        return self.execute(project_id, pod_id, regionid_list)
