# -*- coding:utf-8 -*-
import os

import utils.common.log as logger
import utils.common.software_package_util as FileUtil
from utils.business.manageone_util2 import ManageOneUtil2
from utils.business.param_util import ParamUtil
from utils.common.exception import HCCIException
from utils.common.fic_base import StepBaseInterface
from utils.common.message import Message

from plugins.eBackup.common.util import Utils

logger.init("eBackup")

PACKAGE_PATTERN = "^resource_10UnifiedAccess2MO_eBackup_(.*)\\.tar\\.gz$"


class RegisterAdaptationPackage(StepBaseInterface):
    def __init__(self, project_id, pod_id, regionid_list=None):
        self.project_id = project_id
        self.pod_id = pod_id
        self.regionid_list = regionid_list
        self.region_id = regionid_list[0]
        self.params = ParamUtil()
        super(RegisterAdaptationPackage, self).__init__(project_id, pod_id, regionid_list)

    def pre_check(self, project_id, pod_id):
        """
        插件内部接口：执行安装前的资源预检查，该接口由execute接口调用，工具框架不会直接调用此接口。
        :param project_id:
        :param pod_id:
        :return:
        """
        return Message(200)

    def execute(self, project_id, pod_id, regionid_list=None):
        """
        标准调用接口：执行安装前预检查&安装&配置
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        try:
            self.register_adaptation_package(project_id, pod_id)
        except Exception as ex:
            return Message(500, ex)
        logger.info("Register CloudNetDebug adaptation package to OC successfully")
        return Message(200)

    def register_adaptation_package(self, project_id, pod_id):
        logger.info("begin to register eBackup adaptation package")
        self._pre_store_origin_adapter_version()
        (path, name, version) = FileUtil.find_software_package_by_name_re(PACKAGE_PATTERN)

        logger.info(f"upload adaptation package path:{path}, name:{name}, version:{version}")
        file_path = os.path.join(path, name)
        package_infos = {
            "regionId": self.region_id,
            "file_path": file_path
        }

        result = ManageOneUtil2().upload_adaptation_package(project_id, pod_id, package_infos)
        if result.get("status") == "failed":
            message = result.get("msg")
            logger.error(f"upload adaptation package {name} failed : {message}")
            raise HCCIException(f"upload adaptation package {name} failed : {message}")

        logger.info("End to register CloudNetDebug adaptation package to OC successfully")

    def _pre_store_origin_adapter_version(self):
        adaptation_pkg_info_path = "".join(
            ['adaptation_pkg_info_',
             str(self.project_id), "_",
             self.region_id]
        )

        old_version = Utils.get_value_from_config_file(adaptation_pkg_info_path)
        if old_version:
            return

        result = ManageOneUtil2().query_adaptation_package_record(
            self.project_id, "eBackup", self.region_id)
        logger.info("[eBackup] Query adaptation pkg success: %s" % result[0])
        origin_version = result[0]["version"]
        logger.info(f"Adaptation_pkg_info_path is: {adaptation_pkg_info_path}")

        Utils.set_value_to_config_file(adaptation_pkg_info_path, origin_version)
        logger.info(f"Store origin adapter version to file successfully")

    def rollback(self, project_id, pod_id, regionid_list=None):
        """
        标准调用接口：执行回滚
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        return Message()

    def retry(self, project_id, pod_id, regionid_list=None):
        """
        标准调用接口：重试具体
        :return: Message类对象
        """
        return self.execute(project_id, pod_id, regionid_list)

    def check(self, project_id, pod_id):
        """
        标准调用接口：重试
        :param project_id:
        :param pod_id:
        :return:
        """
        return Message(200)
