# -*- coding:utf-8 -*-
import os
import threading

import utils.common.log as logger
from utils.business.param_util import CheckResult
from utils.common.compat import is_legacy_hcsu_scene
from utils.common.exception import HCCIException

from plugins.eReplication.common.api.file_api import API as FILE_API
from plugins.eReplication.common.client.mo_client import API as MO_API
from plugins.eReplication.common.client.ssh_client import API as SSH_API
from plugins.eReplication.common.constant import Component
from plugins.eReplication.common.constant import Euler
from plugins.eReplication.common.constant import Path
from plugins.eReplication.common.constant import VERSION_MAP
from plugins.eReplication.common.dr_api import API as DR_API
from plugins.eReplication.common.lib.conditions import Condition
from plugins.eReplication.common.lib.model import Auth
from plugins.eReplication.common.lib.params import Nodes
from plugins.eReplication.common.lib.thread import ExcThread


class API(object):

    def __init__(self, project_id, pod_id, region_id):
        self.project_id = project_id
        self.pod_id = pod_id
        self.region_id = region_id
        self.condition = Condition(self.project_id)
        if self.condition.is_sm_compatible:
            self.to_mode = "SMCompatible"
        elif self.condition.is_sm_only:
            self.to_mode = "SMOnly"
        else:
            self.to_mode = "generalCipher"
        config_path = os.path.join(
            os.path.dirname(os.path.dirname(__file__)), 'conf', 'env.ini')
        self.env = FILE_API(config_path)
        self.source_mode = self.env.get_value_by_key_and_sub_key(
            Component.REPLICATION, 'current_cipher_type')
        self.nodes = Nodes(self.project_id, self.pod_id)

    def cipher_change(self, rollback=False):

        change_funcs = list()
        thread_name = threading.current_thread().name
        hosts = self.nodes.all_hosts if is_legacy_hcsu_scene(
            self.project_id) else self.nodes.hosts
        for host in hosts:
            logger.info(f"Change encrypt algorithm on {host} in thread.")
            ssh_client = SSH_API.get_sudo_ssh_client(
                host, self.nodes.ssh_user, self.nodes.ssh_pwd,
                self.nodes.sudo_user,
                self.nodes.sudo_pwd)
            change_func = (self._judge_and_switch, thread_name,
                           (ssh_client, rollback), {})
            change_funcs.append(change_func)
        result = ExcThread.exec_func_in_thread_and_return(change_funcs)
        logger.info("Change encrypt algorithm process finish.")
        # 重启服务
        if True not in result:
            return False
        return True

    def _judge_and_switch(self, ssh_client, rollback):
        is_bcm_sm3, is_os_sm3 = self._check_task_result(ssh_client, rollback)
        if self.condition.is_general_cipher or (
                is_bcm_sm3 and is_os_sm3):
            logger.info(
                f"Current install mode is already {self.to_mode}, "
                f"no need execute change task.")
            return False
        # 停止服务
        self._stop_bcm_service(ssh_client)
        if not is_bcm_sm3:
            self._change_encrypt_algorithm(ssh_client, rollback)
        elif is_bcm_sm3 and not is_os_sm3:
            self._change_os_encrypt_algorithm(ssh_client, rollback)
        return True

    def _change_encrypt_algorithm(self, ssh_client, rollback):
        task_type = "changeCipher"
        cmd = \
            f"sh {Path.BCM_SCRIPTS_PATH}changeEncrypedType.sh " \
            f"{self.to_mode} {task_type};echo CMD_RESULT=$?"
        result = \
            SSH_API.exec_command_return_list(ssh_client, cmd, timeout=120)
        if "CMD_RESULT=0" not in str(result):
            raise Exception(
                f"Change encrypt algorithm task return {result}.")
        is_bcm_sm3, is_os_sm3 = self._check_task_result(ssh_client, rollback)
        if not is_bcm_sm3 or not is_os_sm3:
            raise Exception(
                "Check whether change encrypt "
                "algorithm task execute success.")

    def _change_os_encrypt_algorithm(self, ssh_client, rollback):
        cmd = \
            f"sh {Path.BCM_ROOT_SCRIPTS_PATH}changeOsCipher.sh " \
            f"{self.to_mode};echo CMD_RESULT=$?"
        result = \
            SSH_API.exec_command_return_list(ssh_client, cmd, timeout=120)
        if "CMD_RESULT=0" not in str(result):
            raise Exception(
                f"Change os encrypt algorithm task return {result}.")
        is_bcm_sm3, is_os_sm3 = self._check_task_result(ssh_client, rollback)
        if not is_os_sm3:
            raise Exception(
                "Check whether change os encrypt algorithm task "
                "execute success.")
        logger.info("Change os encrypt algorithm success.")

    def clear_environment(self):
        clear_funcs = list()
        thread_name = threading.current_thread().name
        for host in self.nodes.all_hosts:
            ssh_client = SSH_API.get_sudo_ssh_client(
                host, self.nodes.ssh_user, self.nodes.ssh_pwd,
                self.nodes.sudo_user,
                self.nodes.sudo_pwd)
            logger.info(f"Clear env on {host} in thread.")
            clear_func = (
                self._clear_environment, thread_name, (ssh_client,), {})
            clear_funcs.append(clear_func)
        ExcThread.exec_func_in_thread(clear_funcs)
        logger.info("Clear env success.")

    @staticmethod
    def _clear_environment(ssh_client):
        cmd = f"rm -fr {Path.CIPHER_BACKUP_PATH};echo CMD_RESULT=$?"
        result = \
            SSH_API.exec_command_return_list(ssh_client, cmd, timeout=20)
        if "CMD_RESULT=0" not in str(result):
            raise Exception(
                f"Clear environment task return {result}.")
        logger.info("Clear environment success.")

    def _check_task_result(self, ssh_client, rollback):
        os_sm3, bcm_sm3 = False, False
        bcm_check_cmd = f"cat {Path.LEGO_FILE_PATH} | grep 'cipher.type' | " \
                        f"awk -F'=' '{{print $2}}'"
        os_check_cmd = f"cat {Path.SYS_OS_FILE} | grep 'crypt_style' | " \
                       f"awk -F'=' '{{print $2}}'"
        bcm_result = SSH_API.exec_command_return_list(
            ssh_client, bcm_check_cmd)
        if self.to_mode in str(bcm_result):
            bcm_sm3 = True if not rollback else False
        os_result = SSH_API.exec_command_return_list(ssh_client, os_check_cmd)
        if self.to_mode == "generalCipher":
            if rollback:
                os_sm3 = True if "sm3" in str(os_result) else False
            else:
                os_sm3 = False if "sm3" in str(os_result) else True
        else:
            if self.source_mode == "generalCipher" and rollback:
                os_sm3 = False if "sm3" in str(os_result) else True
            else:
                os_sm3 = True if "sm3" in str(os_result) else False
        logger.info(f"Check task result return bcm_sm3: {bcm_sm3}, "
                    f"os_sm3: {os_sm3}.")
        return bcm_sm3, os_sm3

    @staticmethod
    def _stop_bcm_service(ssh_client):
        stop_cmd = \
            f"echo y|sh {Path.BCM_SCRIPTS_PATH}shutdownSystem.sh"
        stop_result = SSH_API.exec_command_return_list(ssh_client, stop_cmd)
        if "successfully" not in str(stop_result):
            logger.error(
                f"Stop bcm service failed, cmd execute return {stop_result}")
            raise Exception("Stop bcm service failed.")
        logger.info("Stop bcm service success on host.")

    def check_environment(self):
        check_result = list()
        for host in self.nodes.all_hosts:
            ssh_client = SSH_API.get_sudo_ssh_client(
                host, self.nodes.ssh_user, self.nodes.ssh_pwd,
                self.nodes.sudo_user,
                self.nodes.sudo_pwd)
            logger.info(f"Check environment process execute on {host}.")
            check_result = self._check_environment(
                ssh_client, host, check_result)
            auth_provider = Auth(
                host, self.nodes.ssh_user, self.nodes.ssh_pwd,
                self.nodes.sudo_user, self.nodes.sudo_pwd).auth_provider
            check_result = self._check_service_version(
                auth_provider, check_result)
        return check_result

    @staticmethod
    def _check_environment(ssh_client, host, result_check_list):
        disk = "/opt/"
        capacity = 4
        backup_path = Path.CIPHER_BACKUP_PATH
        result_check_list = DR_API.check_whether_backup_file_exists(
            ssh_client, host, backup_path, result_check_list)
        result_check_list = DR_API.check_whether_disk_free_space_is_enough(
            ssh_client, host, disk, capacity, result_check_list)
        result_check_list = DR_API.check_whether_disk_can_be_writen(
            ssh_client, host, disk, result_check_list)
        return result_check_list

    @staticmethod
    def _check_service_version(auth_provider, result_check_list):
        host = auth_provider.host
        service_version = DR_API(
            host, auth_provider.ssh_user, auth_provider.ssh_pwd,
            auth_provider.sudo_user, auth_provider.sudo_pwd). \
            get_current_version()
        if service_version < VERSION_MAP.get("8.1.0"):
            exception = HCCIException('675037', host)
            result_check_list.append(
                CheckResult(
                    itemname_ch=f"检查服务版本[{host}]",
                    itemname_en=f"Check Service Version[{host}]",
                    status="failure", error_msg_cn=exception))
        else:
            result_check_list.append(
                CheckResult(
                    itemname_ch="检查服务版本",
                    itemname_en="Check Service Version", status="success"))
        return result_check_list

    def check_cipher_type(self):
        current_cipher_type = self.get_current_cipher_type()
        if current_cipher_type == self.to_mode:
            logger.error(
                f"The current cipher type {current_cipher_type} is "
                "the same as the target cipher type.")
            raise Exception(
                f"The current cipher type {current_cipher_type} is "
                "the same as the target cipher type.")
        self.env.set_value_by_key_and_sub_key(
            Component.REPLICATION, 'current_cipher_type', current_cipher_type)
        self.env.set_value_by_key_and_sub_key(
            Component.REPLICATION, 'target_cipher_type', self.to_mode)
        logger.info("Check cipher type success.")

    def check_cpu_arch(self):
        check_result = list()
        for host in self.nodes.all_hosts:
            ssh_client = SSH_API.get_sudo_ssh_client(
                host, self.nodes.ssh_user, self.nodes.ssh_pwd,
                self.nodes.sudo_user,
                self.nodes.sudo_pwd)
            logger.info(f"Check cpu arch on {host}.")
            version_result = self._check_os_version(ssh_client, host)
            if not version_result:
                exception = HCCIException('675037', host)
                check_result.append(
                    CheckResult(
                        itemname_ch=f"检查OS版本[{host}]",
                        itemname_en=f"Check OS Version[{host}]",
                        status="failure", error_msg_cn=exception))
        if not check_result:
            check_result.append(
                CheckResult(
                    itemname_ch="检查CPU和OS", itemname_en="Check CPU And OS",
                    status="success"))
        return check_result

    @staticmethod
    def _is_x86_arch(ssh_client, host):
        result = SSH_API.exec_command_return_list(
            ssh_client, "echo arch=`arch`")
        logger.info(f"Get cpu arch on {host} return {result}.")
        if "x86_64" in str(result):
            return True
        return False

    def _check_os_version(self, ssh_client, host):
        cmd = \
            r"uname -r | grep -oP 'eulerosv\K(\d+)r(\d+)' | " \
            r"awk -F'r' '{print $1,$2}'"
        result = SSH_API.exec_command_return_list(ssh_client, cmd)
        if not result:
            raise Exception("Get Euler OS version failed, please check.")
        euler_version = int(result[0].split()[0])
        release_version = int(result[0].split()[1])
        logger.info(f"Get cpu version on {host} return "
                    f"{euler_version}.{release_version}.")
        if euler_version > Euler.LIMIT_VERSION:
            return True
        is_x86_arch = self._is_x86_arch(ssh_client, host)
        # x86操作系统需要系统版本号大于等于2.9
        x86_limit_ok = is_x86_arch and release_version >= Euler.LIMIT_X86_RELEASE
        # arm操作系统需要系统版本号大于等于2.8
        arm_limit_ok = (not is_x86_arch) and release_version >= Euler.LIMIT_ARM_RELEASE
        if euler_version == Euler.LIMIT_VERSION and (x86_limit_ok or arm_limit_ok):
            return True
        return False

    def get_current_cipher_type(self):
        logger.info("Start to get current cipher type.")
        cmd = f"cat {Path.LEGO_FILE_PATH} | grep 'cipher.type' | " \
              f"awk -F'=' '{{print $2}}'"
        for host in self.nodes.all_hosts:
            ssh_client = SSH_API.get_sudo_ssh_client(
                host, self.nodes.ssh_user, self.nodes.ssh_pwd,
                self.nodes.sudo_user,
                self.nodes.sudo_pwd)
            result = SSH_API.exec_command_return_list(ssh_client, cmd)
            if result:
                logger.info(f"Get current cipher type return {result}.")
                return result[0]
        logger.error("Get current cipher type failed.")
        raise Exception("Get current cipher type failed.")

    def refresh_cmdb_extend_info(self, rollback=False):
        target_mode = self.source_mode if rollback else self.to_mode
        mo_api = MO_API(self.project_id, self.pod_id)
        region_info = mo_api.get_region_info()
        for region in region_info:
            region_id = region.get("regionCode")
            cloud_service_info = mo_api.cmdb_ins.get_cloud_service_info(region_id, Component.REPLICATION)
            if not cloud_service_info:
                logger.info(
                    f"The DR service is not installed on {region_id}, "
                    f"no need to update CMDB.")
                continue
            mo_api.update_cloud_service_extend_info(
                region_id, [{"key": "CipherType", "value": target_mode}])
        logger.info(
            f"Refresh cmdb extend info cipher type to {target_mode} success.")
