# -*- coding:utf-8 -*-
import os
import threading

import utils.common.log as logger
from utils.common.exception import HCCIException

from plugins.eReplication.common.api.pkg_api import API as PKG_API
from plugins.eReplication.common.api.file_api import API as FILE_API
from plugins.eReplication.common.client.ssh_client import API as SSH_API
from plugins.eReplication.common.constant import Path, Component, PatchOperation
from plugins.eReplication.common.constant import Pkg
from plugins.eReplication.common.dr_api import API as DR_API
from plugins.eReplication.common.lib.conditions import Condition
from plugins.eReplication.common.lib.params import Nodes
from plugins.eReplication.common.lib.thread import ExcThread
from plugins.eReplication.common.lib.utils import check_param_ip


class API(object):
    def __init__(self, host, ssh_user, ssh_pwd, sudo_user, sudo_pwd):
        check_status = check_param_ip(host)
        if not check_status:
            logger.error(f"{host} is not a valid IP address.")
            raise ValueError(f"{host} is not a valid IP address.")
        self.host = host
        self.ssh_user = ssh_user
        self.sudo_user = sudo_user
        self.ssh_pwd = ssh_pwd
        self.sudo_pwd = sudo_pwd
        self.sudo_client = SSH_API.get_sudo_ssh_client(
            self.host, self.ssh_user, self.ssh_pwd, self.sudo_user, self.sudo_pwd)
        self.software_re = Pkg.PATCH_PKG_RE
        self.pkg_path, self.pkg_name, self.pkg_version = PKG_API.find_file_by_re(self.software_re)
        self.config_path = os.path.join(
            os.path.dirname(os.path.dirname(__file__)), 'conf', 'env.ini')
        self.config_tool = FILE_API(self.config_path)

    def install_patch(self):
        """
        安装Server补丁包
        :return:
        """
        if self._check_installed():
            logger.info(f"Node {self.host} has been patched.")
            return True
        software_flag = "eReplication_for_HCS8.2.0_Euler"
        pkg_count_return = SSH_API.exec_command_return_list(
            self.sudo_client, f"ls {Path.TMP_PATCH_PATCH} | grep {software_flag} | wc -l")
        logger.info(f"Count the number of patch return {pkg_count_return}.")
        if not pkg_count_return:
            logger.error("Count the number of patch failed.")
            raise HCCIException("663601", self.software_re)
        pkg_count = int(pkg_count_return[0])
        if pkg_count == 0:
            logger.error("Count the number of patch failed.")
            raise HCCIException("663601", self.software_re)
        elif pkg_count > 1:
            logger.error("The number of patch package is greater than 1.")
            raise HCCIException("663602", Path.TMP_PATCH_PATCH)
        else:
            cmd = \
                f'file_name=`ls {Path.TMP_PATCH_PATCH} | grep {software_flag}` && unzip -o -d ' \
                f'{Path.TMP_PATCH_PATCH} {Path.TMP_PATCH_PATCH}/"${{file_name}}";echo CMD_RESULT=$?'
            result = SSH_API.exec_command_return_list(self.sudo_client, cmd)
            if "CMD_RESULT=0" not in str(result):
                logger.error(f"Unzip patch return {result}.")
                raise Exception("Unzip patch file failed.")
            # expected output '(y/n)': 表示询问是否停止容灾服务
            SSH_API.send_command(
                self.sudo_client, f"sh {Path.TMP_PATCH_PATCH}/cliPatch.sh", "(y/n)")
            # expected output '(y/n)': 表示询问容灾服务管理数据是否已备份
            SSH_API.send_command(self.sudo_client, "y", "(y/n)")
            # expected output '(y/n)': 表示询问是否需要安装补丁包
            SSH_API.send_command(self.sudo_client, "y", "(y/n)")
            # expected output '(y/n)': 表示询问是否需要立即启动服务
            SSH_API.send_command(self.sudo_client, "y", "(y/n)", 400)
            SSH_API.send_command(self.sudo_client, "n")
            SSH_API.send_command(self.sudo_client, f"rm -fr {Path.TMP_PATCH_PATCH}")
            logger.info(f"Install patch on {self.host} success.")
            return True

    def upload_patch(self):
        """
        上传补丁包
        :return:
        """
        if self._check_installed():
            logger.info(f"Node {self.host} has been patched.")
            return True
        logger.info(f"Start upload patch to {self.host}.")
        if self.pkg_path and self.pkg_name:
            SSH_API.send_command(
                self.sudo_client,
                f"rm -rf {Path.TMP_PATCH_PATCH} && mkdir -p {Path.TMP_PATCH_PATCH} && "
                f"chown -R {self.ssh_user}:LEGO {Path.TMP_PATCH_PATCH}")
            file_path = os.path.join(self.pkg_path, self.pkg_name)
            PKG_API.check_compressed_file(file_path, size_limit=100, file_num_limit=200)
            SSH_API.put_file(
                self.host, self.ssh_user, self.ssh_pwd, file_path, Path.TMP_PATCH_PATCH, 22)
        else:
            logger.error("Server patch software not found.")
            raise HCCIException("663601", self.software_re)
        logger.info(f"Upload patch to {self.host} successfully.")
        return True

    def _check_installed(self):
        """
        检查环境是否已安装了Server补丁
        :return:
        """

        patch_version = DR_API(
            self.host, self.ssh_user, self.ssh_pwd, self.sudo_user, self.sudo_pwd).get_patch_version()
        return True if patch_version == self.pkg_version else False

    def rollback_patch(self):
        if PKG_API.get_pkg_version() != self.config_tool.get_value_by_key_and_sub_key(
                Component.REPLICATION, "current_version"):
            logger.info("No need execute.")
            return True
        if not self._check_installed():
            logger.info(f"Node {self.host} patch has been uninstalled.")
            return True
        SSH_API.send_command(
            self.sudo_client, f"cd {Path.BCM_SCRIPTS_PATH} && sh rollBackPatch.sh", "(y/n)")
        SSH_API.send_command(self.sudo_client, "y", "succeeded", 400)
        logger.info(f"Node {self.host} patch has been uninstalled.")
        return True

    @classmethod
    def execute_install(cls, project_id, pod_id, is_migrate=False):
        """
        执行补丁安装
        :param project_id:
        :param pod_id:
        :param is_migrate: 是否是数据迁移工程
        :return:
        """
        logger.info("Do install patch process.")
        condition = Condition(project_id)
        # 备Region会从主Region同步数据，因此无需执行启动和停止服务
        if condition.is_primary and not is_migrate:
            DR_API.execute_bcm_service(project_id, pod_id, True)
        # 启动后先执行上传补丁包操作，预留充足的时间做数据同步，防止备节点数据库表还未初始化就停止服务
        cls._do_upload_patch(project_id, pod_id, is_migrate)
        if condition.is_primary and not is_migrate:
            DR_API.execute_bcm_service(project_id, pod_id, False)
        cls._do_install_patch(project_id, pod_id, is_migrate)
        logger.info("Install patch success.")

    @classmethod
    def execute_rollback(cls, project_id, pod_id, is_migrate=False):
        """
        执行补丁回退
        :param project_id:
        :param pod_id:
        :param is_migrate: 是否是数据迁移工程
        :return:
        """
        DR_API.execute_bcm_service(project_id, pod_id, False)
        cls._do_rollback_patch(project_id, pod_id, is_migrate)
        logger.info("Uninstall patch success.")

    @classmethod
    def _do_upload_patch(cls, project_id, pod_id, is_migrate=False):
        cls._do_patch_operation(project_id, pod_id, PatchOperation.UPLOAD, is_migrate)

    @classmethod
    def _do_install_patch(cls, project_id, pod_id, is_migrate=False):
        cls._do_patch_operation(project_id, pod_id, PatchOperation.INSTALL, is_migrate)

    @classmethod
    def _do_rollback_patch(cls, project_id, pod_id, is_migrate=False):
        cls._do_patch_operation(project_id, pod_id, PatchOperation.ROLLBACK, is_migrate)

    @classmethod
    def _do_patch_operation(cls, project_id, pod_id, operation_type, is_migrate=False):
        funcs = list()
        nodes = Nodes(project_id, pod_id)
        all_hosts = nodes.target_all_hosts if is_migrate else nodes.all_hosts
        ssh_pwd = nodes.target_ssh_pwd if is_migrate else nodes.ssh_pwd
        sudo_pwd = nodes.target_sudo_pwd if is_migrate else nodes.sudo_pwd
        thread_name = threading.current_thread().name
        for host in all_hosts:
            logger.info(f"Do {operation_type} task on {host} in thread.")
            func = None
            if PatchOperation.UPLOAD == operation_type:
                func = (API(host, nodes.ssh_user, ssh_pwd, nodes.sudo_user, sudo_pwd).upload_patch,
                        thread_name, (), {})
            if PatchOperation.INSTALL == operation_type:
                func = (API(host, nodes.ssh_user, ssh_pwd, nodes.sudo_user, sudo_pwd).install_patch,
                        thread_name, (), {})
            if PatchOperation.ROLLBACK == operation_type:
                func = (API(host, nodes.ssh_user, ssh_pwd, nodes.sudo_user, sudo_pwd).rollback_patch,
                        thread_name, (), {})
            if not func:
                logger.error(f"Unsupported patch operation {operation_type}.")
                raise Exception(f"Unsupported patch operation {operation_type}.")
            funcs.append(func)
        ExcThread.exec_func_in_thread(funcs)

    def __del__(self):
        if self.sudo_client:
            SSH_API.close_ssh(self.sudo_client)
        logger.info(f"SSH channel is closed: {self.host}.")
