# -*- coding:utf-8 -*-

from collections import defaultdict

import utils.common.log as logger
from utils.Logic.FlavorOperate import FlavorOperate
from utils.Logic.ServerOperate import ServerOperate
from utils.business.vm_util import PublicServiceOM
from utils.common.fic_base import ToolClient
from utils.common.openstack_host import get_openstack_info_by_pod_id

from plugins.eReplication.common.client.mo_client import API as MO_API
from plugins.eReplication.common.constant import Component
from plugins.eReplication.common.constant import VMScale
from plugins.eReplication.common.lib.conditions import Condition
from plugins.eReplication.common.lib.params import Params
from plugins.eReplication.common.lib.utils import get_install_services


class API(object):

    @classmethod
    def query_cpu_mem(cls, pod_id, vm_name):
        """
        根据虚拟机名称查询虚拟机信息
        :param pod_id:工程pod_id
        :param vm_name:虚拟机名称
        :return: list [{'id': 虚拟机ID, 'vcpus': 虚拟机CPU数, 'ram': 虚拟机内存}]
        """
        vm_info_list = list()
        auth_provider = get_openstack_info_by_pod_id(pod_id)
        host_and_tool = (auth_provider, ToolClient.OPENSTACK_COMPUTE)
        server_operate_ins = ServerOperate()
        vm_list = server_operate_ins.get_servers_by_name(
            server_name=vm_name, host_and_tool=host_and_tool)
        if not vm_list:
            logger.error("Get VM info list failed.")
            raise Exception("Get VM info list failed.")
        for vm in vm_list['servers']:
            if vm['name'] == vm_name or vm['name'].startswith(vm_name):
                vm_info = defaultdict()
                vm_info['id'] = vm.get("id")
                vms_detail_info = server_operate_ins.get_server_details(
                    vm_info['id'], host_and_tool).get("server")
                flavor_id = vms_detail_info.get('flavor')['id']
                flavor_operate_ins = FlavorOperate()
                flavor_details = flavor_operate_ins.get_flavor_details(
                    flavor_id, host_and_tool)
                if flavor_details['result']:
                    body = flavor_details.get('body')
                    vm_cpus = body['flavor'].get('vcpus')
                    vm_ram = body['flavor'].get('ram')
                else:
                    logger.error(f"Get {vm['name']} flavor info failed.")
                    raise Exception(f"Get {vm['name']} flavor info failed.")
                vm_info['vcpus'] = vm_cpus
                vm_info['ram'] = vm_ram
                vm_info_list.append(vm_info)
        if not vm_info_list:
            logger.info("VMs detail info not found.")
            raise Exception("VMs detail info not found.")
        return vm_info_list

    @classmethod
    def expand_vm_scale(
            cls, pod_id, vm_id, server_name, vm_group_name,
            vm_cpu, vm_mem, disk=0, sys_disk=0):
        logger.info(f"Expand scale for vm: {vm_id}.")
        PublicServiceOM().resize_vm(
            pod_id, vm_id, server_name, vm_group_name, vm_cpu, vm_mem,
            disk, sys_disk)

    @classmethod
    def check_whether_need_expand_vm_scale(cls, project_id):
        is_need_expand_scale = False
        condition = Condition(project_id)
        is_supported_block = condition.is_supported_block
        is_supported_nas = condition.is_supported_nas
        is_dr_service_installed = condition.is_current_dr_installed
        install_services = get_install_services(project_id)
        if is_supported_nas:
            is_need_expand_scale = True
        if is_dr_service_installed and is_supported_block and \
                Component.CSDR_U in install_services:
            is_need_expand_scale = True
        return is_need_expand_scale

    @classmethod
    def get_deploy_vm_scale(cls, project_id, pod_id, scale):
        condition = Condition(project_id)
        is_global_con_dr = condition.is_global_con_dr
        is_sub_region = condition.is_sub
        is_dr_service_installed = condition.is_current_dr_installed
        is_block_dr = condition.is_block_dr
        is_nas_dr = condition.is_nas_dr
        is_supported_nas = condition.is_supported_nas
        is_supported_block = condition.is_supported_block
        primary_vcpus, primary_ram = VMScale.CPU_4, VMScale.RAM_8
        if is_sub_region and is_global_con_dr:
            primary_vcpus, primary_ram = \
                MO_API(project_id, pod_id).get_primary_region_vms_scale()
        params = Params(project_id, pod_id)
        vcpus = params.cpu_num
        ram = params.mem_num
        install_services = get_install_services(project_id)
        should_use_4u8g_flag = (is_supported_nas or (
                                        is_supported_block and is_nas_dr)
                                ) and scale <= 10000
        should_use_8u16g_flag = (len(install_services) > 1 and is_nas_dr
                                 ) or (len(install_services) == 1
                                       and is_block_dr and is_nas_dr)
        if is_dr_service_installed:
            if should_use_4u8g_flag:
                return VMScale.CPU_4, VMScale.RAM_8, VMScale.DISK_0
            return VMScale.CPU_0, VMScale.RAM_0, VMScale.DISK_0
        if should_use_8u16g_flag:
            return VMScale.CPU_8, VMScale.RAM_16, VMScale.DISK_40
        return max(int(vcpus), primary_vcpus), max(
            int(ram), primary_ram), VMScale.DISK_40
