#!/bin/sh

function check_result() {
  if [ $1 -ne 0 ]; then
    echo "$2"
    exit 1
  fi
}

function disable_iptables() {
  echo "[$(date '+%Y-%m-%d %H:%M:%S')] start disable iptables."
  systemctl stop firewalld
  systemctl disable firewalld.service
  systemctl enable haveged.service
  systemctl start haveged
  echo "[$(date '+%Y-%m-%d %H:%M:%S')] disable iptables success."
}

function create_swap() {
  echo "[$(date '+%Y-%m-%d %H:%M:%S')] start create swap."
  if [ -f "/tmp/init_ereplication_os" ]; then
    echo "[$(date '+%Y-%m-%d %H:%M:%S')] create swap success."
    return 0
  fi

  if [ -d "/opt/BCManager/Runtime" ]; then
    echo "[$(date '+%Y-%m-%d %H:%M:%S')] Runtime path exists. no need. return"
    return 0
  fi

  local disks=$(fdisk -l | grep -E -o "^Disk /dev/(s|v|h|xv)d[a-z]" | awk '{print $2}')
  for disk in ${disks[@]}; do
    total_sectors=$(fdisk -l $disk | grep -E "^Disk /dev/(s|v|h|xv)d[a-z]" | cut -d ',' -f 3 | cut -d ' ' -f 2)
    used_sectors=$(fdisk -l $disk | tail -n -1 | awk '{print $3}')
    if [ "$((total_sectors - used_sectors))" -gt "1" ]; then
      echo -e "n\np\n\n\n\nt\n\n8e\nw\nq\n" | fdisk $disk
      partprobe
      check_result $? "partprobe $disk failed."

      part_name=$(fdisk -l $disk | tail -n -1 | awk '{print $1}')
      echo -e "y\n" | pvcreate $part_name
      check_result $? "pvcreate $part_name failed."

      v_groups=$(vgs | grep -v "#PV")
      if [ -z "$v_groups" ]; then
        vg_name=BCVG
        vgcreate $vg_name $part_name
        check_result $? "Create vg $vg_name with $part_name failed."
      else
        vg_name=$(vgs | grep -v "#PV" | tail -n -1 | awk '{print $1}')
        vgextend $vg_name $part_name
        check_result $? "Extend vg $vg_name with $part_name failed."
      fi
    else
      part_name=$(fdisk -l $disk | tail -n -1 | awk '{print $1}')
      vg_name=$(vgs | grep -v "#PV" | tail -n -1 | awk '{print $1}')
      vgextend $vg_name $part_name
    fi
  done

  local total_swap=$(free -m | grep -i "swap" | awk '{print $2}')
  if [ -z "$total_swap" ] || [ "$total_swap" == "0" ]; then
    lvName=swap
    echo "[$(date '+%Y-%m-%d %H:%M:%S')] start create swap space, lvName:$lvName, vgName:$vg_name"
    lvcreate -L 8G -n $lvName $vg_name
    mkswap -c /dev/$vg_name/$lvName
    swapon /dev/mapper/${vg_name}-${lvName}
    check_result $? "swapon /dev/mapper/${vg_name}-${lvName} failed."
    echo "/dev/mapper/${vg_name}-${lvName}  swap                  swap    defaults        0 0" >>/etc/fstab
  fi

  opt_dir=$(df -h | awk '{print $6}' | grep "/opt$")
  if [ -z "$opt_dir" ]; then
    lv_name=opt
    echo "[$(date '+%Y-%m-%d %H:%M:%S')] start create /opt space, lv_name:$lv_name, vgName:$vg_name"
    lvcreate -L 10G -n $lv_name $vg_name
    mkfs -t ext4 /dev/${vg_name}/${lv_name}
    rm -fr /opt &>/dev/null
    mkdir -p /opt && chmod 755 /opt
    mount /dev/${vg_name}/${lv_name} /opt
    mount_dir=$(df -h | grep "/opt$" | awk '{print $1}')
    echo "$mount_dir /opt                    ext4   defaults,nodev       1 2" >>/etc/fstab
  fi

  root_lv=$(df -h | grep "/$" | awk '{print $1}')
  lvextend -l 100%FREE $root_lv
  resize2fs $root_lv

  echo "[$(date '+%Y-%m-%d %H:%M:%S')] create swap success."
  touch /tmp/init_ereplication_os
}

function createUser() {
  echo "[$(date '+%Y-%m-%d %H:%M:%S')] Start create user."
  echo "Please enter user password"
  read -s userPwd
  local result=$(echo "$userPwd" | sed 's/ //g')
  if [ -z "$result" ]; then
    echo "[$(date '+%Y-%m-%d %H:%M:%S')] The password is empty."
    exit 1
  fi
  cat /etc/passwd | grep -w "$userName" &>/dev/null
  if [ $? -eq 0 ]; then
    # do not change this echo message.
    checkUser ${userName}
    if [ $? -eq 0 ]; then
      echo "[$(date '+%Y-%m-%d %H:%M:%S')] Create user successfully, the user already exists, no need create, user is $userName"
      exit 0
    fi
  fi

  useradd -m -s /bin/bash -g wheel ${userName}
  cat /etc/passwd | grep -w "$userName" &>/dev/null
  if [ $? != 0 ]; then
    echo "[$(date '+%Y-%m-%d %H:%M:%S')] Create user failed[$userName]"
    exit 1
  fi

  passwd ${userName} <<EOF
${userPwd}
${userPwd}
EOF
  checkUser ${userName}
  if [ $? != 0 ]; then
    exit 1
  fi
  # do not change this echo message.
  echo "[$(date '+%Y-%m-%d %H:%M:%S')] Create user successfully."
  exit 0
}

function checkUser() {
  echo "Check if the user created successfully."
  cat /etc/shadow | grep $1 | grep -oP '\$.*?\$.*?\$.*?' &>/dev/null
  if [ $? != 0 ]; then
    echo "[$(date '+%Y-%m-%d %H:%M:%S')] Modify user password failed, user name: $userName, need recreate."
    userdel -r $1
    return 1
  fi
  return 0
}

function configTimeZone() {
  echo "[$(date '+%Y-%m-%d %H:%M:%S')] start config timezone."
  local dest_file=/usr/share/zoneinfo/$time_zone
  if [ ! -f "$dest_file" ]; then
    echo "[$(date '+%Y-%m-%d %H:%M:%S')] the time zone is not exists[$time_zone]."
    return
  fi
  echo y | cp "$dest_file" /etc/localtime
  echo "[$(date '+%Y-%m-%d %H:%M:%S')] config timezone success."
}

function config() {
  disable_iptables
  create_swap
  if [ $ip_version = 4 ]; then
    sed -i "/IPV6/d" /etc/sysconfig/network-scripts/ifcfg-eth0
    sed -i "/DHCPV6/d" /etc/sysconfig/network-scripts/ifcfg-eth0
  fi
  configTimeZone
  # please don't modify this echo message.
  echo "init successfully."
}

action=$1
if [ "$action" = "init" ]; then
  sys_ip=$2
  time_zone=$3
  ip_version=$4
  if [ -z "$sys_ip" ]; then
    echo "The ip is null."
    exit 1
  fi
  config
elif [ "$action" = "createUser" ]; then
  userName=$2
  echo "$userName" | grep -wE "DRManager|csha_admin|csdr_admin|vha_admin|quorumuser" &>/dev/null
  if [ $? != 0 ]; then
    echo "Create user failed, the user name is not support[$userName]"
    exit 1
  fi
  createUser
fi

exit $?
