# -*- coding:utf-8 -*-
import threading
import os
import shutil

import utils.common.log as logger
from utils.business.sfsutil import SFSApi

from plugins.eReplication.common.client.ssh_client import API as SSH_API
from plugins.eReplication.common.lib.model import Auth
from plugins.eReplication.common.lib.model import AuthFtp
from plugins.eReplication.common.lib.params import Nodes
from plugins.eReplication.common.lib.params import Params
from plugins.eReplication.common.lib.thread import ExcThread
from plugins.eReplication.common.constant import Path
from plugins.eReplication.common.constant import SFS_CERT_TYPE
from plugins.eReplication.common.constant import NOVA_CERT_TYPE
from plugins.eReplication.common.constant import MO_CERT_TYPE


class API(object):

    def __init__(self, host, ssh_user, ssh_pwd, sudo_user, sudo_pwd):
        self.host = host
        self.ssh_user = ssh_user
        self.ssh_pwd = ssh_pwd
        self.sudo_user = sudo_user
        self.sudo_pwd = sudo_pwd
        self.sudo_client = SSH_API.get_sudo_ssh_client(
            self.host, self.ssh_user, self.ssh_pwd, self.sudo_user,
            self.sudo_pwd)

    def __del__(self):
        if self.sudo_client:
            SSH_API.close_ssh(self.sudo_client)
        logger.info(f"SSH channel is closed: {self.host}.")

    @classmethod
    def get_instance(cls, *args, **kwargs):
        return cls(*args, **kwargs)

    @classmethod
    def get_sfs_certs(cls, project_id, pod_id):
        logger.info("Start copy remote SFS certs to temp path.")
        if not os.path.exists(Path.TEMP_CERT_PATH):
            os.mkdir(Path.TEMP_CERT_PATH)
        params = Params(project_id, pod_id)
        sfs_api = SFSApi(project_id, params.project_region_id)
        sfs_api.get_file(Path.TEMP_CERT_PATH)
        logger.info("Get SFS certs to temp path success.")

    @classmethod
    def import_sys_cert(cls, project_id, pod_id, cert_type):
        logger.info(f"Start import sys cert, cert type is {cert_type}.")
        if cert_type not in [SFS_CERT_TYPE, NOVA_CERT_TYPE, MO_CERT_TYPE]:
            logger.info(f"The cert type is incorrect.")
            return

        nodes = Nodes(project_id, pod_id)
        cert_funcs = list()
        thread_name = threading.current_thread().name
        for host in nodes.all_hosts:
            logger.info(f"Start import {cert_type} certs to {host}.")
            auth_provider = Auth(
                host, nodes.ssh_user, nodes.ssh_pwd, nodes.sudo_user,
                nodes.sudo_pwd).auth_provider
            cert_func = (
                cls._import_sys_cert_to_bcm, thread_name,
                (auth_provider, project_id, pod_id, cert_type), {}
            )
            cert_funcs.append(cert_func)
        ExcThread.exec_func_in_thread(cert_funcs)
        if os.path.exists(Path.TEMP_CERT_PATH):
            shutil.rmtree(Path.TEMP_CERT_PATH)
        logger.info("Import The sys cert is complete.")

    @classmethod
    def _import_sys_cert_to_bcm(
            cls, auth_provider, project_id, pod_id, cert_type):
        host = auth_provider.host
        ssh_user = auth_provider.ssh_user
        ssh_pwd = auth_provider.ssh_pwd
        sudo_user = auth_provider.sudo_user
        sudo_pwd = auth_provider.sudo_pwd
        ins = cls.get_instance(host, ssh_user, ssh_pwd, sudo_user, sudo_pwd)
        cert_lst = cls._get_cert_list(project_id, pod_id, cert_type)
        for cert_alias, cert_name in cert_lst.items():
            cmd = f"[ -d '{Path.DEST_CERT_PATH}' ]; echo CMD_RESULT=$?"
            result = SSH_API.exec_command_return_list(ins.sudo_client, cmd)
            if "CMD_RESULT=0" not in str(result):
                cmd = f"mkdir -p {Path.DEST_CERT_PATH} && chown {ssh_user}:LEGO {Path.DEST_CERT_PATH}"
                SSH_API.send_command(ins.sudo_client, cmd)
            cmd = f"[ -f '{Path.DEST_CERT_PATH}/{cert_name}' ] && " \
                  "echo CMD_RESULT=$?"
            result = SSH_API.exec_command_return_list(ins.sudo_client, cmd)
            if "CMD_RESULT=0" not in str(result):
                logger.info(f"{cert_type} cert {cert_name} not exists, now to get it.")
                cls._upload_sys_certs(host, ssh_user, ssh_pwd, cert_type)
                cmd = f"chown -Rh {ssh_user}:LEGO {Path.DEST_CERT_PATH}"
                SSH_API.send_command(ins.sudo_client, cmd)
                cmd = f"chmod -R 750 {Path.DEST_CERT_PATH}"
                SSH_API.send_command(ins.sudo_client, cmd)
            # 导入证书
            run_cmd = f"cd {Path.BCM_SCRIPTS_PATH} && sudo -u ICUser -s sh import_cert.sh; echo CMD_RESULT=$?"
            SSH_API.send_command(ins.sudo_client, run_cmd, "Please input the cert alias: ")
            SSH_API.send_command(ins.sudo_client, cert_alias, "Please input the cert file path: ")
            result = SSH_API.send_command(ins.sudo_client, f"{Path.DEST_CERT_PATH}/{cert_name}")
            if "CMD_RESULT=0" not in str(result):
                err_msg = f"{cert_type} cert {cert_name} failed to be imported to the node {host}."
                logger.error(err_msg)
                raise Exception(err_msg)
            logger.info(f"{cert_type} cert {cert_name} is imported to the node {host}.")
        else:
            logger.info(f"Import {cert_type} certs to node {host} success.")
            cmd = f"rm -rf {Path.DEST_CERT_PATH}"
            SSH_API.send_command(ins.sudo_client, cmd)

    @classmethod
    def _get_cert_list(cls, project_id, pod_id, cert_type):
        current_region_id = Params(project_id, pod_id).project_region_id
        if cert_type == SFS_CERT_TYPE:
            cert_lst = {f"{current_region_id}.sfs": "haproxy_tenant_ca-cert.pem"}
        elif cert_type == NOVA_CERT_TYPE:
            cert_lst = {f"{current_region_id}.nova": Path.NOVA_CERT_NAME}
        else:
            cert_lst = {
                "iamcert": "trust_apimlb_iam.cer", "sccert": "trust_apimlb_sc.cer",
                "occert": "trust_apimlb_oc.cer", "logcert": "cert.pem"}
        return cert_lst

    @classmethod
    def _upload_sys_certs(cls, host, ssh_user, ssh_pwd, cert_type):
        if cert_type == SFS_CERT_TYPE:
            sfs_file = f"{Path.TEMP_CERT_PATH}/haproxy_tenant_ca-cert.pem"
            cert_file_lst = [sfs_file]
        elif cert_type == NOVA_CERT_TYPE:
            cert_file_lst = [f"{Path.TEMP_CERT_PATH}/{Path.NOVA_CERT_NAME}"]
        else:
            iam_file = f"{Path.TEMP_CERT_PATH}/trust_apimlb_iam.cer"
            sc_file = f"{Path.TEMP_CERT_PATH}/trust_apimlb_sc.cer"
            oc_file = f"{Path.TEMP_CERT_PATH}/trust_apimlb_oc.cer"
            log_file = f"{Path.TEMP_CERT_PATH}/cert.pem"
            cert_file_lst = [iam_file, sc_file, oc_file, log_file]

        for cert_file in cert_file_lst:
            upload_res = SSH_API.put_file(
                host, ssh_user, ssh_pwd, cert_file, Path.DEST_CERT_PATH)
            if not upload_res:
                err_msg = f"Upload cert file to {host} failed."
                logger.error(err_msg)
                raise Exception(err_msg)
        logger.info(f"Upload {cert_type} certs to {host} success.")

    @classmethod
    def get_nova_cert(cls, project_id, pod_id):
        logger.info("Start copy remote nova_ca.crt to temp path.")
        if not os.path.exists(Path.TEMP_CERT_PATH):
            os.mkdir(Path.TEMP_CERT_PATH)
        # 复制到fsp目录
        if not cls._make_cert_file(project_id, pod_id):
            raise Exception("Copy remote nova_ca.crt failed.")
        # scp到本地
        params = Params(project_id, pod_id)
        reverse_ip = params.openstack_reverse_ip
        fsp_pwd = params.openstack_fsp_pwd
        auth_ftp = AuthFtp(reverse_ip, "fsp", fsp_pwd).auth_provider
        result = SSH_API.get_file(auth_ftp, Path.TEMP_NOVA_CERT_FILE,
                                  f"{Path.TEMP_CERT_PATH}/{Path.NOVA_CERT_NAME}")
        if not bool(result):
            raise Exception("Copy remote nova_ca.crt to local cert path failed.")
        logger.info("Copy remote nova_ca.certs to temp path success.")

    @classmethod
    def _make_cert_file(cls, project_id, pod_id):
        logger.info('Start copying nova_ca.crt and change owners.')
        params = Params(project_id, pod_id)
        reverse_ip = params.openstack_reverse_ip
        root_pwd = params.openstack_sudo_pwd
        fsp_pwd = params.openstack_fsp_pwd
        logger.info(f'Ssh to scp node:{reverse_ip}.')
        ssh_client = SSH_API.get_sudo_ssh_client(
            reverse_ip, "fsp", fsp_pwd, "root", root_pwd)
        cmd = fr"\cp -rf {Path.CPS_NOVA_CERT_FILE} {Path.TEMP_NOVA_CERT_FILE} && echo CMD_RESULT=$?"
        logger.info(f'Cmd:{cmd}.')
        result = SSH_API.exec_command_return(ssh_client, cmd)
        if "CMD_RESULT=0" not in str(result):
            raise Exception("Copy nova_ca.crt failed.")
        cmd = f"chown fsp:fsp {Path.TEMP_NOVA_CERT_FILE} && echo CMD_RESULT=$?"
        logger.info(f'Cmd:{cmd}.')
        result = SSH_API.exec_command_return(ssh_client, cmd)
        if "CMD_RESULT=0" not in str(result):
            raise Exception("Change owners failed.")
        logger.info("Copying nova_ca.crt and change owners success.")
        return True
