# -*- coding: utf-8 -*-

import re
import threading
from collections import defaultdict

import yaml
from IPy import IP
from utils.DBAdapter.DBConnector import BaseOps
from utils.common import log as logger
from utils.common.exception import HCCIException

from plugins.eReplication.common.client.dmk_client import API as DMK_API
from plugins.eReplication.common.constant import Action
from plugins.eReplication.common.constant import Component
from plugins.eReplication.common.lib.conditions import Condition
from plugins.eReplication.common.lib.params import Params
from plugins.eReplication.common.lib.utils import check_param_ip


class DomainType:
    console_domain = 'console_domain'
    iam_domain = 'iam_domain'
    vnc_server_domains = 'vnc_server_domains'
    ecs_mount_iso = 'ecs_mount_iso'
    VALID_DOMAIN = {console_domain, iam_domain, vnc_server_domains,
                    ecs_mount_iso}


class NginxConfigChecker:
    MIN_PORT = 0
    MAX_PORT = 65535
    DOMAIN_MATCH = \
        r'^[a-zA-Z0-9][-a-zA-Z0-9]{0,62}(\.[a-zA-Z0-9][-a-zA-Z0-9]{0,62})+$'

    @staticmethod
    def check_action(action):
        msg = ''
        if action not in ["Register_Nginx", "Remove_Nginx"]:
            msg = f'action {action} is invalid.'
        return msg

    @staticmethod
    def check_ip(host):
        return check_param_ip(host)

    @staticmethod
    def check_domain_type(domain_type):
        msg = ''
        if domain_type not in DomainType.VALID_DOMAIN:
            msg = f'domain type {domain_type} invalid, ' \
                  f'should in dict {DomainType.VALID_DOMAIN}'
        return msg

    @staticmethod
    def check_domain(domain_name):
        msg = ''
        pattern = re.compile(NginxConfigChecker.DOMAIN_MATCH)
        if not pattern.match(domain_name):
            msg = f"domain name {domain_name} invalid, " \
                  "should likes console.test.com"
        return msg

    @staticmethod
    def check_upstream(upstream_name):
        msg = ''
        if not upstream_name:
            msg = 'upstream name should not empty'
        return msg

    @staticmethod
    def check_ips(ips):
        msg = ''
        if not ips:
            msg = "ips should not empty. "
            return msg
        for host in ips:
            if not NginxConfigChecker.check_ip(host):
                msg = f"ip address {host} invalid, format is error. "
        return msg

    @staticmethod
    def check_port(port):
        port = int(port)
        msg = ''
        if port < NginxConfigChecker.MIN_PORT \
                or port > NginxConfigChecker.MAX_PORT:
            msg += f'port {port} is invalid, should be > ' \
                   f'{NginxConfigChecker.MIN_PORT} and < ' \
                   f'{NginxConfigChecker.MAX_PORT} '
        return msg

    @staticmethod
    def check_location(location):
        msg = ''
        if not location:
            return msg
        for loc in location:
            if not loc.get('name', None):
                msg += "location name is empty. "
            if loc.get('attach_url', None) \
                    and not loc['attach_url'].startswith('/'):
                msg += "location attach_url must start with '/', " \
                       f"invalid value: {loc['attach_url']}. "
        return msg


class RegionType(object):
    PRIMARY = 'primary'
    STANDBY = 'standby'


class API(object):
    SERVICE_DMK_NAME = 'nginx'
    SERVICE_BUSINESS_NAME = 'onframework'
    COMPONENT_NAME = 'nginx'
    SERVICE_NAME = 'nginx'
    operator_lock = threading.Lock()

    def __init__(self):
        self.condition = None
        self.params = None
        logger.init('NginxPublicAPI')

    def register_to_nginx(
            self, action, pod_id, domain_name, upstream_name, location, ips,
            port, listen_port=443, domain_type=DomainType.console_domain,
            to_primary=True, to_standby=False):
        """
        Nginx注册接口内部可以实现基本场景的判断，覆盖CSDR场景。
        内部实现逻辑：
        Nginx默认应该只在主region需要注册，CSDR场景需要在备region注册。
        除DomainType.vnc_server_domains和DomainType.ecs_mount_iso，其他类型都是
        主region执行则注册到主region，备region执行注册到备region。
        **对于其他场景，如PaaS服务，每个region有独立的Console,需要使用场景调用开关，
        设置要在哪个region注册
        :param action: Register_Nginx or Remove_Nginx
        :param pod_id: POD ID
        :param domain_name: 服务所在web页面的访问域名
        :param upstream_name: 服务名，需唯一
        :param location: 服务转发规则
        :param ips: 服务进程所在IP地址列表
        :param port: 服务进程的端口号
        :param listen_port: web 界面的访问端口
        :param domain_type: 服务所在web界面类型,必须引用DomainType里定义的字段
        :param to_primary: 是否注册到主站点，True表示注册到主站点，False表示不注册到主站点
        :param to_standby: 是否注册到备站点，True表示注册到备站点，False表示不注册到备站点
        :return:
        """
        API.operator_lock.acquire()
        try:
            logger.info('start register nginx')
            params = {
                'action': action,
                'pod_id': pod_id,
                'domain_type': domain_type,
                'domain_name': domain_name,
                'upstream_name': upstream_name,
                'location': location if location else '',
                'ips': ips,
                'port': int(port),
                'listen_port': int(listen_port)
            }
            logger.info(f'params: {params}')
            self._check_params(params)
            project_id = BaseOps().get_project_id_by_pod_id(pod_id)
            self.condition = Condition(project_id)
            self.params = Params(project_id, pod_id)
            if to_primary:
                primary_region_dmk_client = self._get_dmk_client(
                    RegionType.PRIMARY)
                self._update_nginx_conf(params, primary_region_dmk_client)
            if to_standby:
                standby_region_dmk_client = self._get_dmk_client(
                    RegionType.STANDBY)
                self._update_nginx_conf(params, standby_region_dmk_client)
            return True
        except HCCIException as exception:
            logger.error(f"Update nginx conf failed: {exception}")
            raise exception
        except Exception as exception:
            logger.error(f"Update nginx conf failed: {exception}")
            raise HCCIException("215001", 'Nginx', str(exception))
        finally:
            API.operator_lock.release()

    def _get_dmk_client(self, region_type):
        dmk_ip = self.params. \
            dmk_float_ip if region_type == RegionType.PRIMARY else \
            self.params.standby_dmk_float_ip
        dmk_nginx_user = self.params.nginx_dmk_user
        dmk_nginx_password = self.params. \
            nginx_dmk_user_pwd if region_type == RegionType.PRIMARY else \
            self.params.nginx_standby_dmk_user_pwd
        logger.info(f'dmk_ip: {dmk_ip},dmk_nginx_user: {dmk_nginx_user}')
        DMK_API.login_dmk(dmk_ip, dmk_nginx_user, dmk_nginx_password)
        return DMK_API.DMK

    def _update_nginx_conf(self, params, dmk_client):
        user_id = dmk_client.get_dmk_account_id(
            API.SERVICE_DMK_NAME, API.SERVICE_BUSINESS_NAME)
        config, host, ver = dmk_client.get_dmk_new_deploy_config(
            Component.NGINX_L, action=Action.UPDATE_NGINX)
        new_config = self._modify_config(config, params)
        dmk_client.excute_dmk_deployment(
            API.SERVICE_NAME, True, ver, Action.UPDATE_NGINX, host,
            new_config, user_id)
        logger.info("update nginx config successfully")

    @staticmethod
    def _check_params(params):
        msg_list = []

        def add_msg(error_msg):
            if error_msg:
                msg_list.append(error_msg)

        add_msg(NginxConfigChecker.check_action(params.get('action')))
        add_msg(
            NginxConfigChecker.check_domain_type(params.get('domain_type')))
        add_msg(NginxConfigChecker.check_upstream(params.get('upstream_name')))
        if params.get('domain_type') != DomainType.ecs_mount_iso:
            add_msg(NginxConfigChecker.check_domain(params.get('domain_name')))
        add_msg(NginxConfigChecker.check_ips(params.get('ips')))
        add_msg(NginxConfigChecker.check_port(params.get('port')))
        add_msg(NginxConfigChecker.check_port(params.get('listen_port')))
        if params.get('domain_type') not in (DomainType.vnc_server_domains,
                                             DomainType.ecs_mount_iso):
            add_msg(NginxConfigChecker.check_location(params.get('location')))
        if msg_list:
            raise HCCIException("215002", '|'.join(msg_list))

    def _modify_config(self, last_config, params):
        action = params.get('action')
        domain_type = params.get('domain_type')
        upstream_name = params.get('upstream_name')
        domain_name = params.get('domain_name')
        ips = params.get('ips')
        port = params.get('port')
        listen_port = params.get('listen_port')
        location = params.get('location')
        if action == "Register_Nginx":
            return self._register_nginx(
                last_config, domain_type, domain_name, upstream_name, location,
                ips, port, listen_port)
        return self._remove_nginx(
            last_config, domain_type, domain_name, upstream_name, ips,
            listen_port)

    @staticmethod
    def _register_nginx(
            last_config, domain_type, domain_name, upstream_name, location,
            ips, port, listen_port):
        old_config = yaml.safe_load(last_config)
        # find and update domain type
        servers = old_config['servers']
        if not servers:
            old_config['servers'] = {}
        if domain_type not in servers:
            servers[domain_type] = []
        domains = servers.get(domain_type, [])
        domain = defaultdict()
        ip_list = ips[:]
        if domain_type == DomainType.ecs_mount_iso:
            for upstream in domains:
                if upstream["name"] == upstream_name:
                    domains.remove(upstream)
            for index, host in enumerate(ip_list):
                if IP(host).version() == 6:
                    ip_list[index] = '[' + host + ']'
            upstream_dict = {
                'name': upstream_name, 'ips': ip_list, 'port': port,
                'listen_port': listen_port
            }
            domains.append(upstream_dict)
        else:
            for tmp in domains:
                if domain_name in tmp['domain_name'] and int(
                        tmp['listen_port']) == int(listen_port):
                    domain = tmp
                    logger.info("domain name exist ")
            if not domain:
                logger.info("domain name not exist")
                domain['domain_name'] = [domain_name]
                domain['listen_port'] = listen_port
                domain['upstreams'] = []
                domains.append(domain)
            # find and update upstream
            for upstream in domain['upstreams']:
                if upstream['name'] == upstream_name:
                    logger.info(f"upstream exist, old config:  {upstream}")
                    domain['upstreams'].remove(upstream)
            # insert config
            for index, host in enumerate(ip_list):
                if IP(host).version() == 6:
                    ip_list[index] = '[' + host + ']'
            upstream_dict = {'name': upstream_name, 'location': location,
                             'ips': ips, 'port': port}
            logger.info(f"new upstream_dict:  {upstream_dict}")
            domain['upstreams'].append(upstream_dict)
        new_config = yaml.dump(old_config, default_flow_style=False)
        logger.info(f"modify nginx config, new config:  {new_config}")
        return new_config

    @staticmethod
    def _remove_nginx(
            last_config, domain_type, domain_name, upstream_name, ips,
            listen_port):
        old_config = yaml.safe_load(last_config)
        servers = old_config["servers"]
        # 删除之前先判断有没有
        if not servers:
            servers = dict()
            old_config["servers"] = servers
        if domain_type not in servers:
            servers["domain_type"] = []
        domains_delete = servers.get(domain_type)
        if not domains_delete:
            new_config = yaml.dump(old_config, default_flow_style=False)
            logger.info("modify nginx config, new config:  %s " % new_config)
            return new_config
        domain = defaultdict()
        ip_list = ips[:]
        if domain_type == DomainType.ecs_mount_iso:
            for upstream in domains_delete:
                if upstream['name'] == upstream_name:
                    domains_delete.remove(upstream)
            for index, host in enumerate(ip_list):
                if IP(host).version() == 6:
                    ip_list[index] = '[' + host + ']'
        else:
            for tmp in domains_delete:
                if domain_name in tmp['domain_name'] and int(
                        tmp['listen_port']) == int(listen_port):
                    domain = tmp
            if not domain:
                # 当前没有这个domian,直接返回
                new_config = yaml.dump(old_config, default_flow_style=False)
                return new_config
            for upstream in domain['upstreams']:
                if upstream['name'] == upstream_name:
                    domain['upstreams'].remove(upstream)
        new_config = yaml.dump(old_config, default_flow_style=False)
        logger.info("modify nginx config, new config:  %s " % new_config)
        return new_config

    def get_dr_nginx_config(self, pod_id):
        data = dict()
        project_id = BaseOps().get_project_id_by_pod_id(pod_id)
        self.condition = Condition(project_id)
        self.params = Params(project_id, pod_id)
        primary_region_dmk_client = self._get_dmk_client(
            RegionType.PRIMARY)
        ips, domains, services = self._get_dr_service_nginx_data(
            primary_region_dmk_client)
        if "true" in str(services).lower():
            data["primary"] = (ips, domains, services)
        if self.condition.is_global_con_dr:
            standby_region_dmk_client = self._get_dmk_client(
                RegionType.STANDBY)
            data["standby"] = self._get_dr_service_nginx_data(
                standby_region_dmk_client)
        logger.info(f"Get dr nginx config return {data}.")
        return data

    def _get_dr_service_nginx_data(self, dmk_client):
        old_config, host, env_ver = dmk_client.get_dmk_new_deploy_config(
            Component.NGINX_L, action=Action.UPDATE_NGINX)
        nginx_yaml_cfg = yaml.safe_load(old_config)
        console_domains = list()
        console_ips = list()
        console_service_types = {
            'csdr': "False", 'vha': 'False', 'csha': 'False'}
        servers = nginx_yaml_cfg.get("servers")
        for key in servers:
            lst = servers.get(key)
            if not lst or len(lst) == 0:
                continue
            for obj in lst:
                upstreams = obj.get("upstreams")
                if not upstreams or len(upstreams) == 0:
                    continue
                ips, domains, service_types = \
                    self._analysis_nginx_upstreams(obj, upstreams)
                console_ips.extend(ips)
                console_domains.extend(domains)
                for service in service_types:
                    console_service_types[service] = "True"
        logger.info(
            f"Get Nginx config return console ips: {console_ips}, "
            f"console domain: {console_domains}, "
            f"services: {console_service_types}.")
        return console_ips, console_domains, console_service_types

    @staticmethod
    def _analysis_nginx_upstreams(obj, upstreams):
        console_domains = set()
        console_ips = set()
        service_types = set()
        for stream in upstreams:
            locations = stream.get('location')
            if not locations or len(locations) == 0:
                continue
            for location in locations:
                random_code = location.get('random_code')
                if random_code not in ["csdr", "csha", "vha"]:
                    continue
                service_types.add(random_code)
                ips = stream.get('ips')
                domain_lst = obj.get("domain_name")
                for _ip in ips:
                    console_ips.add(_ip)
                for _domain in domain_lst:
                    console_domains.add(_domain)
        return console_ips, console_domains, service_types
