# -*- coding:utf-8 -*-
import utils.common.log as logger
from utils.DBAdapter.DBConnector import BaseOps
from utils.business.param_util import HandleParamCheck
from utils.business.param_util import ParamUtil
from utils.business.project_util import ProjectApi
from utils.common.code2Msg import code2Msg
from utils.common.exception import HCCIException

from plugins.eReplication.common.constant import Component
from plugins.eReplication.common.lib.conditions import Condition
from plugins.eReplication.common.lib.model import Auth
from plugins.eReplication.common.lib.params import Nodes
from plugins.eReplication.common.lib.utils import check_value_null
from plugins.eReplication.common.os_api import API as OS_API


class ParamsCheck(object):
    """
        参数校验类，封装一些公共方法
    """

    def __init__(self, project_id):
        self.project_id = project_id
        self.pod_id = BaseOps().get_pod_id_by_project_id(self.project_id)
        self.region_id = ProjectApi().get_regionlist_by_project_Id(
            self.project_id)[0]
        self.service_type = Component.REPLICATION
        self.source_server_ip_key = "source_site_eReplication_physical_ip"
        self.source_server_business_key = \
            "source_site_server_business_user_pwd"
        self.source_server_root_key = "source_site_server_root_pwd"
        self.target_server_ip_key = "target_site_eReplication_physical_ip"
        self.target_server_business_key = \
            "target_site_server_business_user_pwd"
        self.target_server_root_key = "target_site_server_root_pwd"
        self.source_check_list = [
            self.source_server_ip_key, self.source_server_business_key,
            self.source_server_root_key]
        self.target_check_list = [
            self.target_server_ip_key, self.target_server_business_key,
            self.target_server_root_key]
        self.nodes = Nodes(self.project_id, self.pod_id)
        self.db_opt = ParamUtil()
        self.need_check_data = self.db_opt.get_need_check_cloud_params(
            self.project_id, self.service_type)
        self.os_users_check_lst = list()
        self.condition = Condition(self.project_id)
        self.check_result = HandleParamCheck()

    def _get_check_list(self):
        if self.condition.is_primary:
            check_list = self.source_check_list + self.target_check_list
        elif self.condition.is_sub and self.condition.is_global_con_dr:
            check_list = self.target_check_list + []
        else:
            check_list = self.target_check_list
        return check_list

    def basic_check(self):
        check_list = self._get_check_list()
        # 校验所有参数值是否为空
        for key in check_list:
            if key not in list(self.need_check_data.keys()):
                logger.error(f'Can not find the param key: {key}.')
                self.check_result.set_check_result(
                    param_keys=[key], status=500, error_msg=code2Msg(665004))
            elif check_value_null(self.need_check_data.get(key)):
                logger.error(f'The key value is None: {key}.')
                self.check_result.set_check_result(
                    param_keys=[key], status=500, error_msg=code2Msg(665005))

    def _check_server_ip_login_and_user_expire(
            self, host, ssh_pwd, sudo_pwd, key_list):
        login_ok, check_ok = False, False
        error_list = list()
        users = ["DRManager", "root"]
        auth_provider = Auth(
            host, "DRManager", ssh_pwd, "root", sudo_pwd).auth_provider
        try:
            result = OS_API.check_os_password_expired_day(auth_provider, users)
        except Exception as e:
            logger.error(f"Check os user expire failed: {e}")
            self.check_result.set_check_result(
                key_list, status=500, error_msg=HCCIException(665009))
            return login_ok, check_ok
        login_ok = True
        for user, expire_day in result.items():
            if expire_day > 7:
                continue
            logger.error(
                f"The password of {user} on the {host} expires in "
                f"{expire_day} days. ")
            error_list.append({user: expire_day})
        if error_list:
            self.check_result.set_check_result(
                key_list, status=500, error_msg=HCCIException(
                    675021, error_list))
        else:
            check_ok = True
        return login_ok, check_ok

    def _verify_specified_site_params(self, is_source=False):
        """
        检查服务端的参数
        :param is_source: true:检查source端，false:检查target端
        """
        if is_source:
            ssh_pwd_key = self.source_server_business_key
            sudo_pwd_key = self.source_server_root_key
            host = self.need_check_data.get(self.source_server_ip_key)
            ssh_pwd = self.need_check_data.get(ssh_pwd_key)
            sudo_pwd = self.need_check_data.get(sudo_pwd_key)
            check_list = self.source_check_list
        else:
            ssh_pwd_key = self.target_server_business_key
            sudo_pwd_key = self.target_server_root_key
            host = self.need_check_data.get(self.target_server_ip_key)
            ssh_pwd = self.need_check_data.get(ssh_pwd_key)
            sudo_pwd = self.need_check_data.get(sudo_pwd_key)
            check_list = self.target_check_list
        login1, check1 = self._check_server_ip_login_and_user_expire(
            host, ssh_pwd, sudo_pwd, check_list)
        self.os_users_check_lst.append(check1)
        if not login1:
            logger.error(f'Login {host} failed.')
            return self.check_result
        all_ips = self.nodes.get_all_server_nodes(
            host, "DRManager", ssh_pwd, "root", sudo_pwd)
        if len(all_ips) not in [2, 4] or host not in all_ips:
            logger.error(f"Query eReplication ip from {host} failed.")
            self.check_result.set_check_result(
                param_keys=check_list, status=500, error_msg=code2Msg(665006))
            return self.check_result
        all_ips.remove(host)
        if not all_ips:
            return self.check_result
        for host in all_ips:
            login2, check2 = self._check_server_ip_login_and_user_expire(
                host, ssh_pwd, sudo_pwd, check_list)
            self.os_users_check_lst.append(check2)
        if False not in self.os_users_check_lst:
            self.check_result.set_check_result(
                param_keys=check_list, status=200, error_msg="")
        if not self.check_result.check_list:
            self.check_result.set_check_result(
                param_keys=check_list, status=200, error_msg="")
        return self.check_result

    def do_check(self):
        if self.condition.is_primary:
            self._verify_specified_site_params(is_source=True)
        self._verify_specified_site_params(is_source=False)
        if not self.check_result.check_list:
            self.check_result.set_check_result(
                param_keys=self._get_check_list(), status=200, error_msg='')
        return self.check_result
