import json
import os
import re
import time
import zipfile
from collections import namedtuple

import utils.common.log as logger
from utils.common.check_result import CheckResult
from utils.common.ssh_util import Ssh
from utils.common.exception import HCCIException
from utils.constant.path_constant import SOURCES_ROOT_DIR
from utils.business.project_util import ProjectApi
from utils.DBAdapter.DBConnector import BaseOps
from utils.business.param_util import ParamUtil

from plugins.CSBS.common.upgrade import constant
from plugins.CSBS.common.util import ConfigIniFile
from plugins.CSBS.common.util import check_ip
from plugins.CSBS.common.util import compare_version
from plugins.CSBS.common.ssh_client import SshClient
from plugins.CSBS.common.upgrade import models
from plugins.CSBS.common.upgrade.params import ParamsTools


class ProxySsh(Ssh):
    @classmethod
    def ssh_su_root_cmds(cls, node_ip, cmds, fsp_user, fsp_pass, passwd, port, pub_key=None, timeout=20):
        listout = []

        def do_su_root_cmds():
            nonlocal listout
            ssh = Ssh.ssh_create_client(node_ip, fsp_user, fsp_pass, port=port,
                                        pub_key=pub_key)
            time.sleep(1)
            logger.info("begin swap as root")
            if fsp_user.lower() != 'root':
                Ssh.ssh_send_command(ssh, 'sudo su', 'password', timeout)
                Ssh.ssh_send_command(ssh, passwd, '#', timeout)
            cmdsend = cmds
            outs = Ssh.ssh_exec_command_return_list(ssh, cmdsend, timeout=2)
            Ssh.ssh_close(ssh)
            cls.deal_outs(cmds, listout, outs)

        try:
            do_su_root_cmds()
        except HCCIException as error:
            raise error
        except Exception as error:
            msg = str(error).replace(cmds, "") if cmds in str(error) else str(
                error)
            err_msg = HCCIException(constant.HHCIErrorCode.RUN_SSH_CMD.value, node_ip, msg)
            logger.error(str(err_msg))
            raise Exception(str(err_msg)) from error
        return listout


class KarborProxyOperation(object):
    def __init__(self, project_id):
        self.project_id = project_id
        self.param_util = ParamUtil()
        self.service_name = "CSBS-VBS"
        self.karborproxy_node_list = self.get_proxy_node_list()
        self.project_api = ProjectApi()
        self.ssh_client = SshClient()
        # put_file_to_all_roots need obj.root
        self.root_name = self.get_karborproxy_user_info().user_name
        self.root_pwd = self.get_karborproxy_user_info().user_pwd
        self.port = 22
        self.data_protection_app_mode = "normal"
        self.params_tools = ParamsTools(self.project_id)

    def get_karbor_float_ip(self):
        cbs_node0 = self.params_tools.get_karbor_node_list()[0]
        logger.info(f"Get float_ip from the Karbor node: {cbs_node0.node_ip}.")
        ssh_client = self.ssh_client.get_ssh_client(cbs_node0)
        karbor_version = self.params_tools.get_karbor_version().get("short_version")
        if compare_version(karbor_version, "8.2.0.SPC500"):
            cmds = 'source /opt/huawei/dj/inst/utils.sh;' \
                   'get_info --tenant_float_ip'
        else:
            cmds = 'source /opt/huawei/dj/inst/utils.sh;' \
                   'get_info.py --tenant_float_ip'
        logger.info(f"Cmd: {cmds}.")
        result = self.ssh_client.ssh_exec_command_return(ssh_client, cmds)
        if not self.ssh_client.is_ssh_cmd_executed(result):
            raise Exception(f"Failed to execute cmd: {cmds}.")
        logger.info(f"Get karbor float ip info: {result}.")
        valid_ip_list = [item for item in result if check_ip(item)]
        if valid_ip_list:
            karbor_float_ip = valid_ip_list[0].strip()
            logger.info(f"Karbor float ip is: {karbor_float_ip}.")
            return karbor_float_ip
        else:
            raise Exception("Get Karbor float ip failed.")

    def get_region_id(self):
        region_list = BaseOps().get_regionId_by_projectId(self.project_id)
        if not region_list:
            raise Exception("Failed to obtain the region id.")
        return region_list[0]

    def get_karborproxy_user_info(self):
        region_id = self.get_region_id()
        user_name = self.param_util.get_value_from_cloud_param(
            self.project_id, self.service_name,
            "karborproxy_dpamanager_username", region_id)
        if not user_name:
            raise Exception("Failed to obtain the username of karborproxy.")
        user_pwd = self.param_util.get_value_from_cloud_param(
            self.project_id, self.service_name,
            "karborproxy_dpamanager_pwd", region_id)
        if not user_pwd:
            raise Exception("Failed to obtain the pwd of karborproxy username.")

        root_pwd = self.param_util.get_value_from_cloud_param(
            self.project_id, self.service_name,
            "karborproxy_root_pwd", region_id)
        if not root_pwd:
            raise Exception("Failed to obtain the pwd of proxy root.")
        return models.NodeUser(user_name=user_name,
                               user_pwd=user_pwd,
                               root_name="root",
                               root_pwd=root_pwd)

    def get_proxy_node_list(self):
        karborproxy_ip_dict = self.get_karborproxy_vm_ips()
        karborproxy_user_info = self.get_karborproxy_user_info()
        karborproxy_node_list = []
        for _, karborproxy_ip in karborproxy_ip_dict.items():
            cbs_node = models.CBSNode(
                node_ip=karborproxy_ip,
                user=karborproxy_user_info.user_name,
                user_pwd=karborproxy_user_info.user_pwd,
                root_pwd=karborproxy_user_info.root_pwd,
                extra="")
            karborproxy_node_list.append(cbs_node)
        return karborproxy_node_list

    def get_karborproxy_vm_ips(self):
        vm_names = ["CSBS-KarborProxy-01", "CSBS-KarborProxy-02"]
        vm_name_keys = ['karborproxy_host0', 'karborproxy_host1']
        vm_ips = []
        for vm_name in vm_names:
            vm_info = self.param_util.get_vm_info_by_vm_name(self.project_id, vm_name)
            if not vm_info:
                raise Exception('Failed to getting the VM information based '
                                'on the VM name, vm name:{}.'.format(vm_name))
            vm_ips.append(vm_info[0].get('ip'))
        vm_ip_dict = dict(zip(vm_name_keys, vm_ips))
        return vm_ip_dict

    @property
    def all_ip_list(self):
        proxy_ip_list = [karborproxy_ip for karborproxy_ip in self.get_karborproxy_vm_ips().values()]
        if proxy_ip_list:
            return proxy_ip_list
        raise Exception('Get proxy ip list fail.')

    def switch_karborproxy_os_cipher(self, cipher_type):
        if cipher_type not in (constant.CipherType.SM_COMPATIBLE.value, constant.CipherType.GENERAL_CIPHER.value):
            raise Exception(f"Unsupported cipher type, the input cipher_type parameter is {cipher_type}.")

        logger.info(f"Ssh to karborproxy node and switch OS cipher, target cipher type is {cipher_type}.")
        for node in self.karborproxy_node_list:
            karborproxy_client = self.ssh_client.get_ssh_client(node)
            cmd = f"bash /opt/huawei-data-protection/root_tools/ops/changeOsCipher.sh {cipher_type}"
            result = self.ssh_client.ssh_exec_command_return(karborproxy_client, cmd)
            if not self.ssh_client.is_ssh_cmd_executed(result):
                logger.error(f"Failed to switch OS cipher type on karborproxy node, node ip:{node.node_ip}.")
                self.ssh_client.ssh_close(karborproxy_client)
                return False
            logger.info(f"Succeed to switch OS cipher type on karborproxy node, node ip:{node.node_ip}.")
            self.ssh_client.ssh_close(karborproxy_client)
        return True

    def switch_karborproxy_cipher(self, cipher_type):
        if cipher_type not in (constant.CipherType.SM_COMPATIBLE.value, constant.CipherType.GENERAL_CIPHER.value):
            raise Exception(f"Unsupported cipher type, the input cipher_type parameter is {cipher_type}.")

        logger.info(f"Ssh to karborproxy node and switch cipher, target cipher type is {cipher_type}.")
        for node in self.karborproxy_node_list[:2]:
            karborproxy_client = self.ssh_client.get_ssh_client(node)
            cmd = "proxython /opt/huawei-data-protection/agentproxy/bin/switch_cipher_type.py " \
                  f"{cipher_type}"
            result = self.ssh_client.ssh_exec_command_return(karborproxy_client, cmd)
            if not self.ssh_client.is_ssh_cmd_executed(result):
                logger.error(f"Failed to switch cipher type on karborproxy node, node ip:{node.node_ip}.")
                self.ssh_client.ssh_close(karborproxy_client)
                continue
            logger.info(f"Succeed to switch cipher type on karborproxy node, node ip:{node.node_ip}.")
            self.ssh_client.ssh_close(karborproxy_client)
            return True
        return False

    def get_karborproxy_cur_cipher_type(self):
        logger.info("Ssh to karborproxy node and get cipher type.")
        exec_result = None
        for node in self.karborproxy_node_list[:2]:
            karborproxy_client = self.ssh_client.get_ssh_client(node)
            cmd = "proxython /usr/bin/get_info.py cipher_type"
            exec_result = self.ssh_client.ssh_exec_command_return(karborproxy_client, cmd)
            if not self.ssh_client.is_ssh_cmd_executed(exec_result):
                logger.error(f"Failed to get cipher type on karborproxy node, node ip:{node.node_ip}.")
                self.ssh_client.ssh_close(karborproxy_client)
                continue
            logger.info(f"Succeed to execute command of '{cmd}' on karborproxy node.")
            self.ssh_client.ssh_close(karborproxy_client)
            break
        if constant.CipherType.SM_COMPATIBLE.value in exec_result:
            return constant.CipherType.SM_COMPATIBLE.value
        elif constant.CipherType.GENERAL_CIPHER.value in exec_result:
            return constant.CipherType.GENERAL_CIPHER.value
        raise Exception("Failed to obtain the cipher type from karborproxy node.")

    def get_karborproxy_version(self):
        karborproxy_client = self.ssh_client.get_ssh_client(self.karborproxy_node_list[0])
        cmd = "grep release_version /opt/huawei-data-protection/agentproxy/conf/version | cut -d = -f 2"
        result = self.ssh_client.ssh_exec_command_return(karborproxy_client, cmd)
        if not self.ssh_client.is_ssh_cmd_executed(result):
            raise Exception(f"Failed to execute cmd grep {cmd}.")
        self.ssh_client.ssh_close(karborproxy_client)
        return result[0]

    def get_karborproxy_cpu_arch(self):
        logger.info("Ssh to karborproxy node and get CPU architecture.")
        exec_result = None
        karborproxy_node_list = self.karborproxy_node_list
        for node in karborproxy_node_list[:2]:
            karborproxy_client = self.ssh_client.get_ssh_client(node)
            cmd = "uname -m"
            exec_result = self.ssh_client.ssh_exec_command_return(karborproxy_client, cmd)
            if not self.ssh_client.is_ssh_cmd_executed(exec_result):
                logger.error(f"Failed to get CPU architecture on proxy node, node ip:{node.node_ip}.")
                self.ssh_client.ssh_close(karborproxy_client)
                continue
            logger.info(f"Succeed to execute command of {cmd} on proxy node.")
            self.ssh_client.ssh_close(karborproxy_client)
            break
        if not exec_result:
            raise Exception("Failed to obtain the CPU architecture from karborproxy node.")

        result_str = "::".join(exec_result)
        if "x86_64" in result_str:
            return "x86"
        elif "aarch64" in result_str:
            return "arm"
        else:
            raise Exception("Failed to obtain the CPU architecture from karborproxy node.")

    def check_karborproxy_os_version(self):
        logger.info("Ssh to proxy node and check os version.")
        exec_result = None
        for node in self.karborproxy_node_list[:2]:
            karborproxy_client = self.ssh_client.get_ssh_client(node)
            cmd = f'uname -r | grep -E "{constant.EULEROS.get("EL2R8")}|{constant.EULEROS.get("EL2R11")}|' \
                  f'{constant.EULEROS.get("EL2R9")}|{constant.EULEROS.get("EL2R10")}" && echo successful'
            exec_result = self.ssh_client.ssh_exec_command_return(karborproxy_client, cmd)
            if not self.ssh_client.is_ssh_cmd_executed(exec_result):
                logger.error(f"Failed to get os version info on karborproxy node, node ip:{node.node_ip}.")
                self.ssh_client.ssh_close(karborproxy_client)
                continue
            logger.info(f"Succeed to execute command of '{cmd}' on karborproxy node.")
            self.ssh_client.ssh_close(karborproxy_client)
            break
        if not exec_result:
            raise Exception("Failed to obtain the CPU architecture from karborproxy node.")

        result_str = "::".join(exec_result)
        return "successful" in result_str

    @staticmethod
    def get_proxy_version_from_conf():
        env_config_path = os.path.join(SOURCES_ROOT_DIR, constant.ENV_CONFIG_PATH)
        conf_env_file = ConfigIniFile(env_config_path)
        proxy_params = conf_env_file.get_params_dict_by_key_name("KarborProxy")
        original_version = proxy_params.get("cur_version", "")
        target_version = proxy_params.get("tar_version", "")
        if not (original_version or target_version):
            raise Exception("Failed to karborproxy version from env_param.ini file.")
        version_model = namedtuple("karborproxy_version", ["original_version", "target_version"])
        return version_model(original_version=original_version,
                             target_version=target_version)

    @staticmethod
    def get_proxy_cipher_type():
        env_config_path = os.path.join(SOURCES_ROOT_DIR, constant.ENV_CONFIG_PATH)
        conf_env_file = ConfigIniFile(env_config_path)
        karbor_params = conf_env_file.get_params_dict_by_key_name("KarborProxy")
        cur_cipher_type = karbor_params.get("cur_cipher_type", "")
        tar_cipher_type = karbor_params.get("tar_cipher_type", "")
        if not (cur_cipher_type or tar_cipher_type):
            raise Exception("Failed to obtain cipher type.")
        return models.CipherType(cur_cipher_type=cur_cipher_type, tar_cipher_type=tar_cipher_type)

    def get_karborproxy_tar_cipher_type(self):
        project_conditions = self.project_api.get_project_conditions(self.project_id)
        if project_conditions.get(constant.CipherType.GENERAL_CIPHER.value):
            return constant.CipherType.GENERAL_CIPHER.value
        elif project_conditions.get(constant.CipherType.SM_COMPATIBLE.value):
            return constant.CipherType.SM_COMPATIBLE.value
        else:
            raise Exception("Failed to obtain the target cipher type.")

    def get_karborproxy_pkg_version(self):
        pkg_suffix = "Proxy-x86_64.zip" if self.params_tools.is_x86_scene else "Proxy-aarch64.zip"
        pkg_root_path, pkg_name = self.params_tools.get_pkg_path_and_pkg_name(pkg_suffix)
        pkg_path = os.path.join(pkg_root_path, pkg_name)
        file_fs = None
        pkg_version = None

        def do_get_pkg_version():
            nonlocal file_fs, pkg_version
            file_fs = zipfile.ZipFile(r'{}'.format(pkg_path), 'r')
            for file in file_fs.namelist():
                if file.endswith("release.json"):
                    pkg_version = json.loads(file_fs.open(file).read().decode())["release_version"]

        try:
            do_get_pkg_version()
        except Exception as err:
            raise Exception(f"Get package version failed, package name: {pkg_name}, err_msg: {str(err)}.") from err
        finally:
            if file_fs:
                file_fs.close()
        if not pkg_version:
            raise Exception(f"Get package version failed, package name: {pkg_name}.")
        return pkg_version

    def get_agent_assist_version(self):
        cmd = "ls /opt/huawei-data-protection/agentproxy/pkg/" \
              "agent_assistant/OceanStor_BCManager_*_AgentAssist_windows.zip"
        exec_result = None
        for node in self.karborproxy_node_list:
            karborproxy_client = self.ssh_client.get_ssh_client(node)
            exec_result = self.ssh_client.ssh_exec_command_return(karborproxy_client, cmd)
            if not self.ssh_client.is_ssh_cmd_executed(exec_result):
                logger.error(f"Failed to get os version info on karborproxy node, node ip:{node.node_ip}.")
                self.ssh_client.ssh_close(karborproxy_client)
                continue
            logger.info(f"Succeed to execute command of '{cmd}' on karborproxy node.")
            self.ssh_client.ssh_close(karborproxy_client)
            break
        if not exec_result:
            raise Exception(f"Get AgentAssist pkg version failed.")
        version = re.findall(r".*OceanStor_BCManager_(.*)_AgentAssist_windows.zip", exec_result[0])[0]
        if not version:
            raise Exception(f"Get AgentAssist pkg version failed.")
        logger.info(f'This AgentAssist pkg version is {version} .')
        return version

    def get_agent_version(self):
        cmd = "ls /opt/huawei-data-protection/agentproxy/pkg/agent/CDM_*_Client_Windows_All_x64.zip"
        exec_result = None
        for node in self.karborproxy_node_list:
            karborproxy_client = self.ssh_client.get_ssh_client(node)
            exec_result = self.ssh_client.ssh_exec_command_return(karborproxy_client, cmd)
            if not self.ssh_client.is_ssh_cmd_executed(exec_result):
                logger.error(f"Failed to get os version info on karborproxy node, node ip:{node.node_ip}.")
                self.ssh_client.ssh_close(karborproxy_client)
                continue
            logger.info(f"Succeed to execute command of '{cmd}' on karborproxy node.")
            self.ssh_client.ssh_close(karborproxy_client)
            break
        if not exec_result:
            raise Exception(f"Get CDM pkg version failed.")

        version = re.findall(r".*CDM_(.*)_Client_Windows_All_x64.zip", exec_result[0])[0]
        if not version:
            raise Exception(f"Get AgentAssist pkg version failed.")
        logger.info(f'This CDM pkg version is {version} .')
        return version

    def get_management_float_ip(self):
        cmd = "cat /opt/huawei-data-protection/agentproxy/conf/agentproxy.conf" \
              " | grep haproxy_float_ip | awk -F '=' '{print $2}'"
        node = self.karborproxy_node_list[0]
        karborproxy_client = self.ssh_client.get_ssh_client(node)
        exec_result = self.ssh_client.ssh_exec_command_return(karborproxy_client, cmd)
        if not self.ssh_client.is_ssh_cmd_executed(exec_result):
            self.ssh_client.ssh_close(karborproxy_client)
            raise Exception(f"Failed to get os float ip on karborproxy node, node ip:{node.node_ip}.")

        logger.info(f"Succeed to execute command of '{cmd}' on karborproxy node.")
        self.ssh_client.ssh_close(karborproxy_client)
        if not exec_result:
            raise Exception(f"Get karborproxy float ip failed.")
        return exec_result[0].strip()

    def get_karborproxy_upgrade_path_version(self, service="CSBS-VBS"):
        solution, version = self.project_api.get_solution_and_version(self.project_id)
        sol_original_version, sol_target_version = version.strip().split("-")
        plugin_file_path = "{}/{}".format(os.path.dirname(os.path.realpath(__file__)), "../../plugin.json")
        ser_version_dict = dict()
        if os.path.exists(plugin_file_path):
            with open(plugin_file_path, 'r') as file:
                json_str = file.read()
                plugin_dic = json.loads(json_str)
                ser_version_dict["original_version"] = \
                    plugin_dic['plugins'][0]['workflow'][2]['VersionMatching'][sol_original_version]
                ser_version_dict["target_version"] = \
                    plugin_dic['plugins'][0]['workflow'][2]['VersionMatching'][sol_target_version]
                return ser_version_dict
        else:
            raise Exception("The 'plugin.json' file does not exist.")

    def check_proxy_service(self):
        cmd = "showsys services"
        count = 0
        bad_service = list()
        for node in self.karborproxy_node_list:
            karborproxy_client = self.ssh_client.get_ssh_client(node)
            exec_result = self.ssh_client.ssh_exec_command_return(karborproxy_client, cmd)
            if not self.ssh_client.is_ssh_cmd_executed(exec_result):
                self.ssh_client.ssh_close(karborproxy_client)
                raise Exception(f"Failed to get proxy services, node ip:{node.node_ip}.")
            self.ssh_client.ssh_close(karborproxy_client)
            if not exec_result:
                raise Exception(f"check proxy service exec_result is None.")
            service_dict = dict()
            for result in exec_result:
                if result.count("|") != 1:
                    continue
                service_list = result.strip().replace(" ", "").split("|")
                service_dict[service_list[0]] = service_list[1]

            for service, state in service_dict.items():
                if service in ["AgentProxy", "HAProxy", "CPS-Monitor"] and state != "running":
                    bad_service.append(service)
                if service in ["ZooKeeper"] and state == "dead":
                    bad_service.append(service)
                if service in ["OMM-HA"] and state == "active" and service_dict.get("V_ZooKeeper") != "dead":
                    count += 1
                elif service in ["OMM-HA"] and state == "standby" and service_dict.get("V_ZooKeeper") == "dead":
                    count += 1
        if count != 2:
            bad_service.append("OMM-HA")
            bad_service.append("V_ZooKeeper")
        return bad_service

    def check_proxy_service_result(self, check_results, check_config):
        if self.check_proxy_service():
            result = CheckResult(
                itemname_ch=check_config["itemname_ch"],
                itemname_en=check_config["itemname_en"],
                status=check_config["failure_status"],
                error_msg_cn=check_config["error_msg_cn"],
                error_msg_en=check_config["error_msg_en"],
                suggestion_cn=check_config["suggestion_cn"],
                suggestion_en=check_config["suggestion_en"])
        else:
            result = CheckResult(
                itemname_ch=check_config["itemname_ch"],
                itemname_en=check_config["itemname_en"],
                status=check_config["success_status"])
        check_results.append(result)
        return check_results
