import os
import re
import zipfile

from plugins.CSBS.common import node_installer
from plugins.CSBS.common.constant import KARBOR_PKG_SUFFIX_X86, KARBOR_PKG_SUFFIX_ARM
from plugins.CSBS.common.karbor import KarborUtil
from plugins.CSBS.common.params_tool import ParamTool
from plugins.CSBS.common.ssh_client import SshClient
from plugins.CSBS.common.util import check_shell_inject_args
from plugins.CSBS.scripts.deploy.karborproxy.common.funtions import check_zipfile


class FileUpload(node_installer.ThreadInstaller):
    def __init__(self, project_id, pod_id):
        self.karbor_util = KarborUtil(project_id, pod_id)
        nod_list = self.karbor_util.get_karbor_node_list()
        super(FileUpload, self).__init__(nod_list)
        self.param_tool = ParamTool(project_id, pod_id)
        self.project_id = project_id
        self.pkg_pre_name = "OceanStor BCManager"
        self.ssh_client = SshClient()

    def clear_work_dir(self, karbor_node):
        karbor_client = self.ssh_client.get_ssh_client(karbor_node)
        work_dir = self.karbor_util.get_installer_workdir()
        cmds = "rm -rf " + work_dir
        self.ssh_client.ssh_exec_command_return(karbor_client, cmds)
        cmds = "mkdir " + work_dir
        self.ssh_client.ssh_exec_command_return(karbor_client, cmds)
        cmds = "cd " + work_dir
        self.ssh_client.ssh_exec_command_return(karbor_client, cmds)
        cmds = "sed -i s/'PermitRootLogin no'" \
               "/'PermitRootLogin yes'/g /etc/ssh/sshd_config"
        self.ssh_client.ssh_exec_command_return(karbor_client, cmds)
        cmds = "service sshd restart"
        self.ssh_client.ssh_exec_command_return(karbor_client, cmds)
        self.ssh_client.ssh_close(karbor_client)

    def upload_package(self, karbor_node):
        pkg_path = self.param_tool.get_pkg_path(self.pkg_pre_name, self.get_karbor_pkg_suffix(), None)
        check_zipfile([pkg_path])
        source_list = [pkg_path, self.param_tool.get_installer_config()]
        work_dir = self.karbor_util.get_installer_workdir()
        for source in source_list:
            if os.path.exists(source):
                self.ssh_client.put(karbor_node.node_ip, "root", karbor_node.root_pwd, source, work_dir)

    def untar_package(self, karbor_node):
        karbor_client = self.ssh_client.get_ssh_client(karbor_node)
        work_dir = self.karbor_util.get_installer_workdir()
        self.ssh_client.ssh_exec_command_return(karbor_client, "cd " + work_dir)
        karbor_pkg = os.path.basename(self.param_tool.get_pkg_path(self.pkg_pre_name,
                                                                   self.get_karbor_pkg_suffix(), None))
        cmds = f"unzip '{karbor_pkg}'"
        result = self.ssh_client.ssh_exec_command_return(karbor_client, cmds)
        if not self.ssh_client.is_ssh_cmd_executed(result):
            self.install_result[karbor_node.node_ip] = False
            return False
        self.check_csbs_full_file(karbor_client)
        cmds = f"tar -zxvf CSBS-*-full.tar.gz"
        result = self.ssh_client.ssh_exec_command_return(karbor_client, cmds)
        self.install_result[karbor_node.node_ip] \
            = self.ssh_client.is_ssh_cmd_executed(result)
        cmds = "find CSBS -name '*.sh' | xargs dos2unix"
        self.ssh_client.ssh_exec_command_return(karbor_client, cmds)
        config = os.path.basename(self.param_tool.get_installer_config())
        cmds = "dos2unix " + config
        self.ssh_client.ssh_exec_command_return(karbor_client, cmds)
        self.ssh_client.ssh_close(karbor_client)
        return True

    def install_thread(self, node):
        self.upload_package(node)
        self.untar_package(node)

    def check_thread(self, node):
        self.check_result[node.node_ip] = True

    def rollback_thread(self, node):
        self.clear_work_dir(node)
        self.rollback_result[node.node_ip] = True

    def get_karbor_pkg_suffix(self):
        return KARBOR_PKG_SUFFIX_ARM if self.param_tool.is_cpu_arm(self.project_id) else KARBOR_PKG_SUFFIX_X86

    def check_csbs_full_file(self, karbor_client):
        file_size = 0
        mb_size = 1024 * 1024 * 1024 * 5
        # 最大文件数量限定10万个
        max_file_count = 1 * 1000 * 100
        file_list_uniq = []
        check_cmd = f"tar -tvf CSBS-*-full.tar.gz"
        check_shell_inject_args(check_cmd)
        result = self.ssh_client.ssh_send_command_expect(karbor_client, check_cmd)
        for csbs_file in result:
            file_list = re.split('\s+', csbs_file)
            if file_list[-1].count('/') == 1 and file_list[-1].endswith('/'):
                file_list_uniq.append(file_list)
            if re.match(r'[0-9]+', file_list[2]):
                file_size += float(file_list[2])

        if len(file_list_uniq) > max_file_count:
            raise Exception("The file is too many")
        for file_uniq in file_list_uniq:
            if len(file_uniq) != 6:
                continue
            size = file_uniq[2].strip()
            if re.match(r'[0-9]+', size):
                file_size += float(size)
        if file_size > mb_size:
            raise Exception("The file is too large.")
