# -*- coding:utf-8 -*-
import time
from utils.business.param_util import ParamUtil
from utils.business.project_condition_utils import get_project_condition_boolean
from utils.common import log as logger
from utils.common.exception import HCCIException
from utils.common.fic_base import StepBaseInterface
from utils.common.message import Message
from utils.common.ssh_util import Ssh

logger.init("eBackup StandardSuseHardening")


class StandardSuseHardening(StepBaseInterface):
    def __init__(self, project_id, pod_id):
        params = ParamUtil()
        self.__config_dict = params.get_service_cloud_param(pod_id, "eBackup")
        self.project_id = project_id
        self.pod_id = pod_id
        self.ssh_obj = Ssh()
        self.harden_cmd = "dos2unix /home/hcp/StandardSuseHardening.sh;" \
                          "chmod 550 /home/hcp/StandardSuseHardening.sh;" \
                          "bash /home/hcp/StandardSuseHardening.sh"
        self.root_passwd = self.__config_dict['eBackup_os_root_password']
        self.hcp_user_pwd = self.__config_dict['hcp_ssh_password']
        self.gaussdb_pwd = self.__config_dict['eBackup_gaussdb_pwd']

    def pre_check(self, project_id, pod_id):
        """
        插件内部接口：执行安装前的资源预检查，该接口由execute接口调用，工具框架不会直接调用此接口。
        :param project_id:
        :param pod_id:
        :return:
        """

    def main_handle(self, project_id):
        self.standard_suse_hardening(project_id)
        time.sleep(120)
        self.set_admin_first_login(project_id)
        self.set_first_login(project_id)
        return Message(200)

    def execute(self, project_id, pod_id, *args, **kwargs):
        """
        标准调用接口：执行安装前预检查&安装&配置
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        try:
            return self.main_handle(project_id)
        except HCCIException as err:
            return Message(500, err)
        except Exception as err:
            logger.error("ebackup harden failed:%s" % str(err))
            return Message(500, err, err)

    def standard_suse_hardening(self, project_id):
        ip_list = self.get_ip_list()
        for os_ip in ip_list:
            self.ssh_obj.put_file(os_ip, "hcp", self.hcp_user_pwd,
                                  "/opt/cloud/hcci/src/HCCI/plugins/eBackup/shell_tool/StandardSuseHardening.sh",
                                  "/home/hcp", 22)
            logger.info("put file succ.")

            self.ssh_obj.ssh_cmds(os_ip, self.harden_cmd, "hcp", self.hcp_user_pwd, self.root_passwd, "", "")
        time.sleep(120)

        success_ip_list = []
        num = 0
        while num < 60:
            success_ip_list = []
            for server_ip in ip_list:
                logger.info("begin to connect:" + server_ip)
                if self.is_connected_server(server_ip):
                    logger.info(server_ip + " can connected.")
                    success_ip_list.append(server_ip)
            if len(success_ip_list) == len(ip_list):
                break
            time.sleep(20)
            num = num + 1

        if num >= 60:
            ip_str = ','.join(success_ip_list)
            raise HCCIException(653041, ip_str)

    def do_set_first_login(self, os_ip, first_login_cmd):
        result = self.ssh_obj.ssh_cmds(os_ip, first_login_cmd, "hcp", self.hcp_user_pwd, self.root_passwd, "", "")
        if str(result).find("exec-result: 0") == -1:
            raise Exception("exec set first login command failed on node(%s). stdout:%s" % (os_ip, str(result)))

    def set_first_login(self, project_id):
        ip_list = self.get_ip_list()
        first_login_cmd = 'chage -d 0 hcp;echo exec-result: $?'
        for node_ip in ip_list:
            try:
                self.do_set_first_login(node_ip, first_login_cmd)
            except HCCIException as err:
                if str(err).find("Bad authentication type") != -1:
                    logger.info("node %s has already set first login." % node_ip)
                else:
                    raise err

    def set_admin_first_login(self, project_id):
        if get_project_condition_boolean(project_id, 'eBackup_Proxy'):
            logger.info("Skip set admin first login on eBackup_Proxy condition.")
            return
        ip_list = self.get_ip_list()
        primary_ip_list = self.get_primary_ip(ip_list)
        cmd = "export LD_LIBRARY_PATH=/opt/huawei-data-protection/" \
              "ebackup/db/lib/;su -s /bin/bash gaussdb -c " \
              "\"/opt/huawei-data-protection/ebackup/db/bin/gsql" \
              " -d iamdb -U gaussdb -p 6432" \
              " -c \\\"UPDATE USERS SET PWDSTATUS=1 " \
              "where USERNAME='admin';\\\"\""
        for primary_ip in primary_ip_list:
            sshclient = self.ssh_obj.ssh_create_client(primary_ip, "hcp", self.hcp_user_pwd)
            self.ssh_obj.ssh_send_command(sshclient, 'su', 'Password:', 100)
            self.ssh_obj.ssh_send_command(sshclient, self.root_passwd, '#', 100)
            self.ssh_obj.ssh_send_command(sshclient, cmd, 'Password for user gaussdb', 100)
            result = self.ssh_obj.ssh_send_command(sshclient, self.gaussdb_pwd, '#', 100)
            if str(result).find("UPDATE 1") == -1:
                raise Exception(f"Failed to set admin first login on node {primary_ip}. stdout :{str(result)}.")
            logger.info(f"Success to set admin first login on node {primary_ip}.")

    def get_primary_ip(self, ip_list):
        primary_list = []
        cmd = "service hcp status | grep  AdminNode 2>/dev/null 1>&2" \
              ";echo result: $?"
        for primary_ip in ip_list:
            result = self.ssh_obj.ssh_cmds(primary_ip, cmd, "hcp", self.hcp_user_pwd, self.root_passwd, "", "")
            if str(result).find("result: 0") != -1:
                logger.info("node %s is primary node." % primary_ip)
                primary_list.append(primary_ip)
        if not primary_list:
            raise Exception("Primary ip list is null. set admin first login failed.")
        return primary_list

    def is_connected_server(self, server_ip):
        try:
            result = self.ssh_obj.ssh_cmds(
                server_ip,
                'uname -a',
                "hcp",
                self.hcp_user_pwd,
                self.root_passwd,
                "",
                ""
            )
        except Exception as err:
            logger.info("%s" % err)
            result = ['False']
        return True if "Linux" in result[0] else False

    def get_ip_list(self):
        ip_list = []
        externalom_ip_lower = self.__config_dict['datamover_externalom_iplist'].lower()
        ip_list = ip_list + externalom_ip_lower.split(',')
        return ip_list

    def rollback(self, project_id, pod_id):
        """
        标准调用接口：执行回滚
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        pass

    def retry(self, project_id, pod_id):
        """
        标准调用接口：重试
        :return: Message类对象
        """
        return self.execute(project_id, pod_id)

    def check(self, project_id, pod_id):
        """
        标准调用接口：重试
        :param project_id:
        :param pod_id:
        :return:
        """

# the code has been updated to python3.7
