import os
import time

from utils.common import log as logger
from utils.common.exception import HCCIException
from utils.DBAdapter.DBConnector import BaseOps
from utils.business.dmk_util import DmkApi
from utils.business.param_util import ParamUtil
from utils.business.project_util import ProjectApi

from plugins.CSBS.common.upgrade import models


class ProxyDMKClient(object):
    def __init__(self, project_id, pod_id, regionid_list):
        self.project_id = project_id
        self.pod_id = pod_id
        self.db_api = BaseOps()
        self.region_id = regionid_list[0]
        self.dmk_api = DmkApi()
        self.param_util = ParamUtil()
        self.project_api = ProjectApi()
        self.db_params_dict = self.get_params()
        self.dmk_params = self.get_dmk_params()

    def get_karborproxy_vm_ips(self):
        vm_names = ["CSBS-KarborProxy-01", "CSBS-KarborProxy-02"]
        vm_name_keys = ['karborproxy_host0', 'karborproxy_host1']
        vm_ips = []
        for vm_name in vm_names:
            vm_info = self.param_util.get_vm_info_by_vm_name(self.project_id, vm_name)
            if not vm_info:
                raise Exception(f'Failed to getting the VM information based '
                                'on the VM name, vm name:{vm_name}.')
            vm_ips.append(vm_info[0].get('ip'))
        vm_ip_dict = dict(zip(vm_name_keys, vm_ips))
        return vm_ip_dict

    def get_params(self):
        db_params_dict = {}
        service_items = self.db_api.get_service_cloud_param(self.pod_id, "CSBS-VBS", self.region_id)
        for key, value in service_items.items():
            db_params_dict[key] = value
        dmk_items = self.dmk_api.get_dmk_info(self.project_id)
        for key, value in dmk_items.items():
            db_params_dict[key] = value
        dmk_business_keys = ["DMK_dmk_os_business_username",
                             "DMK_dmk_os_business_user_password"]
        for key in dmk_business_keys:
            val = self.param_util.get_param_value(self.pod_id, "DMK", key)
            if val:
                db_params_dict[key] = val

        db_params_dict.update(self.get_karborproxy_vm_ips())
        return db_params_dict

    def get_dmk_params(self):
        dmk_params = models.DmkParams(
            account=self.db_params_dict.get("dmk_account_proxy", ""),
            tmp_account_pwd=self.db_params_dict.get("tmp_dmk_account_pwd_proxy", ""),
            account_pwd=self.db_params_dict.get("dmk_account_pwd_proxy", ""),
            account_group=self.db_params_dict.get("dmk_account_group_proxy", ""),
            float_ip=self.db_params_dict.get("DMK_dmk_floatIp", ""),
            # Super Administrator Account of DMK Portal, example: sysadmin
            ui_username=self.db_params_dict.get("DMK_dmk_ui_username", ""),
            ui_password=self.db_params_dict.get("DMK_dmk_ui_password", ""),
            # DMK Service Account (dmk/udmk)
            os_business_username=self.db_params_dict.get("DMK_dmk_os_business_username", ""),
            os_business_password=self.db_params_dict.get("DMK_dmk_os_business_user_password", "")
        )
        return dmk_params

    def get_karborproxy_params(self):
        proxy_params = models.ProxyParams(
            dpamanager_account_name=self.db_params_dict.get("karborproxy_dpamanager_username", ""),
            dpamanager_account_pwd=self.db_params_dict.get("karborproxy_dpamanager_pwd", ""),
            root_account_name='root',
            root_account_pwd=self.db_params_dict.get("karborproxy_root_pwd", ""),
            proxy_ip_dict=self.get_karborproxy_vm_ips()
        )
        return proxy_params

    def _login_dmk_by_cbs_account(self):
        self.dmk_api.login_dmk(self.dmk_params.float_ip,
                               self.dmk_params.account,
                               self.dmk_params.account_pwd)

    def execute_dmk_deploy(self, app_name, app_version, deploy_action,
                           hosts_content, vars_content, to_run="true",
                           user="dpamanager"):
        self._login_dmk_by_cbs_account()
        account_id = \
            self.dmk_api.get_dmk_account_id(self.dmk_params.account, user)
        result = \
            self.dmk_api.excute_dmk_deployment(app_name=app_name,
                                               to_run=to_run,
                                               app_version=app_version,
                                               deploy_action=deploy_action,
                                               hosts_content=hosts_content,
                                               vars_content=vars_content,
                                               account_id=account_id)
        return result

    def dmk_deploy_karborproxy(self, action, pkg_version):
        self.db_params_dict['check_karborproxy'] = 'false'
        self.db_params_dict['karborproxy_version'] = pkg_version
        config = ConfigFiles(self.db_params_dict)
        dmk_config_content = config.get_karborproxy_config_files()
        result = \
            self.execute_dmk_deploy(app_name="KarborProxy",
                                    app_version=pkg_version,
                                    deploy_action=action,
                                    hosts_content=dmk_config_content.get("host"),
                                    vars_content=dmk_config_content.get("config"),
                                    user="dpamanager")
        return result

    def upload_pkg_to_dmk(self, pkg_path, pod_id):
        self._login_dmk_by_cbs_account()
        self.dmk_api.upload_pkg_to_dmk(
            self.dmk_params.float_ip,
            self.dmk_params.os_business_username,
            self.dmk_params.os_business_password,
            pkg_path,
            pod_id)

    def _delete_dmk_account_by_account_name(self, user_name, account_name):
        while True:
            try:
                account_id = self.dmk_api.get_dmk_account_id(user_name, account_name)
            except HCCIException as err:
                logger.warn(f"The {account_name} account does not exist, err_msg: {str(err)}.")
                return
            except Exception as err:
                logger.warn(f"get account failed, err_msg: {str(err)}.")
                return
            if account_id:
                self.dmk_api.delete_dmk_account(account_id)
                time.sleep(1)

    def create_cbs_account_for_upgrade(self):
        # login by admin account and create new account
        self.dmk_api.login_dmk(self.dmk_params.float_ip,
                               self.dmk_params.ui_username,
                               self.dmk_params.ui_password,
                               dmk_port='8443')
        role = '2'
        user_info = self.dmk_api.create_dmk_user(
            self.dmk_params.account,
            self.dmk_params.tmp_account_pwd,
            role,
            self.dmk_params.account_group)
        user_id = user_info[1]
        team_id = user_info[0]

        # login by new account and change the original password
        self.dmk_api.login_dmk(self.dmk_params.float_ip,
                               self.dmk_params.account,
                               self.dmk_params.tmp_account_pwd)

        self.dmk_api.change_dmk_user_passwd(user_id,
                                            self.dmk_params.tmp_account_pwd,
                                            self.dmk_params.account_pwd)

        # login dmk by new password
        self._login_dmk_by_cbs_account()
        karborproxy_params = self.get_karborproxy_params()
        # delete dmk account before create account
        self._delete_dmk_account_by_account_name(self.dmk_params.account, karborproxy_params.dpamanager_account_name)
        # create account for karborproxy dpamanager
        self.dmk_api.add_account_to_dmk(karborproxy_params.dpamanager_account_name,
                                        team_id,
                                        karborproxy_params.dpamanager_account_pwd,
                                        karborproxy_params.root_account_pwd)

        # login by admin account, and update user to multi team,
        # so that ICAgent service can be deployed
        team_list = [self.dmk_params.account, 'OTD']
        self.dmk_api.login_dmk(self.dmk_params.float_ip,
                               self.dmk_params.ui_username,
                               self.dmk_params.ui_password)
        result_dmk = self.dmk_api.update_user_to_multi_team(user_id, 2, team_list)
        if not result_dmk:
            logger.error("Failed to update user to multi_team for CBS_UPGRADE.")
            raise HCCIException("640004")
        logger.info("Succeed creating DMK Account.")


class ConfigFiles(object):
    def __init__(self, params_dict):
        self.config_file_dir = \
            f"{os.path.dirname(os.path.realpath(__file__))}/../../conf/upgrade"
        self.config_dict = params_dict
        self.karborproxy_dmk_config_content = {}

    def get_karborproxy_config_files(self):
        if not self.karborproxy_dmk_config_content:
            self.init_karborproxy_config_file()
        return self.karborproxy_dmk_config_content

    def get_config(self, db_dict, file_path):
        """Configuring the DMK Template File

        :param db_dict:The dict type, read from corresponding service
        in the database.Generally is __db_param_dict.
        :param file_path: DMK configuration file template path
        :return:Returns the character string that is automatically filled
        in the DMK configuration file by the dictionary.
        """
        config_str = self.get_config_content(file_path)
        return config_str.format(**db_dict)

    @staticmethod
    def get_config_content(file_path):
        """Reading the DMK Configuration Template

        :param file_path: DMK configuration file template path
        :return:Returns the character string of the DMK configuration file.
        """
        with open(file_path, encoding='utf-8') as file:
            return file.read()

    def init_karborproxy_config_file(self):
        """Rendering the 'console upgrade config&host' files"""
        karborproxy_config = f"{self.config_file_dir}/karborproxy_config"
        karborproxy_host = f"{self.config_file_dir}/karborproxy_host"

        self.karborproxy_dmk_config_content['config'] = self.get_config(self.config_dict, karborproxy_config)
        self.karborproxy_dmk_config_content['host'] = self.get_config_content(karborproxy_host)
