#!/usr/bin/env python
# -*- coding: utf-8 -*-
# Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.

import os

import utils.common.log as logger
from utils.common.message import Message
from utils.common.exception import HCCIException

from plugins.CSBS.common.ssh_client import SshClient
from plugins.CSBS.common.upgrade.karbor import KarborOperation
from plugins.CSBS.common.upgrade.params import ParamsTools
from plugins.CSBS.scripts.upgrade.karbor.base import BaseSubJob

logger.init("CSBS-VBS")


class InstallOSPatch(BaseSubJob):
    def __init__(self, project_id, pod_id, regionid_list):
        super(InstallOSPatch, self).__init__(project_id, pod_id, regionid_list)
        self.ssh_client = SshClient()
        self.param_tool = ParamsTools(self.project_id)
        pkg_path, pkg_name = self.param_tool.get_pkg_path_and_pkg_name(pkg_suffix="_CBS_OS_Patch_X86.zip")
        self.file_path = os.path.join(pkg_path, pkg_name)
        self.patch_path = "/home/djmanager/os_patch"
        self.karbor_operation = KarborOperation(self.project_id)
        self.karbor_client = None

    def execute(self, project_id, pod_id, regionid_list=None):
        logger.info("Start to check os version.")
        result = self._check_is_x86_2r7()
        if not result:
            logger.info("The current os version is not x86_2r7, not need to install the patch.")
            return Message(200)
        try:
            self._install_patch()
        except Exception as err:
            logger.error(f"Failed to install os patch on karbor nodes, err_msg:{err}.")
            return Message(500, HCCIException(645031))
        finally:
            if self.karbor_client:
                self.ssh_client.ssh_close(self.karbor_client)
        logger.info("Succeed to upgrade os patch on karbor nodes.")
        return Message(200)

    def _install_patch(self):
        for karbor_node in self.karbor_operation.karbor_node_list:
            logger.info(f"Start to install patch on karbor node, node ip：{karbor_node.node_ip}.")
            self.karbor_client = self.ssh_client.get_ssh_client(karbor_node)
            self._clear_work_dir()
            self._upload_package(karbor_node)
            self._untar_package()
            if self._check_patch_is_installed():
                logger.info(f"Current node has been installed the patch, node ip: {karbor_node.node_ip}.")
                continue
            self._install_os_patch()
            if self.karbor_client:
                self.ssh_client.ssh_close(self.karbor_client)
                self.karbor_client = None
            logger.info(f"Succeeded to install patch on {karbor_node.node_ip} node.")

    def _exec_cmds_on_karbor_node(self, cmd_list):
        for cmd in cmd_list:
            result = self.ssh_client.ssh_exec_command_return(self.karbor_client, cmd)
            logger.info(result)
            if not self.ssh_client.is_ssh_cmd_executed(result):
                raise Exception(f"Failed to execute the cmd: {cmd}, please check.")

    def _clear_work_dir(self, ):
        cmds = [
            f"rm -rf {self.patch_path}",
            f"mkdir -p {self.patch_path}",
            f"chown -R djmanager:openstack {self.patch_path}"
        ]
        logger.info("Start to create patch path.")
        self._exec_cmds_on_karbor_node(cmds)
        logger.info("Create patch path successfully.")

    def _upload_package(self, karbor_node):
        if not os.path.exists(self.file_path):
            raise Exception("The OS patch package does not exist.")
        self.ssh_client.put(karbor_node.node_ip,
                            karbor_node.user,
                            karbor_node.user_pwd,
                            self.file_path,
                            self.patch_path)
        logger.info(f"Upload patch package to {karbor_node.node_ip} node successfully.")

    def _untar_package(self):
        cmds = [
            f"cd {self.patch_path}",
            "unzip -o OceanStor_BCManager_*_CBS_OS_Patch_X86.zip >/dev/null 2>&1",
            "find euler_patch -name '*.sh' | xargs dos2unix"
        ]
        self._exec_cmds_on_karbor_node(cmds)
        logger.info("Decompress the patch package successfully.")

    def _install_os_patch(self, operate="patch"):
        cmds = [
            f"cd {self.patch_path}/euler_patch",
            f"sh patch.sh {operate}"
        ]
        self._exec_cmds_on_karbor_node(cmds)

    def _check_patch_is_installed(self):
        cmd_0 = f"cd {self.patch_path}/euler_patch"
        exec_result = self.ssh_client.ssh_exec_command_return(self.karbor_client, cmd_0)
        if not self.ssh_client.is_ssh_cmd_executed(exec_result):
            raise Exception(f"Failed to execute the cmd: {cmd_0}, please check.")
        cmd_1 = f"sh patch.sh check"
        exec_result = self.ssh_client.ssh_exec_command_return(self.karbor_client, cmd_1)
        if not self.ssh_client.is_ssh_cmd_executed(exec_result):
            raise Exception(f"Failed to execute the cmd: {cmd_0}, please check.")
        result_str = "::".join(exec_result)
        return "has been installed" in result_str

    def _check_is_x86_2r7(self):
        logger.info("Ssh to karbor node and check os version.")
        exec_result = None
        for node in self.karbor_operation.karbor_node_list:
            karbor_client = self.ssh_client.get_ssh_client(node)
            cmd = 'uname -r | grep -iE "eulerosv2r7.x86" && echo successful'
            exec_result = self.ssh_client.ssh_exec_command_return(karbor_client, cmd)
            if not self.ssh_client.is_ssh_cmd_executed(exec_result):
                logger.warn(f"Failed to get os version info on karbor node, node ip:{node.node_ip}.")
                self.ssh_client.ssh_close(karbor_client)
                continue
            logger.info(f"Succeed to execute command of '{cmd}' on karbor node.")
            self.ssh_client.ssh_close(karbor_client)
            break
        if not exec_result:
            raise Exception("Failed to obtain the CPU architecture from karbor node.")
        result_str = "::".join(exec_result)
        return "successful" in result_str
