#!/usr/bin/env python
# -*- coding: utf-8 -*-
# Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.

import utils.common.log as logger

from utils.common.exception import HCCIException
from utils.common.message import Message

from plugins.CSBS.common.ssh_client import SshClient
from plugins.CSBS.common.upgrade.karbor import KarborOperation
from plugins.CSBS.scripts.upgrade.karbor.base import BaseSubJob

logger.init("CSBS-VBS")


class ClearOSPatch(BaseSubJob):
    def __init__(self, project_id, pod_id, regionid_list):
        super(ClearOSPatch, self).__init__(project_id, pod_id, regionid_list)
        self.ssh_client = SshClient()
        self.patch_path = "/home/djmanager/os_patch"
        self.karbor_operation = KarborOperation(self.project_id)
        self.karbor_client = None

    def execute(self, project_id, pod_id, regionid_list=None):
        logger.info("Start to check os version.")
        result = self._check_is_x86_2r7()
        if not result:
            logger.info("The current os version is not x86_2r7, not need to clear.")
            return Message(200)
        logger.info("Start to clear os patch on karbor nodes.")
        try:
            self._clear_os_patch()
        except Exception as err:
            logger.error(f"Failed to clear os patch file on  karbor nodes, err_msg:{err}.")
            return Message(500, HCCIException(645043))
        finally:
            if self.karbor_client:
                self.ssh_client.ssh_close(self.karbor_client)
        logger.info("Succeed to clear os patch file on karbor nodes.")
        return Message(200)

    def _clear_os_patch(self):
        cmd = f"rm -rf {self.patch_path}"
        for karbor_node in self.karbor_operation.karbor_node_list:
            logger.info(f"Start to clear os patch on karbor node, node ip：{karbor_node.node_ip}.")
            karbor_client = self.ssh_client.get_ssh_client(karbor_node)
            result = self.ssh_client.ssh_exec_command_return(karbor_client, cmd)
            logger.info(result)
            if not self.ssh_client.is_ssh_cmd_executed(result):
                self.ssh_client.ssh_close(karbor_client)
                raise Exception(f"Failed to execute the cmd: {cmd}, please check.")
            if self.karbor_client:
                self.ssh_client.ssh_close(self.karbor_client)
                self.karbor_client = None
            logger.info(f"Succeeded to clear os patch on karbor node, node ip：{karbor_node.node_ip}.")

    def _check_is_x86_2r7(self):
        logger.info("Ssh to karbor node and check os version.")
        exec_result = None
        for node in self.karbor_operation.karbor_node_list:
            karbor_client = self.ssh_client.get_ssh_client(node)
            cmd = 'uname -r | grep -iE "eulerosv2r7.x86" && echo successful'
            exec_result = self.ssh_client.ssh_exec_command_return(karbor_client, cmd)
            if not self.ssh_client.is_ssh_cmd_executed(exec_result):
                logger.warn(f"Failed to get os version info on karbor node, node ip:{node.node_ip}.")
                self.ssh_client.ssh_close(karbor_client)
                continue
            logger.info(f"Succeed to execute command of '{cmd}' on karbor node.")
            self.ssh_client.ssh_close(karbor_client)
            break
        if not exec_result:
            raise Exception("Failed to obtain the CPU architecture from karbor node.")
        result_str = "::".join(exec_result)
        return "successful" in result_str
