#!/usr/bin/python
# coding: utf-8

import utils.common.log as logger
from utils.PasswordManager.PasswordManager import FSPasswordManager
from utils.common.exception import HCCIException
from utils.common.openstack_host import get_opentack_ssh_info_by_pod_id

from plugins.CSBS.common.model import Node
from plugins.CSBS.common.ssh_client import SshClient
from plugins.CSBS.common.util import auto_retry


class CPSInfo(object):
    cps_user = "fsp"

    def __init__(self, pod_id):
        self.pod_id = pod_id
        self.fsp_pwd_manager = FSPasswordManager(self.pod_id)

    def get_cps_ip(self):
        cps_ip = get_opentack_ssh_info_by_pod_id(self.pod_id).get("host_ip")
        if not cps_ip:
            raise HCCIException("640031")
        return cps_ip

    def get_cps_envpwd(self):
        cps_envpwd = self.fsp_pwd_manager.get_env_os_password()
        if not cps_envpwd:
            raise HCCIException("640032")
        return cps_envpwd

    def get_cps_userpwd(self):
        cps_userpwd = self.fsp_pwd_manager.get_fsp_password()
        if not cps_userpwd:
            raise HCCIException("640033")
        return cps_userpwd

    def get_cps_rootpwd(self):
        cps_rootpwd = self.fsp_pwd_manager.get_root_password()
        if not cps_rootpwd:
            raise HCCIException("640034")
        return cps_rootpwd

    def get_cps_node(self):
        return Node(self.get_cps_ip(),
                    CPSInfo.cps_user,
                    self.get_cps_userpwd(),
                    'root',
                    self.get_cps_rootpwd(),
                    self.get_cps_envpwd())


class CpsHelper(object):
    def __init__(self, node):
        self.node = node

    def get_cps_client(self, ssh_client):
        logger.info('Get cps ssh client, node:{}'.format(self.node.node_ip))
        cps_client = ssh_client.get_ssh_client(self.node, sudo_type="su")
        cmds = "source set_env"
        expect_str = "please choose:[1|2"
        ssh_client.ssh_send_command_expect(cps_client, cmds, expect_str)
        cmds = "1"
        expect_str = "Enter OS_PASSWORD="
        ssh_client.ssh_send_command_expect(cps_client, cmds, expect_str)
        ssh_client.ssh_send_command_expect(cps_client, self.node.extra)
        return cps_client

    def exec_cps_cmds(self, cmds, ret_success=False):
        ssh_client = SshClient()
        cps_client = self.get_cps_client(ssh_client)
        result = ssh_client.ssh_exec_command_return(cps_client, cmds)
        ssh_client.ssh_close(cps_client)
        if not ret_success:
            return result
        logger.info('Cmd result:{}'.format(result))
        return ssh_client.is_ssh_cmd_executed(result)

    @auto_retry(max_retry_times=3, delay_time=60)
    def get_nova_endpoint(self):
        nova_cmd = "openstack endpoint list" \
                   "|grep $OS_REGION_NAME " \
                   "|grep nova|grep admin" \
                   "|sed 's/ //g'" \
                   "|sed 's/\/$(tenant_id)s//g'"
        logger.info(f'Get nova endpoint, cmd:{nova_cmd}.')
        result_list = self.exec_cps_cmds(nova_cmd)
        logger.info(f'The result:{result_list}.')
        return self._get_endpoint(result_list, "nova")

    @auto_retry(max_retry_times=3, delay_time=60)
    def get_cinder_endpoint(self):
        cinder_cmd = "openstack endpoint list" \
                     "|grep $OS_REGION_NAME " \
                     "|grep cinder|grep admin" \
                     "|sed 's/ //g'" \
                     "|sed 's/\/$(tenant_id)s//g'"
        logger.info(f'Get cinder endpoint, cmd:{cinder_cmd}.')
        result_list = self.exec_cps_cmds(cinder_cmd)
        logger.info(f'The result:{result_list}.')
        return self._get_endpoint(result_list, "cinder")

    @auto_retry(max_retry_times=3, delay_time=60)
    def get_ceilometer_endpoint(self):
        ceilometer_cmd = "openstack endpoint list" \
                         "|grep $OS_REGION_NAME " \
                         "|grep ceilometer" \
                         "|grep admin" \
                         "|sed 's/ //g'" \
                         "|sed 's/\/$(tenant_id)s//g'"
        logger.info(f'Get ceilometer endpoint, cmd:{ceilometer_cmd}.')
        result_list = self.exec_cps_cmds(ceilometer_cmd)
        logger.info(f'The result:{result_list}.')
        return self._get_endpoint(result_list, "ceilometer")

    @auto_retry(max_retry_times=3, delay_time=60)
    def get_glance_endpoint(self):
        glance_cmd = "openstack endpoint list" \
                     "|grep $OS_REGION_NAME " \
                     "|grep glance" \
                     "|grep admin" \
                     "|sed 's/ //g'" \
                     "|sed 's/\/$(tenant_id)s//g'"
        logger.info(f'Get glance endpoint, cmd:{glance_cmd}.')
        result_list = self.exec_cps_cmds(glance_cmd)
        logger.info(f'The result:{result_list}.')
        return self._get_endpoint(result_list, "glance")

    @auto_retry(max_retry_times=3, delay_time=60)
    def get_neutron_endpoint(self):
        neutron_cmd = "openstack endpoint list" \
                      "|grep $OS_REGION_NAME " \
                      "|grep neutron" \
                      "|grep admin" \
                      "|sed 's/ //g'" \
                      "|sed 's/\/$(tenant_id)s//g'"
        logger.info(f'Get neutron endpoint, cmd:{neutron_cmd}.')
        result_list = self.exec_cps_cmds(neutron_cmd)
        logger.info(f'The result:{result_list}.')
        return self._get_endpoint(result_list, "neutron")

    def get_keystone_endpoint(self):
        keystone_cmd = "echo $OS_AUTH_URL"
        logger.info('Get keystone endpoint, cmd:{}'.format(keystone_cmd))
        result_dict = self.exec_cps_cmds(keystone_cmd)
        logger.info('result:{}'.format(result_dict))
        if len(result_dict) != 3 or "https://" not in result_dict[1]:
            return ""
        return result_dict[1]

    def get_portid_macaddr(self, ip_address):
        cmds = "neutron port-list|grep \\\"%s\\\"|sed 's/ //g'" % ip_address
        logger.info('Get portid macaddr, cmd:{}'.format(cmds))
        result_list = self.exec_cps_cmds(cmds)
        logger.info('result:{}'.format(result_list))
        if len(result_list) == 0:
            return dict()
        # 查询结果可能出现其他警告信息, 导致解析出错,增加包含'|'过滤
        find_list = [_ for _ in result_list if _.__contains__('|')]
        logger.info('Find data list:{}'.format(find_list))
        if not find_list:
            logger.error('Return data error, errMsg: no "|" in data to split')
            return dict()
        port_list = [_ for _ in find_list if _.__contains__('ip_address')]
        port_result = port_list[0].split("|")
        if len(port_result) != 6:
            logger.error('Length of splited port result is not 6, '
                         'port result:{}'.format(port_result))
            return dict()
        return {"portid": port_result[1], "macaddr": port_result[3]}

    def allowed_address_pairs(self, ip_address, float_ip1, float_ip2):
        param = self.get_portid_macaddr(ip_address)
        if not param:
            return False
        update_cmd = "neutron port-update %s --allowed-address-pairs " \
                     "type=dict list=true ip_address=%s,mac_address=%s " \
                     "ip_address=%s,mac_address=%s" % (
                         param.get('portid'), float_ip1, param.get('macaddr'),
                         float_ip2,
                         param.get('macaddr'))
        logger.info('Allowed address pairs, cmd:{}'.format(update_cmd))
        return self.exec_cps_cmds(update_cmd, True)

    def set_backup_quota(self):
        backup_quota = "cps template-params-update " \
                       "--parameter use_default_quota_class=false " \
                       "--service cinder cinder-api; " \
                       "cps template-params-update " \
                       "--service cinder cinder-api " \
                       "--parameter quota_gigabytes=-1 " \
                       "quota_snapshots=-1 " \
                       "quota_backup_gigabytes=-1 " \
                       "quota_backups=-1 " \
                       "quota_volumes=-1;" \
                       "cps commit"
        logger.info('Set backup quota, cmd:{}'.format(backup_quota))
        return self.exec_cps_cmds(backup_quota, True)

    @staticmethod
    def _get_endpoint(parse_list, service_name):
        _op_endpoint_cli_output_line_list_len = 9
        _op_endpoint_cli_output_service_name_index = 3
        _op_endpoint_cli_output_url_index = 7
        endpoint = None
        for one_result in parse_list:
            tar_list = one_result.split("|")
            if len(tar_list) == _op_endpoint_cli_output_line_list_len and \
                    tar_list[_op_endpoint_cli_output_service_name_index] == service_name:
                endpoint = tar_list[_op_endpoint_cli_output_url_index]
                break
        if not endpoint:
            raise Exception(f"Failed to obtain {service_name} endpoint.")
        logger.info(f"Succeeded to get {service_name} endpoint, the endpoint:{endpoint}.")
        return endpoint
