/*
 * Copyright (c) Huawei Technologies Co., Ltd. 2020-2022. All rights reserved.
 */
// 树组件
function Tree(setting, dataSource) {
  this.setting = setting;
  this.dataSource = dataSource;
  this.ItemIdChildrenMap = new Map();
  this.ItemIdItemMap = new Map();
  this.SetItemChildrenMap(dataSource);
  this.Render();
}

function escapeHtml(str) {
  if (typeof str == 'undefined') {
    return '';
  } else if (!str || typeof str != 'string') {
    return str;
  }
  let result = str;
  result = result.replace(/&/g, '&amp;');
  result = result.replace(/\"/g, '&quot;');
  result = result.replace(/'/g, '&apos;');
  result = result.replace(/</g, '&lt;');
  result = result.replace(/>/g, '&gt;');
  result = result.replace(/\\t/g, '&nbsp;&nbsp;');
  result = result.replace(/\\r\\n/g, '<br/>');
  return result;
}

Tree.prototype = {
  // 渲染树组件
  Render: function() {
    const dataSource = this.dataSource;
    const container = this.setting.container;
    container.classList.add('tree');
    this.BuildRootItem(dataSource, container);
  },

  BuildRootItem: function(data, container) {
    if (Array.isArray(data) && data.length) {
      const ul = document.createElement('ul');
      data.forEach(
        function(item) {
          if (!item.isHidden) {
            const liStr =
              '<li class="level_0">' +
              '<span class="button button-close" treenode_switch data-id="' +
              escapeHtml(item.id) +
              '"></span>' +
              '<a class="level_0" title="' +
              escapeHtml(item.name) +
              '" data-id="' +
              escapeHtml(item.id) +
              '">' +
              '<span class="icon"></span>' +
              '<span class="node_name">' +
              escapeHtml(item.name) +
              '</span>' +
              '</a>' +
              '</li>';
            const itemUi = this.CreateElementByStr(liStr);
            ul.appendChild(itemUi);
            const nodeItem = this.getNodeByTId(item.id);
            nodeItem.itemUi = itemUi;
            nodeItem.level = 0;
            // 绑定点击事件
            this.AttachNodeClickEvent(nodeItem);

            this.AttachNodeDblClickEvent(nodeItem);

            this.AttachNodeExpendEvent(nodeItem);

            this.UpdateIcon(nodeItem);
          }
        }.bind(this)
      );
      container.appendChild(ul);
    }
  },

  AttachNodeExpendEvent: function(node) {
    const switchNode = node.itemUi.querySelector('span[treenode_switch]');
    if (switchNode) {
      switchNode.removeEventListener('click', this.ExpandClickHandler);
      switchNode.addEventListener('click', this.ExpandClickHandler.bind(this, switchNode), false);
    }
  },

  ExpandClickHandler: function(expandBtn) {
    const itemId = expandBtn.getAttribute('data-id');
    const nodeItem = this.getNodeByTId(itemId);
    if (!nodeItem.isExpend) {
      this.ExpandItem(nodeItem);
    } else {
      this.CloseItem(nodeItem);
    }
    this.UpdateIcon(nodeItem);
  },

  AttachNodeDblClickEvent: function(item) {
    const aTag = item.itemUi.getElementsByTagName('a')[0];
    if (aTag) {
      aTag.removeEventListener('dblclick', this.ExpandClickHandler);
      aTag.addEventListener('dblclick', this.ExpandClickHandler.bind(this, aTag), false);
    }
  },

  AttachNodeClickEvent: function(item) {
    const aTag = item.itemUi.getElementsByTagName('a')[0];
    if (aTag) {
      aTag.removeEventListener('click', this.NodeClickHandler);
      aTag.addEventListener('click', this.NodeClickHandler.bind(this, aTag), false);
    }
  },

  NodeClickHandler: function(aTag) {
    const itemId = aTag.getAttribute('data-id');
    const nodeItem = this.getNodeByTId(itemId);
    this.selectNode(nodeItem);
    if (
      this.setting.callback &&
      this.setting.callback.beforeClick &&
      Object.prototype.toString.call(this.setting.callback.beforeClick) === '[object Function]'
    ) {
      this.setting.callback.beforeClick.call(this, nodeItem);
    }
  },

  // 更新节点图标状态
  UpdateIcon: function(node) {
    const iconNode = node.itemUi.getElementsByClassName('icon')[0];
    const buttonNode = node.itemUi.getElementsByClassName('button')[0];
    if (!node.children) {
      iconNode.classList.add('icon-nochild');
      buttonNode.classList.add('button-nochild');
    } else {
      iconNode.classList.remove('icon-nochild');
      if (node.isExpend) {
        iconNode.classList.remove('icon-close');
        iconNode.classList.add('icon-open');

        buttonNode.classList.remove('button-close');
        buttonNode.classList.add('button-open');
      } else {
        iconNode.classList.remove('icon-open');
        iconNode.classList.add('icon-close');

        buttonNode.classList.remove('button-open');
        buttonNode.classList.add('button-close');
      }
    }
  },

  // 构建id-children Map

  SetItemChildrenMap: function(dataSource) {
    const self = this;
    dataSource.forEach(function(item) {
      self.ItemIdItemMap.set(item.id, item);
      if (item.children) {
        self.ItemIdChildrenMap.set(item.id, item.children);
        self.SetItemChildrenMap(item.children);
      }
    });
  },

  // 展开节点
  ExpandItem: function(item) {
    if (item.isExpend || !item.children || !item.children.length) {
      return;
    }
    const childrenUi = item.itemUi.childNodes[2];
    if (childrenUi) {
      childrenUi.style.display = 'block';
      item.isExpend = true;
    } else {
      const itemChildList = item.children;
      const childLevel = item.level + 1;
      if (Array.isArray(itemChildList)) {
        const ul = document.createElement('ul');
        itemChildList.forEach(
          function(node) {
            if (!node.isHidden) {
              const liStr =
                '<li class="level_' +
                childLevel +
                '">' +
                '<span class="button" treenode_switch data-id="' +
                escapeHtml(node.id) +
                '"></span>' +
                '<a class="level_' +
                childLevel +
                '" title="' +
                escapeHtml(node.name) +
                '" data-id="' +
                escapeHtml(node.id) +
                '">' +
                '<span class="icon"></span>' +
                '<span class="node_name">' +
                escapeHtml(node.name) +
                '</span>' +
                '</a>' +
                '</li>';
              const itemUi = this.CreateElementByStr(liStr);
              ul.appendChild(itemUi);
              const nodeItem = this.getNodeByTId(node.id);
              nodeItem.itemUi = itemUi;
              nodeItem.level = childLevel;
              // 绑定点击事件
              this.AttachNodeClickEvent(nodeItem);

              this.AttachNodeDblClickEvent(nodeItem);

              this.AttachNodeExpendEvent(nodeItem);

              this.UpdateIcon(nodeItem);
            }
          }.bind(this)
        );
        item.itemUi.appendChild(ul);
        item.isExpend = true;
      }
    }
  },

  // 折叠节点
  CloseItem: function(item) {
    if (item.isExpend) {
      const childrenUi = item.itemUi.childNodes[2];
      if (childrenUi) {
        childrenUi.style.display = 'none';
        item.isExpend = false;
      }
    }
  },

  // 选中一个节点
  selectNode: function(node) {
    this.CancelSelectNodes();
    const nodeList = this.FindParentExpandList(node);
    for (let i = 0; i < nodeList.length; i++) {
      this.ExpandItem(nodeList[i]);
    }
    // dom加载过了
    const aTag = node.itemUi.getElementsByTagName('a')[0];
    if (aTag) {
      aTag.classList.add('a_selected');
    }
    this.ScrollToView(node);
  },

  /**
   * 向上寻找父节点至根节点
   * @param node
   * @returns {[]}
   * @constructor
   */
  FindParentExpandList: function(node) {
    const list = [];
    while (node.level !== 0) {
      const parentNode = this.getNodeByTId(node.parentId);
      list.unshift(parentNode);
      node = parentNode;
    }
    return list;
  },

  // 取消所有的选中
  CancelSelectNodes: function() {
    const nodeList = this.getNodes();
    for (let i = 0; i < nodeList.length; i++) {
      const curUi = nodeList[i].itemUi;
      if (curUi) {
        curUi.getElementsByTagName('a')[0].classList.remove('a_selected');
      }
    }
  },

  ScrollToView: function(node) {
    if (!node.itemUi) {
      return;
    }
    const nodeRect = node.itemUi.getBoundingClientRect();
    const containerRect = this.setting.container.getBoundingClientRect();

    const topGap = nodeRect.top - containerRect.top;
    const bottomGap = nodeRect.bottom - containerRect.bottom;

    if (topGap < 0) {
      this.setting.container.scrollTop += topGap - 34;
    } else if (bottomGap > 0) {
      this.setting.container.scrollTop += bottomGap + 34;
    }
  },

  getNodeByTId: function(id) {
    return this.ItemIdItemMap.get(id);
  },

  // 取到所有节点，包括没挂载dom的
  getNodes: function() {
    const nodeList = [];
    for (const key in this.ItemIdItemMap.data) {
      nodeList.push(this.ItemIdItemMap.data[key]);
    }
    return nodeList;
  },

  GetChildrenById: function(id) {
    return this.ItemIdChildrenMap.get(id);
  },

  /**
   * 1、如无结果，返回 null
   * 2、如有多个节点满足查询条件，只返回第一个匹配到的节点
   * @param attr
   * @param value
   */
  getNodeByParam: function(attr, value) {
    const nodeList = this.getNodes();
    for (let i = 0; i < nodeList.length; i++) {
      if (nodeList[i][attr] === value) {
        return nodeList[i];
      }
    }
    return null;
  },

  /**
   * 根据节点数据的属性搜索，获取条件模糊匹配的节点数据 JSON 对象集合
   * @param attr
   * @param value
   */
  getNodesByParamFuzzy: function(attr, value) {
    const nodeList = this.getNodes();
    return nodeList.filter(function(node) {
      return node[attr].indexOf(value) > -1;
    });
  },

  CreateElementByStr: function(str) {
    const tempDiv = document.createElement('div');
    tempDiv.innerHTML = str;
    const element = tempDiv.firstElementChild.cloneNode(true);
    return element;
  }
};
