import datetime

from utils.business.param_util import ParamUtil
from utils.business.project_util import ProjectApi
from utils.common import param_check as paramCheckUnit
from utils.common.error.hcci_error_code import get_code_msg
from utils.common.exception import HCCIException
import utils.common.log as logger
from utils.common.ssh_util import Ssh

from plugins.eBackup.common.param_tool import ParamTool
from plugins.eBackup.common.api_adapter import API
from plugins.eBackup.common.ebackup_rest import EbackupRest
from plugins.eBackup.scripts.common.ebackup_util import CommonTools

# 命令返回的标准长度
SSH_RET_LENGTH = 3
EXCEPT_CODE = -3
ERROR_CODE = -2
COMMON_CODE = 0
# 最大天数
EXPIRING_DAY = 7


class EBackupNode:
    def __init__(self, pod_id, project_id):
        self.params = ParamUtil()
        self.ssh_obj = Ssh()
        self.hcp_name = "hcp"
        self.project_id = project_id
        self.pod_id = pod_id
        self.error_infos = []
        self.param_tool = ParamTool(self.project_id, self.pod_id)
        self.datamover_ip_list = CommonTools(project_id, pod_id).get_datamover_iplist()

    def is_valid_ip(self, node_ip):
        if ProjectApi().is_ipv6_project(self.project_id):
            ip_version = 6
        else:
            ip_version = 4
        return paramCheckUnit.check_param_ip(node_ip, ip_version)

    def get_account_info(self, float_ip, sub_component_name='eBackupServer',
                         externalom_ips=''):
        accounts = API.get_account_info(
            self.project_id, self.pod_id, sub_component_name)
        if not accounts.get(float_ip) and externalom_ips:
            externalom_ips = externalom_ips.split(',')
            float_ip = externalom_ips[0]

        account_info = accounts.get(float_ip)
        if not account_info:
            raise HCCIException(653091, float_ip)

        account_info['float_ip'] = float_ip
        return account_info

    def get_node_info(self, node_ip, sub_component_name):
        accounts = API.get_account_info(
            self.project_id, self.pod_id, sub_component_name)
        account_info = accounts.get(node_ip)
        if not account_info:
            raise HCCIException(653091, node_ip)
        logger.info(
            'Information about the %s node is successfully obtained.' % node_ip
        )
        return account_info

    def check_nodes_password_infos(self):
        if self.param_tool.is_install_csbs():
            logger.info('This is a new deployment and does not require password verification.')
            return
        logger.info('Password consistency check started.')
        try:
            ebackup_server = API.get_account_info(self.project_id, self.pod_id, 'eBackupServer')
        except Exception as err:
            logger.error(err)
            return
        self.check_nodes_password_unifications(ebackup_server)
        self.check_nodes_password_expired(ebackup_server)

    def unified_error_reporting(self):
        msg = '\n'
        if self.error_infos:
            for item in self.error_infos:
                if len(item) == 1:
                    msg = msg + get_code_msg(item[0]) + '\n'
                if len(item) == 2:
                    msg = msg + get_code_msg(item[0]) % item[1] + '\n'
                if len(item) == 3:
                    msg = msg + get_code_msg(item[0]) % (item[1], item[2]) + '\n'
            raise Exception(msg)

    def check_nodes_password_unifications(self, ebackup_server):
        hcp_pwd, root_pwd = None, None
        error_info = 'eBackup Unified Account Passwords Are Inconsistent'
        for key, item in ebackup_server.items():
            if not hcp_pwd:
                hcp_pwd = item.get('hcp')
            if not root_pwd:
                root_pwd = item.get('root')

            if hcp_pwd and item.get('hcp') and item.get('hcp') != hcp_pwd:
                logger.error(error_info)
                self.error_infos.append((653105, key, 'hcp'))

            if item.get('root') and item.get('root') != root_pwd:
                logger.error(error_info)
                self.error_infos.append((653105, key, 'root'))

    def check_nodes_password_expired(self, ebackup_info):
        for node_ip, account_info in ebackup_info.items():
            if node_ip in self.datamover_ip_list:
                logger.info("The node is to be reinstalled, no need to check the login of the node.")
                continue
            if not self.is_valid_ip(node_ip):
                continue
            hcp_pwd = account_info.get('hcp')
            root_pwd = account_info.get('root')
            admin_pwd = account_info.get('admin')
            if admin_pwd:
                headers = self.login(account_info.get('admin'), node_ip)
                if headers:
                    self.check_permit_for_admin(headers, node_ip)
            if not hcp_pwd or not root_pwd:
                continue
            self.check_node_password_expired(node_ip, account_info)

    def check_node_password_expired(self, node_ip, account_info, user='hcp', root='root'):
        try:
            return_data = self.check_os_password_expired_day(account_info, node_ip, [user, root])
        except Exception as err:
            logger.error(err)
            self.error_infos.append((653092, node_ip, user))
            return
        hcp_expire_day = return_data.get(user)
        root_expire_day = return_data.get(root)
        if hcp_expire_day != 'never' and hcp_expire_day <= 7:
            self.error_infos.append((653093, node_ip, user))
        if root_expire_day != 'never' and root_expire_day <= 7:
            self.error_infos.append((653106, node_ip))
        logger.info('Node %s Passes the Password Expiration Check ' % node_ip)

    def check_permit_for_admin(self, headers, float_ip):
        logger.info("begin to check permit of admin account.")
        rest_api = EbackupRest()
        base_url = 'https://' + float_ip + ':8088/rest/dev'
        check_url = base_url + '/vbackup_server'
        rsp = rest_api.get(check_url, headers)
        if rsp.status_code != 200:
            logger.error("rest req return not 200.")
            self.error_infos.append((653020,))
        errcode = rsp.json()["error"]["code"]
        if errcode is None:
            logger.error("errcode is empty.")
            self.error_infos.append((653020,))
        if errcode == 1610664355:
            logger.error("admin account is abnormal.")
            self.error_infos.append((653099, float_ip))
        elif errcode != 0:
            description = rsp.json()["error"]["description"]
            logger.error("login failed,description:" + description)
            ip_dec = 'The IP address is %s ;' % float_ip
            self.error_infos.append((653022, ip_dec))
        else:
            logger.info("check admin account succ.")

    def login(self, pwd, float_ip):
        logger.info("begin to login ebackup.")
        rest_api = EbackupRest()
        base_url = 'https://' + float_ip + ':8088/rest/dev'
        login_url = base_url + '/login'
        headers = {'Accept': 'application/json'}
        data = '{"scope":0,"username":"admin","password":"' + pwd + '"}'
        rsp = None
        try:
            rsp = rest_api.post(login_url, headers, data)
        except Exception as err:
            logger.error("Failed to login eBackup(%s) with admin, reason: %s." % (float_ip, str(err)))
            self.error_infos.append((653100, float_ip))
        if not rsp:
            return {}
        if rsp.status_code != 200:
            logger.error("rest req return not 200.")
            self.error_infos.append((653020,))
        errcode = rsp.json()["error"]["code"]
        if errcode == 0:
            logger.info("login ebackup succ.")
            token = rsp.json()["data"]["iBaseToken"]
            cookie = rsp.headers.get('Set-Cookie')
            session_content = cookie.split(';')[0]
            headers = {
                'Accept': 'application/json;version=2.2;charset=UTF-8',
                'iBaseToken': token,
                'Cookie':
                    'language=en;' + session_content + '; '
                                                       'DEVICE_ID=dev; sessionIdleTime=60000; MACHINEROLE=0;'
                                                       ' CSRF_IBASE_TOKEN=' + token}
            return headers
        else:
            description = rsp.json()["error"]["description"]
            ip_dec = 'The IP address is %s ;' % float_ip
            description = ip_dec + description
            logger.error("login failed,description:" + description)
            self.error_infos.append((653022, description))
            return {}

    @staticmethod
    def get_server_ssh_client(host, ssh_user=None, ssh_pwd=None,
                              sudo_user=None, sudo_pwd=None):
        ssh_ins = Ssh()
        if ssh_user and ssh_pwd:
            ssh_client = ssh_ins.ssh_create_client(host, ssh_user, ssh_pwd, port=22, timeout=60)
            if sudo_user and sudo_pwd:
                ssh_ins.ssh_send_command(ssh_client, "su - {}".format(sudo_user), "Password", 20)
                ssh_ins.ssh_send_command(ssh_client, sudo_pwd, "#", 20)
                ssh_ins.ssh_send_command(ssh_client, "TMOUT=0", "#", 20)
        else:
            logger.error("The user name or password required for "
                         "login is missing.")
            ssh_client = None
        return ssh_client

    def check_os_password_expired_day(self, account_info, node_ip, user_lst):
        ssh_client = self.get_server_ssh_client(
            node_ip, ssh_user=user_lst[0], ssh_pwd=account_info.get(user_lst[0]),
            sudo_user='root', sudo_pwd=account_info.get(user_lst[1]))
        return_data = {}
        for user in user_lst:
            cmd = "chage -l {}".format(user)
            output = Ssh.ssh_send_command(ssh_client, cmd, "#", 20)
            output.pop(0)
            output.pop(-1)
            logger.info(
                "Execute command success, result: {}.".format(output))
            if not output or len(output) < SSH_RET_LENGTH:
                err_msg = "Execution command did not get the expected " \
                          "output, please check."
                logger.error(err_msg)
                raise Exception(err_msg)
            code, expires, inactive = self.get_password_expires(
                "".join(output))
            logger.info(
                "Ended to check {} user, result: {}, {}, {}.".format(
                    user, code, expires, inactive))
            if code == ERROR_CODE:
                err_msg = "Execution command did not get the expected " \
                          "output, please check."
                logger.error(err_msg)
                raise Exception(err_msg)
            elif code == EXCEPT_CODE:
                err_msg = \
                    "Analysis command output error, please try again."
                logger.error(err_msg)
                raise Exception(err_msg)
            elif code == 0:
                if expires == -1:
                    expire_days = 'never'
                    logger.info(f"The password of the user {user} never expire.")
                else:
                    expire_days = (expires - datetime.datetime.utcnow()).days
                    logger.info(f"The password of the user {user} expires after {expire_days} days.")
                return_data[user] = expire_days
        Ssh.ssh_close(ssh_client)
        return return_data

    def get_password_expires(self, output_list):
        """
        获取密码的过期时长和失效时间
        :param output_list: SSH命令返回结果
        :return:
        """
        # 对输出进行分行处理
        expired_time, inactive_time = "", ""
        outputs = output_list.splitlines(False)
        if len(output_list) < SSH_RET_LENGTH:
            # 执行命令结果长度不够，错误
            return ERROR_CODE, "", ""
        try:
            for out_put_line in outputs:
                tmp_expired_time, tmp_inactive_time = self.handle_chage_string(out_put_line)
                if tmp_expired_time:
                    expired_time = tmp_expired_time
                if tmp_inactive_time:
                    inactive_time = tmp_inactive_time
                if expired_time and inactive_time:
                    break
        except Exception as error:
            # 执行命令结果解析，错误
            logger.error(f"Failed to parse ssh command results. "
                         f"Error Message: {error}")
            return EXCEPT_CODE, "", ""
        ret_code = COMMON_CODE
        if (not expired_time) or (not inactive_time):
            ret_code = ERROR_CODE
        return ret_code, expired_time, inactive_time

    @staticmethod
    def handle_chage_string(out_put_line):
        expired_time = inactive_time = None
        split_line = out_put_line.split(":")
        if len(split_line) < 2:
            return expired_time, inactive_time
        chage_name = split_line[0].strip()
        chage_string = split_line[1].strip()
        if chage_name.lower() == 'password expires':
            if chage_string.lower() == 'never':
                expired_time = -1
            else:
                expired_time = datetime.datetime.strptime(chage_string, '%b %d, %Y')
        elif chage_name.lower() == 'password inactive':
            if chage_string.lower() == 'never':
                inactive_time = -1
            else:
                inactive_time = datetime.datetime.strptime(chage_string, '%b %d, %Y')
        return expired_time, inactive_time
