# -*- coding:utf-8 -*-
import json
import os
import re

import time
import utils.common.log as logger
from utils.business.iam_util import IamApi
from utils.business.manageone_cmdb_util import ManageOneCmdbUtil
from utils.business.manageone_util2 import ManageOneUtil2
from utils.business.param_util import ParamUtil
from utils.business.unite_password.unite_pwd_api import UnitePwdApi
from utils.common.exception import HCCIException
from utils.common.fic_base import StepBaseInterface
from utils.common.message import Message
from utils.common.software_package_util import find_software_package_by_name
from utils.common.ssh_util import Ssh

from plugins.eBackup.common.constant import ONE_DAY_IN_SEC
from plugins.eBackup.common.ebackup_rest import EbackupRest
from plugins.eBackup.common.ebackup_util import EBackupUtil
from plugins.eBackup.common.iam_util import IamUtil
from plugins.eBackup.common.util import Utils, AccountTool
from plugins.eBackup.common.model import SshInfo
from plugins.eBackup.scripts.common.ebackup_util import check_compress_file
from plugins.eBackup.scripts.cipher_change.ebackup_cipher_change import EbackupCipherChanger
from plugins.eBackup.scripts.upgrade.util import CommonConfig
from plugins.eBackup.scripts.upgrade.util import ManageOneOcApi

CLEAR_HISTORY_CMD = ">~/.bash_history&&history -c"
IAM_ACCOUNT_DESC = '{"zh-cn":"eBackup服务使用的IAM账号。",' \
                   '"en-us":"The IAM account for eBackup service."}'
ADMIN_ACCOUNT_DESC = '{"zh-cn":"用于登录GUI和CLI，具有超级管理员权限。",' \
                     '"en-us":"Used to log in to GUI and CLI"}'
REST_URI_GOV_UNIFYPWD_MODIY = "/v1/srv_admin/modifypwd"
REST_URI_GOV_UNIFYPWD_RESET = "/v1/srv_admin/resetpwd"
REST_URI_GOV_UNIFYPWD_QUERYJOB = "/v1/srv_admin/querypwdstat"
REST_URI_GOV_UNIFYPWD_VERIFY = "/v1/srv_admin/verifypwd"
RETRY_TIME_LIMIT = 5
CMD_OUTPUT_LENGTH = 2
SSH_RET_LENGTH = 7
EXCEPT_CODE = -3
ERROR_CODE = -2

INSTALL_OS_CIPHER_PATCH_CMD = 'uname -r | grep -E "eulerosv2r9.aarch64|' \
                              'eulerosv2r8.aarch64" && ' \
                              '(grep "patch_success" /home/hcp/patch_result ' \
                              '|| ((cd /home/hcp; tar xvf ' \
                              "OceanStor\\ BCManager\\ *_EulerOS_Patch_*.tar" \
                              '.gz) && (sh /home/hcp/euler_patch/secret_' \
                              'rpm_patch.sh patch && (echo patch_success' \
                              ' > /home/hcp/patch_result))))' \
                              '; echo "last cmd execute res: $?"'
# 允许传入的命令正则表达式
CMD_ALLOW_PATTERN = re.compile("^[:a-zA-Z0-9/=. ,_-]+$")


class ConfigeBackup(StepBaseInterface):
    def __init__(self, project_id, pod_id, regionid_list=None):
        super(ConfigeBackup, self).__init__(project_id, pod_id, regionid_list)
        self.ebk_iam_headers = None
        self.project_id = project_id
        self.pod_id = pod_id
        self.regionid_list = regionid_list
        self.iam_api = IamApi()
        self.param_dict = Utils.init_system_params(project_id, regionid_list[0])
        self.ssh_obj = None
        self.ebackup_ips = []
        self.hcp_password = self.param_dict["eBackup_hcp_pwd"]
        self.root_password = self.param_dict["eBackup_root_pwd"]
        self.iam_util = IamUtil(self.project_id, self.pod_id)
        self.common_config = CommonConfig(self.param_dict, pod_id=self.pod_id, project_id=project_id)
        self.manageone = ManageOneUtil2()
        self.params = ParamUtil()
        self.config_dict = self.params.get_service_cloud_param(pod_id, "eBackup")
        self.az_ip_list = self.param_dict["eBackup_Datamover_nodes"].split('|')
        self.ips = [{'datamover_ips': x.split(';')} for x in self.az_ip_list]
        self.rest_api = EbackupRest()

    def execute(self, project_id, pod_id, regionid_list=None):
        def do_execute():
            self.ssh_obj = Ssh()
            service_system_info = self.get_ebackup_service_system_info()
            self.report_service_info2_manageone(service_system_info)
            self.modify_alarm_region_conf(project_id)
            self.modify_ntp_conf()
            self.install_pacific_api_package()
            for ip_dict in self.ips:
                datamover_ips = ip_dict.get('datamover_ips')
                datamover_ssh_info = SshInfo(datamover_ips, "hcp", self.hcp_password, self.root_password)
                management_float_ip = Utils.find_float_ip(datamover_ssh_info)
                ip_dict.update({'management_float_ip': management_float_ip})
                self.config_op_svc_account(datamover_ips)
                self.config_alarm_report(datamover_ips, management_float_ip)
                self.modify_dbbak_script(datamover_ssh_info)
                self.modify_log_monitor(datamover_ssh_info)
            self.reencrypt_data()
            self.common_config.ha_sync_file(self.ebackup_ips)
            for ip_dict in self.ips:
                management_float_ip = ip_dict.get('management_float_ip')
                EBackupUtil(self.project_id, self.pod_id).add_new_user(management_float_ip, "EBKUser", is_upgrade=True)
            return Message(200)

        try:
            return do_execute()
        except HCCIException as ex:
            logger.error(str(ex))
            return Message(500, ex)
        except Exception as ex:
            logger.error(str(ex))
            return Message(500, error_msg_cn="升级后配置出现异常，请联系技术支持",
                           error_msg_en="Exception occurs when config eBackup, please contact support engineers")

    def reencrypt_data(self):
        logger.info("Begin to reencrypt data for eBackup")
        ebackup_changer = EbackupCipherChanger(None, self.param_dict, self.project_id, region_id=self.regionid_list[0],
                                               pod_id=self.pod_id)
        for ebackup_ip in self.ebackup_ips:
            ebackup_changer.reencrypt_nginx_cert(ebackup_ip)

    def config_op_svc_account(self, ip_lists):
        def do_config_op_svc_account():
            logger.info("Begin to config op svc account for eBackup")
            float_ip = Utils.find_internal_ip(ip_lists[0], self.hcp_password, self.root_password)
            self.common_config.no_limit_8090_iptables(ip_lists, float_ip)
            ebk_iam_headers = self.common_config.login_ebk_iam(f"{float_ip}:8090")
            self.config_iam_to_datamover(float_ip, ebk_iam_headers)
            self.common_config.logout_ebk_iam(f"{float_ip}:8090", ebk_iam_headers)
            self.common_config.limit_8090_iptables(ip_lists, float_ip)
            logger.info("Succeed to config IAM account.")

        try:
            do_config_op_svc_account()
        except Exception as iam_err:
            logger.error(f"Failed to config op svc account for eBackup, reason: {repr(iam_err)}")
            raise iam_err

    @staticmethod
    def modify_dbbak_script(ssh_info: SshInfo):
        try:
            Utils.set_s3_dbbak_config(ssh_info)
        except Exception as err:
            logger.error(f"Modify S3 management data restore script failed. reason{repr(err)}")
            raise HCCIException(650075, str(repr(err))) from err

    @staticmethod
    def modify_log_monitor(ssh_info: SshInfo):
        bin_path = "/opt/huawei-data-protection/ebackup/bin"
        log_monitor_path = f"{bin_path}/log_monitor.sh"
        check_cmd = f"echo lines:$(grep -n '\[ \! \-f \\\"\\${{file}}\\\" \]' {log_monitor_path} | wc -l)"
        for datamover_ip in ssh_info.node_ips:
            ssh_client = Utils.get_ssh_client(ssh_info, login_ip=datamover_ip)
            ret_strings = Utils.ssh_send_comand_return(ssh_client, check_cmd)
            if str(ret_strings).find("lines:2") == -1:
                logger.info(f"Already modify log monitor scripts, Skip on node({datamover_ip})")
                continue
            modify_cmd = f"chattr -i {bin_path};lines=$(grep -n '\[ \! \-f \\\"\${{file}}\\\" \]' {log_monitor_path}" \
                         f" | tail -n1 | awk -F':' '{{print $1}}');sed -i \"${{lines}}c\                " \
                         f"if ! sudo \\\"\${{G_ROOT_TOOL_PATH}}/common/sudo_common_func.sh\\\" check_dblog_size " \
                         f"2>/dev/null | grep -wq \\\"\${{file}}\\\";then\" {log_monitor_path} && " \
                         f"sed -i \"55c\     G_ALARM_SEND_TMP_FILE=\\\"\${{G_HOME_PATH}}/tmp/" \
                         f"file_size_alarm_tmp.txt\\\"\" {log_monitor_path} && sed -i \"116c\     " \
                         f"G_ALARM_SEND_TMP_FILE=\\\"\${{G_HOME_PATH}}/tmp/file_size_alarm_tmp_db.txt\\\"\" " \
                         f"{log_monitor_path};echo \"modify status:$?\";chattr +i {bin_path}"
            ret_strings = Utils.ssh_send_comand_return(ssh_client, modify_cmd)
            if str(ret_strings).find("modify status:0") == -1:
                logger.error(f"Failed modify log_monitor script. stdout: {ret_strings}")
                raise HCCIException(650077, datamover_ip, str(ret_strings))
            logger.info(f"Success modify log_monitor script on node({datamover_ip})")

    def config_iam_to_datamover(self, float_ip, headers):
        account = self.iam_util.get_iam_account_info("op_service")
        iam_address = self.common_config.get_iam_domain_name()
        data = '{ "ServiceName":["ebk_alarm", "ebk_iam"], "Settings":[{"ConfDomain" : ' \
               '"OpenstackConfig","ConfItem": "OpenstackConfigValue",' \
               '"ConfValue" : "{\\"AuthType\\": \\"IAM\\",' \
               '\\"AuthUserName\\": \\"%s\\",\\"AuthPassword\\": \\"%s\\",' \
               '\\"AuthURL\\": \\"https://%s\\"}"}]}' % \
               (account.account_name, account.account_pwd, iam_address)
        url = "https://%s:8090/v1/srv_governance/service_settings" % float_ip
        rsp = self.rest_api.patch(url, headers, data)
        if rsp.status_code != 200:
            logger.error(f"rest req return not 200. status_code = {rsp.status_code}")
            raise HCCIException(650066)
        errcode = rsp.json().get("Error", {}).get("Code")
        if errcode is None:
            logger.error("errcode is empty.")
            raise HCCIException(650067)
        if errcode != 0:
            description = rsp.json().get("Error", {}).get("Description")
            logger.error("config openstack failed, description:%s", description)
            raise HCCIException(650068, description)
        logger.info("config iam account to datamover succ.")

    def add_white_list(self, datamover_ips, management_float_ip):
        logger.info("Add XAAS whitelist(ebackup_server) start.")
        region_name = self.common_config.get_region_name_by_region_id(self.regionid_list[0])
        is_ok = self.manageone.add_xaas_white_list(
            params_list=[self.project_id, "eBackup", datamover_ips, '', '', region_name], rest_protocol="1",
            snmp_protocol="", float_ip=management_float_ip)

        if not is_ok:
            logger.error("Add XAAS whitelist(ebackup_server) failed.")
            raise Exception("Add XAAS whitelist(ebackup_server) failed.")
        logger.info("Add XAAS whitelist succeeded.")

    def config_alarm_report(self, datamover_ips, management_float_ip):
        logger.info("Begin to config alarm report.")
        self.add_white_list(datamover_ips, management_float_ip)

        i_base_token, session_content = Utils.login(management_float_ip, self.param_dict["eBackup_admin_pwd"])
        alarm_user, alarm_pwd = self.manageone.get_thirdparty_infor(self.pod_id)
        if not alarm_user:
            logger.error("Alarm user is empty.")
            raise Exception("Alarm user is empty.")
        if not alarm_pwd:
            logger.error("Alarm pwd is empty.")
            raise Exception("Alarm pwd is empty.")

        headers = {
            'Accept': 'application/json;version=2.2;charset=UTF-8',
            'iBaseToken': i_base_token,
            'Cookie': f'language=en;{session_content}; DEVICE_ID=dev; sessionIdleTime=60000; '
                      f'MACHINEROLE=0; CSRF_IBASE_TOKEN={i_base_token}'}

        oc_addrss = self.common_config.get_oc_domain_name()
        logger.info("Alarm url:" + oc_addrss)
        data = '{"ISALARMON":1,"ALARMSERVICETYPE":"1","ALARMSERVERURL":"' + oc_addrss + '","ALARMUSERNAME":"' + \
               alarm_user + '","ALARMPASSWORD":"' + alarm_pwd + '"}'

        url = Utils.get_server_url(management_float_ip) + '/alarm_settings/default'
        num = 0
        while num < 12:
            rsp = self.rest_api.put(url, headers, data)
            if rsp.status_code != 200:
                logger.error("Rest req return not 200.")
                break
            errcode = rsp.json()["error"]["code"]
            description = rsp.json()["error"]["description"]
            if errcode is None or description is None:
                logger.error("Errcode or description is empty.")
                break
            if errcode != 0:
                logger.error("Config alarm report failed,description:" + description)
                time.sleep(20)
                num += 1
                continue
            logger.info("Config alarm report succeeded.")
            break
        if not Utils.logout(management_float_ip, i_base_token, session_content, 0):
            logger.error("fail to logout")

    @staticmethod
    def check_file_name(name):
        if not re.match(CMD_ALLOW_PATTERN, name):
            raise ValueError(f"filename {name} may have shell inject risk, terminate it.")

    def check_password_for_pacific_api(self, datamover_ip):
        enc_old_password = self.config_dict.get("pacific_api_old_enc_password")  # 密文密码
        enc_new_password = self.config_dict.get("pacific_api_new_enc_password")  # 密文密码
        conf_file_path = "/opt/root_tool/vbstool/conf/dsware-api.properties"
        datamover_ssh_info = SshInfo(datamover_ip, "hcp", self.hcp_password, self.root_password)
        ssh_client = Utils.get_ssh_client(datamover_ssh_info)

        check_used_old_password_cmd = f"grep -q '{enc_old_password}' {conf_file_path};echo \"check status:$?\""
        ret_strings = Ssh.ssh_send_command(ssh_client, check_used_old_password_cmd, "#", 300)
        if str(ret_strings).find("check status:0") == -1:
            logger.info("Cannot find old enc password, skip replace password.")
            return
        replace_password_cmd = f"sed -i 's#passwd=.*#passwd={enc_new_password}#g' {conf_file_path};" \
                               f"echo \"replace status:$?\""
        ret_strings = Ssh.ssh_send_command(ssh_client, replace_password_cmd, "#", 300)
        if str(ret_strings).find("replace status:0") == -1:
            clean_stdout = str(ret_strings).replace(enc_old_password, 'XXXXX')
            clean_stdout = clean_stdout.replace(enc_new_password, 'XXXXX')
            logger.error(f"Replace api password to new one failed. stdout:{clean_stdout}")
            raise HCCIException(650076, datamover_ip, clean_stdout)
        logger.info(f"Success replace api password on node({datamover_ip}).")

    def install_pacific_api_package(self):
        file_path, name = find_software_package_by_name(
            pkg_pre_name="OceanStor-Pacific_",
            pkg_post_name="_api.tar.gz",
            project_id=self.project_id)
        self.check_file_name(name)
        file_path = os.path.join(file_path, name)
        if not os.path.exists(file_path):
            raise HCCIException("not found OceanStor-Pacific_*.tar.gz")
        check_compress_file([file_path])
        ebackup_ips = []
        datamover_hosts = self.param_dict["eBackup_Datamover_nodes"]
        for group in datamover_hosts.split('|'):
            ebackup_datamover_ips = group.split(';')
            ebackup_ips += ebackup_datamover_ips

        # IP SAN场景不需要去升级fusionstorage api包
        check_fs_cmd = "ls /opt/dsware/agent;echo execute check result: $?"
        for datamover_ip in ebackup_ips:
            result = self.ssh_obj.ssh_cmds(
                datamover_ip, check_fs_cmd, "hcp", self.hcp_password,
                self.root_password, "", "")
            if "execute check result: 0" in result:
                logger.info("it is fs.")
                self.ssh_obj.put_file(datamover_ip, "hcp", self.hcp_password,
                                      file_path, ".", 22)
                self.install_pacific_api_to_ebackup_node(name, datamover_ip)
                self.check_password_for_pacific_api(datamover_ip)

    def install_pacific_api_to_ebackup_node(self, name, datamover_ip):
        if name.endswith("zip"):
            uncompress_cmd = "unzip -o '" + name + "'"
        else:
            uncompress_cmd = "tar -zxvf '" + name + "'"

        check_archi_cmd = "lscpu |grep  aarch64; echo execute check result: $?"
        result = self.ssh_obj.ssh_cmds(datamover_ip, check_archi_cmd, "hcp", self.hcp_password, self.root_password, "",
                                       "")
        if "execute check result: 0" in result:
            architecture = "aarch64"
        else:
            architecture = "x86-64"

        logger.info("architecture is %s.", architecture)
        vbstool_path = "/opt/root_tool/vbstool"
        cmd = f"rm -rf OceanStor-Pacific_*_*/;{uncompress_cmd};" \
              f"mkdir /var/ebackup_bak;/bin/cp {vbstool_path} /var/ebackup_bak/ -prf;" \
              f"rm -rf {vbstool_path}/lib/commons-lang*.jar;" \
              f"rm -rf {vbstool_path}/lib/dsware-api*.jar;" \
              f"mkdir -p {vbstool_path}/lib/scripts/;" \
              f"mkdir -p {vbstool_path}/javarunenv/;" \
              f"tar -zmxvf /opt/dsware/agent/jre-*.tar.gz -C {vbstool_path}/javarunenv/;" \
              f"cd OceanStor-Pacific_*_*/;rm -rf lib/log4j-* ;sed -i 's#export LD_LIBRARY_PATH=" \
              f".*#export LD_LIBRARY_PATH=$LD_LIBRARY_PATH:{vbstool_path}/lib#g' {vbstool_path}/vrmVBSTool.sh;" \
              f"/bin/cp fsa_server.key primary_ks.key standby_ks.key {vbstool_path}/conf/;" \
              f"/bin/cp dr_cli.xml readme.txt version zk-client.jks {vbstool_path};" \
              f"/bin/cp dsware-api-*.jar {vbstool_path}/lib/;" \
              f"/bin/cp  primary_ks.key  standby_ks.key {vbstool_path}/lib/;" \
              f"/bin/cp -r  scripts/* {vbstool_path}/lib/scripts/;" \
              f"/bin/cp -r  lib/* {vbstool_path}/lib/;" \
              f"touch {vbstool_path}/storage_port.ini;" \
              f"/bin/cp -r {vbstool_path}/lib/linux-{architecture}/* {vbstool_path}/lib/;" \
              f"export LD_LIBRARY_PATH=/usr/lib64:{vbstool_path}/lib/linux-{architecture}"
        self.ssh_obj.ssh_cmds(datamover_ip, cmd, "hcp", self.hcp_password, self.root_password, "", "")
        logger.info("execute install api package succ:" + datamover_ip)

    def get_ebackup_service_system_info(self):
        service_system_info = []
        datamover_hosts = self.param_dict["eBackup_Datamover_nodes"]
        for index, group in enumerate(datamover_hosts.split('|')):
            ebackup_datamover_ips = group.split(';')
            self.ebackup_ips += ebackup_datamover_ips
            datamover_ssh_info = SshInfo(ebackup_datamover_ips, "hcp", self.hcp_password, self.root_password)
            datamover_float_ip = Utils.find_float_ip(datamover_ssh_info)
            datamover_system_info = {
                "name": self.param_dict[
                            "current_region_id"] + '_' + "eBackup-Server",
                "alias": self.param_dict[
                             "current_region_id"] + '_' + "eBackup-Server",
                "product_name": "eBackup",
                "product_version": self.param_dict["eBackup_Version"],
                "product_patch_version": [],
                "ipaddress": datamover_float_ip
            }
            logger.info("eBackup-Server %s  info:%s " %
                        (index + 1, str(datamover_system_info)))
            service_system_info.append(datamover_system_info)
        return service_system_info

    def report_service_info2_manageone(self, service_system_info):
        logger.info("Begin to report system info to manageone.")
        mo_api = ManageOneOcApi(self.project_id, self.param_dict["current_region_id"])
        if not mo_api.login():
            logger.error("Login to OC failed.")
            raise HCCIException("Login to OC failed.")
        if not mo_api.report_system_info(service_system_info):
            logger.error("Report system info to OC failed.")
            mo_api.logout()
            raise HCCIException("Report system info to OC failed.")
        logger.info("Report system info to manageone successfully.")
        mo_api.logout()

    def modify_alarm_region_conf(self, project_id):
        logger.info("start to get om_ip")
        cmdb_util = ManageOneCmdbUtil(project_id)
        region_infos = cmdb_util.get_region_info(self.param_dict.get('current_region_id'))[0]
        region_name = region_infos.get('name', '')

        logger.info("all nodes:%s." % str(self.ebackup_ips))
        for ebk_ip in self.ebackup_ips:
            cmd = "sed -i 's/CurrentRegion=.*/CurrentRegion=%s/g'" \
                  " /opt/huawei-data-protection/ebackup/microservice" \
                  "/ebk_alarm/conf/hcpconf.ini" % region_name

            self.ssh_obj.ssh_cmds(ebk_ip, cmd, "hcp", self.hcp_password, self.root_password, "", "")

    def modify_ntp_conf(self):
        logger.info("all nodes:%s." % str(self.ebackup_ips))
        for ebk_ip in self.ebackup_ips:
            logger.info("The NTP node to be configured is %s ." % ebk_ip)
            f_cmd = "; echo execute ntp config result: $?"
            ntp_cmd = """sed -i '/maxdist/d' /etc/ntp.conf && sed -i """ \
                      """'1 i tos maxdist 30' /etc/ntp.conf && """ \
                      """cat /etc/crontab|grep "/sbin/hwclock -w" || """ \
                      """sed -i '$a */10 * * * * root /sbin/hwclock""" \
                      """ -w' /etc/crontab && systemctl restart ntpd """ \
                      """ && service cron restart""" + "%s" % f_cmd

            result = self.ssh_obj.ssh_cmds(
                ebk_ip, ntp_cmd, "hcp", self.hcp_password,
                self.root_password, "", "")
            if "execute ntp config result: 0" not in result:
                logger.error(
                    "config ntp on node(%s) failed,description:%s" %
                    (ebk_ip, result))
                raise Exception("failed to configure the NTP service.")
        logger.info("config ntp succ.")

    def rollback(self, project_id, pod_id, regionid_list=None):
        return Message(200)

    def retry(self, project_id, pod_id, regionid_list=None):
        return self.execute(project_id, pod_id, regionid_list)


class UpdateCMDB(StepBaseInterface):
    def __init__(self, project_id, pod_id, regionid_list=None):
        super(UpdateCMDB, self).__init__(project_id, pod_id, regionid_list)
        self.project_id = project_id
        self.pod_id = pod_id
        self.regionid_list = regionid_list
        self.param_dict = Utils.init_system_params(project_id, regionid_list[0])
        self.cmdb_util = ManageOneCmdbUtil(project_id, pod_id)
        self.hcp_password = self.param_dict["eBackup_hcp_pwd"]
        self.root_password = self.param_dict["eBackup_root_pwd"]
        self.fsp_pwd = self.param_dict.get('openstack_fsp_pwd')
        self.fsp_root_pwd = self.param_dict.get('openstack_root_pwd')
        self.cps_user_pwd = self.param_dict.get('openstack_cps_admin_pwd')
        self.reverse_proxy_ip = self.param_dict.get('openstack_reverse_proxy_ip')

    def execute(self, project_id, pod_id, regionid_list=None):
        def do_execute():
            logger.info("Begin to register eBackup info to CMDB.")
            self.set_cloud_service_info_of_ebackup()
            logger.info("Update CMDB info completed.")
            return Message(200)

        try:
            return do_execute()
        except HCCIException as ex:
            logger.error(str(ex))
            return Message(500, ex)
        except Exception as err:
            return Message(500, error_msg_cn=f"更新eBackup CMDB信息出现异常：{str(err)} ",
                           error_msg_en=f"Exception occurs when update eBackup info in CMDB:{str(err)}")

    def login_openstack(self, ssh_client):
        Ssh.ssh_send_command(ssh_client, "TMOUT=0", "#", 300)
        Ssh.ssh_send_command(ssh_client, "source set_env", "1|2", 300)
        Ssh.ssh_send_command(ssh_client, "2", "CPS_USERNAME", 300)
        Ssh.ssh_send_command(ssh_client, "cps_admin", "CPS_PASSWORD", 300)
        resp = Ssh.ssh_send_command(ssh_client, self.cps_user_pwd, "#", 300)
        if [line for line in resp if line.find("Authenticate Failed") >= 0]:
            raise HCCIException(650051)

    def get_az_info(self):
        fsp_ssh_info = SshInfo(self.reverse_proxy_ip, "fsp", self.fsp_pwd, self.fsp_root_pwd)
        ssh_client = Utils.get_ssh_client(fsp_ssh_info)
        self.login_openstack(ssh_client)
        get_az_list_cmd = "template_list=`cps template-list --service cinder | grep 'cinder-backup-' | " \
                          "awk '{print $4}' | sort | uniq`;az_list=`for template_name in ${template_list[@]};" \
                          " do cps template-ext-params-show --service cinder ${template_name} | " \
                          "grep -P 'ebk_.*?\.azs' | awk '{print $4}';done`;echo \"${az_list}\" | sort | uniq"
        az_list_str = ""
        output_list = Ssh.ssh_send_command(ssh_client, get_az_list_cmd, "#", 300)
        if len(output_list) > CMD_OUTPUT_LENGTH:
            az_list = output_list[1:-1]
            az_list_str = ",".join([az.strip('\n') for az in az_list])
        return az_list_str

    def set_cloud_service_info_of_ebackup(self):
        index_name = 'eBackup'
        current_version = self.param_dict.get('eBackup_Version')
        az_list_str = self.get_az_info()
        cloud_info_list = self.cmdb_util.get_cloud_service_info_v3(self.param_dict.get('current_region_id'), index_name)
        for cloud_service in cloud_info_list:
            deploy_node = self.cmdb_util.get_deploy_node_info(self.param_dict.get('current_region_id'), index_name)
            if cloud_service.get("name", "").startswith("eBackup_service1"):
                extend_infos = cloud_service.get("extendInfos", [])
                extend_infos_list = [extend_info for extend_info in extend_infos if extend_info.get("key") != "az_list"]
                extend_infos_list.append({"key": "az_list", "value": az_list_str})
                cloud_service.update({"extendInfos": extend_infos_list})

            cloud_service.update({"deploy_node_infos": deploy_node})
            cloud_service.update({"version": current_version})
            logger.info(f"start set cloud info, info {str(cloud_service)}.")
            self.cmdb_util.set_cloud_service_info_v3(self.param_dict.get('current_region_id'), cloud_service)

    def rollback(self, project_id, pod_id, regionid_list=None):
        return Message(200)

    def retry(self, project_id, pod_id, regionid_list=None):
        return self.execute(project_id, pod_id, regionid_list)


class ClearingDataAfterUpgrade(StepBaseInterface):
    def __init__(self, project_id, pod_id, regionid_list=None):
        super(ClearingDataAfterUpgrade, self).__init__(project_id, pod_id, regionid_list)
        self.project_id = project_id
        self.pod_id = pod_id
        self.param_dict = Utils.init_system_params(project_id, regionid_list[0])
        self.hcp_password = self.param_dict["eBackup_hcp_pwd"]
        self.root_password = self.param_dict["eBackup_root_pwd"]

    def execute(self, project_id, pod_id, regionid_list=None):
        ebackup_ips = self.get_ebackup_ips()
        clean_useless_path_cmd = 'rm -rf /tmp/upgrade/upgrade_pkg ' \
                                 '/opt/link_var_ebackup_bak ' \
                                 '/var/ebackup_bak'

        def do_execute():
            for ebackup_ip in ebackup_ips:
                Ssh().ssh_cmds(ebackup_ip, clean_useless_path_cmd, 'hcp', self.hcp_password, self.root_password, "", "")
            else:
                return Message(200)

        try:
            return do_execute()
        except HCCIException as ex:
            logger.error(str(ex))
            return Message(500, ex)
        except Exception as error:
            return Message(500, error_msg_cn=f"刪除废弃的文件夹异常：{str(error)} ",
                           error_msg_en=f"Exception occurs when delete useless path:{str(error)}")

    def get_ebackup_ips(self):
        ebackup_ips = []
        datamover_hosts = self.param_dict["eBackup_Datamover_nodes"]
        for group in datamover_hosts.split('|'):
            ebackup_ips += group.split(';')
        return ebackup_ips

    def rollback(self, project_id, pod_id, regionid_list=None):
        return Message(200)

    def retry(self, project_id, pod_id, regionid_list=None):
        return self.execute(project_id, pod_id, regionid_list)


class RegisterUnifyToManageone(StepBaseInterface):
    def __init__(self, project_id, pod_id, regionid_list=None):
        super(RegisterUnifyToManageone, self).__init__(
            project_id, pod_id, regionid_list)
        self.datamover_hosts = None
        self.account_info = None
        self.ebk_iam_headers = None
        self.project_id = project_id
        self.pod_id = pod_id
        self.regionid_list = regionid_list
        self.unitepwd_api = UnitePwdApi(self.project_id)
        self.param_dict = Utils.init_system_params(project_id,
                                                   regionid_list[0])
        self.hcp_password = self.param_dict["eBackup_hcp_pwd"]
        self.root_password = self.param_dict["eBackup_root_pwd"]
        self.params = ParamUtil()
        self.ssh_obj = Ssh()
        self.mo_api = ManageOneOcApi(self.project_id,
                                     self.param_dict["current_region_id"])

        self.iam_util = IamUtil(self.project_id, self.pod_id)
        self.configer = CommonConfig(self.param_dict, pod_id=self.pod_id, project_id=project_id)
        self.rest_api = EbackupRest()

    def execute(self, project_id, pod_id, regionid_list=None):
        def do_execute():
            self.datamover_hosts = self.param_dict["eBackup_Datamover_nodes"]
            self.account_info = self.get_ebackup_account_info()
            for group in self.datamover_hosts.split('|'):
                ebackup_datamover_ips = group.split(';')
                datamover_ssh_info = SshInfo(ebackup_datamover_ips, "hcp", self.hcp_password, self.root_password)
                datamover_float_ip = Utils.find_float_ip(datamover_ssh_info)
                datamover_inter_ip = Utils.find_internal_ip(ebackup_datamover_ips[0], self.hcp_password,
                                                            self.root_password)
                self.update_os_account_info(datamover_float_ip, ebackup_datamover_ips)
                self.ebackup_register_unify_pwd(datamover_inter_ip, datamover_float_ip, ebackup_datamover_ips)
            self.delete_iam_users()
            return Message(200)

        try:
            return do_execute()
        except HCCIException as err:
            logger.error("register  unify password fail. reason: %s" % err)
            return Message(500, error_msg_cn="对接eBackup统一密码失败，请联系华为技术工程师",
                           error_msg_en="Exception occurs when Register eBackup, "
                                        "please contact huawei technical engineers")

    def get_ebackup_account_info(self):
        """
        从MO上获取eBackup统一密码账户信息
        :return: { IP ：[accountName ,...],.. }
        """
        try:
            result = \
                self.unitepwd_api.get_account_info(
                    region_id=self.regionid_list[0],
                    used_scene=None,
                    account_list=[],
                    component_name="eBackup",
                    sub_component_name="eBackup")
        except Exception as err:
            logger.error(
                "Failed to execute get_account_info, reason: %s" % str(repr(err)))
            result = {}
        message = result.get("message")
        if message != "success":
            raise HCCIException("Failed to get account info of eBackup from MO.")
        data = result.get("data")
        account_list = {}
        for account_info in data:
            if account_info.get("component") == "eBackup":
                ipaddress = account_info.get('ip')
                account_name = account_info.get('accountName')
                if account_name == "admin":
                    continue
                if account_list.get(ipaddress):
                    account_list.get(ipaddress).append(account_name)
                else:
                    account_list[ipaddress] = [account_name]
        return account_list

    def delete_iam_users(self):
        component_data = {
            "componentName": "eBackup",
            "subComponents": [
                {
                    "subComponentName": "eBackupManager",
                    "createdAccountList": []
                }
            ]
        }
        quality_item = {
            "difok": 3,
            "minlen": 8,
            "ucredit": 1,
            "lcredit": 1,
            "dcredit": 1,
            "ocredit": 1,
            "passwdChangeMinBetweenTime": 0,
            "passwdChangeMaxBetweenTime": 0
        }
        for account in ["ebackup_karbor", "op_svc_ebackup"]:
            account_dic = {
                "accountName": account,
                "region": self.regionid_list[0],
                "accountType": 5,
                "accountDescription": IAM_ACCOUNT_DESC,
                "modifyType": 3,
                "riskMessage": '{"zh-cn":"","en-us":""}',
                "passwdComplexity": quality_item,
                "operationType": 2
            }
            component_data.get("subComponents")[0]["createdAccountList"].append(account_dic)
        result = self.send_unify_pwd_req(component_data)
        if not result:
            raise Exception("Failed to delete IAM user accounts.")

    def update_os_account_info(self, float_ip, ebackup_ips):
        float_account_list = self.account_info.get(float_ip, [])
        password_list = {"hcp": self.hcp_password, "root": self.root_password}
        if "hcp" not in float_account_list and "root" not in float_account_list:
            return
        for account in ["hcp", "root"]:
            ssh_info = SshInfo(ebackup_ips[0], "hcp", password_list.get("hcp"), password_list.get("root"))
            os_account_info = AccountTool.get_hcp_or_root_time_about(account, ssh_info)
            # 删除原os账户信息
            data = AccountTool.build_unify_pwd_body_os(
                os_account_info, user_account=account, region_id=self.regionid_list[0],
                operation_type=2, user_pwd=password_list.get(account))
            result = self.send_unify_pwd_req(data)
            if not result:
                raise Exception("Failed to send delete OS account info.")

        for account in ["hcp", "root"]:
            for ebackup_ip in ebackup_ips:
                ssh_info = SshInfo(ebackup_ip, "hcp", password_list.get("hcp"), password_list.get("root"))
                os_account_info = AccountTool.get_hcp_or_root_time_about(account, ssh_info)
                # 重新注册账户
                data = AccountTool.build_unify_pwd_body_os(
                    os_account_info, user_account=account, region_id=self.regionid_list[0],
                    user_pwd=password_list.get(account))
                result = self.send_unify_pwd_req(data)
                if not result:
                    raise Exception("Failed to send reg %s account(%s) info." % (ebackup_ip, account))

    def ebackup_register_unify_pwd(self, datamover_inter_ip, datamover_float_ip, ebackup_ips):
        lb_listen_addr1 = "%s:8090" % datamover_inter_ip
        if len(ebackup_ips) > 1:
            lb_listen_addr2 = "%s:8088" % datamover_float_ip
        else:
            lb_listen_addr2 = "%s:8088" % ebackup_ips[0]
        self.configer.no_limit_8090_iptables(ebackup_ips, datamover_inter_ip)
        data = self.build_unify_pwd_body(lb_listen_addr1, lb_listen_addr2)
        result = self.send_unify_pwd_req(data)
        self.configer.limit_8090_iptables(ebackup_ips, datamover_inter_ip)
        if not result:
            raise HCCIException("send unif pwd request fail")
        logger.info("register ebackup unify pwd to manageone succ.")

    def build_unify_pwd_body(self, lb_listen_addr1, lb_listen_addr2):
        sub_com_name = "eBackupServer"
        data = {"componentName": "eBackup",
                "subComponents": [{"subComponentName": sub_com_name, "createdAccountList": []}]}
        unify_pwd_url_ip = lb_listen_addr2.split(":")[0]
        unify_pwd_modify_url = "https://%s%s" % (lb_listen_addr2, REST_URI_GOV_UNIFYPWD_MODIY)
        unify_pwd_reset_url = "https://%s%s" % (lb_listen_addr2, REST_URI_GOV_UNIFYPWD_RESET)
        unify_pwd_sync_status_url = "https://%s%s" % (lb_listen_addr2, REST_URI_GOV_UNIFYPWD_QUERYJOB)
        unify_pwd_verify_url = "https://%s%s" % (lb_listen_addr2, REST_URI_GOV_UNIFYPWD_VERIFY)

        self.ebk_iam_headers = self.configer.login_ebk_iam(lb_listen_addr1)
        quality_item = self.get_admin_pwd_quality(lb_listen_addr1)
        last_time_second = self.get_admin_change_time(lb_listen_addr1)

        last_time = int(last_time_second) * 1000
        expire_time = (int(last_time_second) + int(quality_item["passwdChangeMaxBetweenTime"]) * ONE_DAY_IN_SEC) * 1000

        account_pwd = self.param_dict.get('eBackup_admin_pwd')
        account_old_pwd = self.param_dict.get('eBackup_admin_pwd')
        account_dic = {"accountName": "admin",
                       "region": self.regionid_list[0],
                       "accountType": 6,
                       "accountDescription": ADMIN_ACCOUNT_DESC,
                       "passwd": account_pwd,
                       "oldPasswd": account_old_pwd,
                       "lastPasswdChange": last_time,
                       "modifyType": 1,
                       "passwdExpires": expire_time,
                       "riskMessage": '{"zh-cn":"","en-us":""}',
                       "passwdComplexity": quality_item,
                       "ip": unify_pwd_url_ip,
                       "usedScene": unify_pwd_url_ip,
                       "urlModify": unify_pwd_modify_url,
                       "urlReset": unify_pwd_reset_url,
                       "urlSyncStatus": unify_pwd_sync_status_url,
                       "urlValid": unify_pwd_verify_url,
                       "operationType": 0}
        data.get("subComponents")[0].get("createdAccountList").append(account_dic)
        return data

    def get_admin_pwd_quality(self, lb_listen_addr):
        url = "https://%s/v3/auth/security_policy" % \
              lb_listen_addr
        rsp = self.rest_api.get(url, self.ebk_iam_headers)
        if rsp.status_code != 200:
            logger.error("rest req return not 200.")
            return ""
        errcode = rsp.json()["Error"]["Code"]
        if errcode != 0:
            description = rsp.json()["Error"]["Description"]
            logger.error(
                "get admin pwd quality failed, description:%s" % description)
            raise HCCIException(
                "get the security stratage of password from"
                " ebk_iam failed. reason:%s" % description)
        data_info = rsp.json()["Data"]
        quality_item = {
            "difok": int(data_info["PwdComplex"]) + 3,
            "minlen": int(data_info["PasswordMinLength"]),
            "ucredit": 1,
            "lcredit": 1,
            "dcredit": 1,
            "ocredit": 1,
            "passwdChangeMinBetweenTime": int(data_info["MinValidPeriod"]),
            "passwdChangeMaxBetweenTime": int(data_info["MaxValidPeriod"])
        }
        return quality_item

    def get_admin_change_time(self, lb_listen_addr):
        url = "https://%s/v3/auth/singleuser?Id=admin" % lb_listen_addr
        rsp = self.rest_api.get(url, self.ebk_iam_headers)
        if rsp.status_code != 200:
            logger.error("rest req return not 200.")
            raise HCCIException(
                "response code is not 200, check the HCP_AdminNode.log.")
        errcode = rsp.json()["Error"]["Code"]
        description = rsp.json()["Error"]["Description"]
        if errcode is None or description is None:
            logger.error("errcode or description is empty.")
            raise HCCIException(
                "response code is not 200, check the HCP_AdminNode.log.")
        if errcode != 0:
            logger.error("get admin pwd change time failed, "
                         "description:%s" % description)
            raise HCCIException(
                "get modify time of admin`s password fail"
                ", reason: %s" % description)
        data = rsp.json()["Data"]
        pwd_change_time = data["Users"][0]["CHANGETIME"]
        return pwd_change_time

    def send_unify_pwd_req(self, data):
        iam_account = self.iam_util.get_iam_account_info("op_service")
        iam_token = self.iam_util.get_iam_token(iam_account)
        register_url = "https://%s:26335/rest/mounpwdservice/v1/account" % \
                       self.mo_api.oc_ip
        header = {
            "Content-Type": "application/json",
            "X-Auth-Token": iam_token
        }
        register_retry_times = 0
        logger.info("Register unify pass_word float_ip:%s" % self.mo_api.oc_ip)
        rsp = None

        while True:
            try:
                rsp = self.rest_api.post(register_url, header, json.dumps(data))
            except Exception as err:
                logger.error("Register eBackup service to mo unify pass_word occur exc, info:%s", str(err))
            if rsp:
                logger.info("Register unify pass_word result code:%s", rsp.status_code)
                if rsp.status_code <= 300 and rsp.content:
                    rsp_content = json.loads(rsp.content)
                else:
                    return False
                if rsp_content.get("code") == "00000000":
                    logger.info("Register eBackup service to mo unify pass_word success, response code:%s",
                                rsp_content.get('code'))
                    return True
                logger.error("Register eBackup service to mo unify pass_word failed, msg:%s", rsp.content)
            register_retry_times += 1
            if register_retry_times > RETRY_TIME_LIMIT:
                logger.error("Unify pass_word reach the maximum number of retries, so do not try again")
                return False
            time.sleep(30)

    def retry(self, project_id, pod_id, regionid_list=None):
        return self.execute(project_id, pod_id, regionid_list)
