# -*- coding:utf-8 -*-
import os

import utils.common.log as logger
from utils.common.ssh_util import Ssh

from plugins.eBackup.common.util import Utils
from plugins.eBackup.common.model import DmkTaskInfo
from plugins.eBackup.scripts.upgrade.dmk_task import DMKTask
from plugins.eBackup.scripts.upgrade.util import DriverConfig
from plugins.eBackup.scripts.upgrade.util import get_ebackup_driver_nodes_from_file


class EbackupDriverUpgrader(object):
    def __init__(self, reverse_proxy_ip, param_dict):
        self.reverse_proxy_ip = reverse_proxy_ip
        self.param_dict = param_dict
        self.dmk_task = DMKTask(param_dict.get('dmk_floatIp'),
                                param_dict.get('eBackup_dmk_user'),
                                param_dict.get('eBackup_dmk_password'))
        self.hosts_dict = self._get_host_dict(reverse_proxy_ip)

    def upgrade(self):
        if 0 == len(self.hosts_dict):
            logger.info("driver node is empty, reverse proxy ip:%s" % self.
                        reverse_proxy_ip)
            return True
        is_true = self.is_all_upgraded()
        if is_true:
            logger.info("All nodes are upgraded.")
            return True
        return self.deploy_with_dmk("upgrade")

    def rollback(self):
        if 0 == len(self.hosts_dict):
            logger.info("driver node is empty, reverse proxy ip:%s" % self.
                        reverse_proxy_ip)
            return True
        is_true = self.is_all_rollbacked()
        if is_true:
            logger.info("All nodes are rollbacked.")
            return True
        return self.deploy_with_dmk("rollback")

    def _get_host_dict(self, reverse_proxy_ip):
        region_id = self.param_dict["current_region_id"]
        file_path = os.path.realpath(
            __file__ + '/../' + region_id + '_' + reverse_proxy_ip + '.txt')
        driver_host_dict = get_ebackup_driver_nodes_from_file(file_path)
        return driver_host_dict

    def is_all_upgraded(self):
        logger.info("Begin to check whether all node are upgraded.")
        for driver_ip in self.hosts_dict.get('eBackupDriver'):
            if not self.is_upgraded(driver_ip):
                logger.info("There some nodes are not upgraded.")
                return False
        logger.info("All nodes are upgraded.")
        return True

    def is_all_rollbacked(self):
        logger.info("Begin to check whether all node are rollbacked.")
        for driver_ip in self.hosts_dict.get('eBackupDriver'):
            if self.is_upgraded(driver_ip):
                logger.info("There some nodes are not rollbacked.")
                return False
        logger.info("All nodes are rollbacked.")
        return True

    def is_upgraded(self, driver_ip):
        ssh = Ssh()
        ssh_client = None
        logger.info("Begin to check current version at host %s" % driver_ip)

        def do_is_upgraded():
            nonlocal ssh_client
            ssh_client = ssh.ssh_create_client(driver_ip, 'fsp', self.param_dict.get('openstack_fsp_pwd'))
            ssh.ssh_send_command(ssh_client, 'su - root', 'Password:', 100)
            ssh.ssh_send_command(ssh_client, self.param_dict.get('openstack_root_pwd'), '#', 100)
            version_check_cmds = self.check_python_version(ssh, ssh_client)

            result = ssh.ssh_exec_command_return_list(ssh_client, version_check_cmds)
            current_version = "No patch" if len(result) == 0 else result[0].strip().replace('\n', '')
            logger.info("The current version is %s at host %s" % (current_version, driver_ip))
            return current_version == self.param_dict.get('eBackup_Version')

        try:
            res = do_is_upgraded()
        except Exception as err:
            logger.error("Exception occurs when check current version,the Exception is:" + str(err))
            raise err
        finally:
            if ssh_client:
                Utils.close_ssh_clinet(ssh_client)
        return res

    def check_python_version(self, ssh, ssh_client):

        # 底座未升级 单升eBackup时 路径为python3.7 与底座一起升级时路径为python
        version_check_cmd = '''version_info=$(''' \
                            '''cat /usr/lib/python/site-packages/cinder/backup/drivers/ebackupversion.conf ''' \
                            '''2>/dev/null || ''' \
                            '''cat /usr/lib/python3.7/site-packages/cinder/backup/drivers/ebackupversion.conf ''' \
                            '''2>/dev/null);echo ${version_info} | awk -F "[=| ]" '{print $2}' '''

        return version_check_cmd

    def deploy_with_dmk(self, action):
        ebackup_driver_config, ebackup_driver_host = self.get_all_params()

        # install eBackup
        ebackup_action = ""
        if action == "upgrade":
            ebackup_action = "[upgrade]3.Upgrade eBackup Driver" if \
                self.param_dict.get('update_scene') == 'upgrade' \
                else '[Upgrade] Upgrade eBackup Driver Patch'
        elif action == "rollback":
            ebackup_action = "[rollback]2.Rollback eBackup Driver" if \
                self.param_dict.get('update_scene') == 'upgrade' \
                else '[Rollback] Rollback eBackup Driver Patch'
        application = 'eBackup_Upgrade' if \
            self.param_dict.get('update_scene') == 'upgrade' \
            else 'eBackupSoftware_Patch'

        upgrade_task = DmkTaskInfo(
            self.param_dict["eBackup_Version"], ebackup_action,
            [ebackup_driver_host, ebackup_driver_config], application)

        result = self.dmk_task.do_task(upgrade_task, "fsp")

        if not result:
            logger.error("Do DMK task %s failed." % ebackup_action)
        return result

    def get_all_params(self):
        config = DriverConfig(hosts_dict=self.hosts_dict,
                              param_dict=self.param_dict, one2all=0)
        return config.get_config()
