# -*- coding:utf-8 -*-

CSBS_SERVICE_NAME = "CSBS-VBS"
KAEBOR_EXT_PORT = "8799"
KARBOR_INSTALL_WORKDIR = "/karbor"
CSBS_PLUGIN_PATH = "/opt/cloud/hcci/src/HCCI/plugins/CSBS"

KARBOR_PKG_SUFFIX_X86 = '_CBS_Karbor_X86.zip'
KARBOR_PKG_SUFFIX_ARM = '_CBS_Karbor_ARM.zip'


class RegisterSDRData(object):
    def __init__(self, region_id, ext_float_ip):
        self.__csbs_id = "csbs"
        self.__vbs_id = "vbs"
        self.__app_backup_id = "app_backup"
        self.__region_id = region_id
        self.__ext_float_ip = ext_float_ip
        self.__sdr_data()

    def __sdr_data(self):
        self.csbs_data = {
            "meterId": self.__csbs_id,
            "resourceType": self.__csbs_id,
            "service": "csbs",
            "csbiName": "CSBS",
            "enable": True,
            "regionId": self.__region_id,
            "dataSource":
                f"https://{self.__ext_float_ip}:{KAEBOR_EXT_PORT}/v1/meters?"
                f"service_type={self.__csbs_id}&resource_type={self.__csbs_id}"
                f"&limit={{0}}&index={{1}}&start={{2}}&end={{3}}",
            "activeDataSource": "",
            "folder": self.__csbs_id,
            "className": "CommonPluginImpl",
            "version": "v2",
            "cycleTime": 3600000,
            "generationDelayTime": 900000,
            "statistics": False,
            "limit": 1000,
            "collectThreadNum": 5
        }
        self.vbs_data = {
            "meterId": self.__vbs_id,
            "resourceType": self.__vbs_id,
            "service": "vbs",
            "csbiName": "VBS",
            "enable": True,
            "regionId": self.__region_id,
            "dataSource":
                f"https://{self.__ext_float_ip}:{KAEBOR_EXT_PORT}/v1/meters?"
                f"service_type={self.__vbs_id}&resource_type={self.__vbs_id}"
                f"&limit={{0}}&index={{1}}&start={{2}}&end={{3}}",
            "activeDataSource": "",
            "folder": self.__vbs_id,
            "className": "CommonPluginImpl",
            "version": "v2",
            "cycleTime": 3600000,
            "generationDelayTime": 900000,
            "statistics": False,
            "limit": 1000,
            "collectThreadNum": 5
        }

        self.appbs_data = {
            "meterId": self.__app_backup_id,
            "resourceType": self.__app_backup_id,
            "service": "csbs",
            "csbiName": "AppBackup",
            "enable": True,
            "regionId": self.__region_id,
            "dataSource":
                f"https://{self.__ext_float_ip}:{KAEBOR_EXT_PORT}/v1/meters?"
                f"service_type={self.__csbs_id}"
                f"&resource_type={self.__app_backup_id}"
                f"&limit={{0}}&index={{1}}&start={{2}}&end={{3}}",
            "activeDataSource": "",
            "folder": self.__app_backup_id,
            "className": "CommonPluginImpl",
            "version": "v2",
            "cycleTime": 3600000,
            "generationDelayTime": 900000,
            "statistics": False,
            "limit": 1000,
            "collectThreadNum": 5
        }


class RegisterToMOInfo(object):
    def __init__(self, region_id, domain_name=None):
        self.__csbs_id = "csbs"
        self.__vbs_id = "vbs"
        self.__dbs_product_type = "dbs"
        self.__fsbs_product_type = "fsbs"
        self.__app_backup_id = "app_backup"
        self.__byte_to_gb = 1073741824
        self.__region_id = region_id
        self.__domain_name = domain_name
        self.__endpoint_info()
        self.__links_info()
        self.__product_action()
        self.__product_info()
        self.__quota_indicators()
        self.__metrics_info()

    def __endpoint_info(self):
        self.csbs_endpoint_info_zh = {
            "id": self.__csbs_id,
            "name": "云服务器备份",
            "shortName": "CSBS",
            "description": "安全可靠的云服务器备份",
            "version": "v1.0",
            "endpoint": f"https://{self.__domain_name}/cbs/#"
                        f"/{self.__csbs_id}/manager/tabs/backup",
            "serviceCss": self.__csbs_id,
            "serviceOrder": 1,
            "defaultOpen": False,
            "domainType": "BR",
            "nonsupportRegions": "",
            "endpointScope": "region",
            "locale": "zh-cn",
            "catalogId": 2,
            "regionId": self.__region_id
        }
        self.csbs_endpoint_info_en = {
            "id": self.__csbs_id,
            "name": "Cloud Server Backup Service",
            "shortName": "CSBS",
            "description": "Secure and reliable protection for cloud servers",
            "version": "v1.0",
            "endpoint": f"https://{self.__domain_name}/cbs/#"
                        f"/{self.__csbs_id}/manager/tabs/backup",
            "serviceCss": self.__csbs_id,
            "serviceOrder": 1,
            "defaultOpen": False,
            "domainType": "BR",
            "nonsupportRegions": "",
            "endpointScope": "region",
            "locale": "en-us",
            "catalogId": 2,
            "regionId": self.__region_id
        }
        self.vbs_endpoint_info_zh = {
            "id": self.__vbs_id,
            "name": "云硬盘备份",
            "shortName": "VBS",
            "description": "安全可靠的块存储备份",
            "version": "v1.0",
            "endpoint": f"https://{self.__domain_name}/cbs/#"
                        f"/{self.__vbs_id}/manager/tabs/backup",
            "serviceCss": self.__vbs_id,
            "serviceOrder": 1,
            "defaultOpen": False,
            "domainType": "BR",
            "nonsupportRegions": "",
            "endpointScope": "region",
            "locale": "zh-cn",
            "catalogId": 2,
            "regionId": self.__region_id
        }
        self.vbs_endpoint_info_en = {
            "id": self.__vbs_id,
            "name": "Volume Backup Service",
            "shortName": "VBS",
            "description": "Secure and reliable protection for block storage",
            "version": "v1.0",
            "endpoint": f"https://{self.__domain_name}/cbs/#"
                        f"/{self.__vbs_id}/manager/tabs/backup",
            "serviceCss": self.__vbs_id,
            "serviceOrder": 1,
            "defaultOpen": False,
            "domainType": "BR",
            "nonsupportRegions": "",
            "endpointScope": "region",
            "locale": "en-us",
            "catalogId": 2,
            "regionId": self.__region_id
        }

    def __links_info(self):
        # Cloud server backup service links.
        self.csbs_apply_link = f"https://{self.__domain_name}/cbs/#" \
                               f"/{self.__csbs_id}/createBackup"
        self.csbs_detail_link = f"https://{self.__domain_name}/cbs/#" \
                                f"/{self.__csbs_id}/productDetail"

        # Volume backup service links.
        self.vbs_apply_link = f"https://{self.__domain_name}/cbs/#" \
                              f"/{self.__vbs_id}/createBackup"
        self.vbs_detail_link = f"https://{self.__domain_name}/cbs/#" \
                               f"/{self.__vbs_id}/productDetail"
        # Database backup service links.
        self.dbs_apply_link = f"https://{self.__domain_name}/cbs/#" \
                              f"/{self.__csbs_id}/createBackup"

        # File set backup service links.
        self.fsbs_apply_link = f"https://{self.__domain_name}/cbs/#" \
                               f"/{self.__csbs_id}/createBackup"

    def __product_action(self):
        self.csbs_vbs_action_list = [
            {
                "service_type": self.__csbs_id,
                "action": "apply"
            },
            {
                "service_type": self.__csbs_id,
                "action": "modify"
            },
            {
                "service_type": self.__csbs_id,
                "action": "delete"
            },
            {
                "service_type": self.__vbs_id,
                "action": "apply"
            },
            {
                "service_type": self.__vbs_id,
                "action": "modify"
            },
            {
                "service_type": self.__vbs_id,
                "action": "delete"
            }
        ]

    def __product_info(self):
        # Cloud server backup service product info.
        self.csbs_product_info = {
            "proNameZh": "云服务器备份",
            "proNameEn": "Cloud Server Backup Service",
            "serviceId": self.__csbs_id,
            "iconId": "",
            "serviceParams": "{}",
            "proDesZh":
                "云服务器备份（Cloud Server Backup Service）为云服务器下所有云"
                "硬盘创建一致性在线备份，无需关机。针对病毒入侵、人为误删除、软"
                "硬件故障等场景，可将数据恢复到任意备份点。",
            "proDesEn":
                "Cloud Server Backup Service (CSBS) backs up Elastic Volume "
                "Service (EVS) disks consistently on Elastic Cloud Servers "
                "(ECSs) without interrupting services. "
                "Leveraging state-of-the-art protection from virus attacks "
                "and software and hardware faults, CSBS allows you to"
                " restore data from any backup point in time.",
            "catalogId": "2",
            "isDefault": "true"
        }
        # Volume backup service product info.
        self.vbs_product_info = {
            "proNameZh": "云硬盘备份",
            "proNameEn": "Volume Backup Service",
            "serviceId": self.__vbs_id,
            "iconId": "",
            "serviceParams": "{}",
            "proDesZh":
                "云硬盘备份（Volume Backup Service）为云硬盘创建在线备"
                "份，无需关机/重启。针对病毒入侵、人为误删除、软硬件故障"
                "等场景，可将数据恢复到任意备份点。",
            "proDesEn":
                "Volume Backup Service (VBS) provides online backup for "
                "Elastic Volume Service (EVS) disks and allows data "
                "restoration from backups, offering protection against "
                "virus attacks and software and hardware faults.",
            "catalogId": "2",
            "isDefault": "true"
        }
        # Database backup service product info.
        self.dbs_product_info = {
            "proNameZh": "数据库备份",
            "proNameEn": "Database Backup",
            "serviceId": self.__csbs_id,
            "iconId": "",
            "serviceParams": "{}",
            "proDesZh":
                "支持备份云服务器或者裸金属服务器中的数据库应用，当发生人为误"
                "删除、软硬件故障等事件时，可将数据恢复到任意备份时间点。",
            "proDesEn":
                "Database on ECSs or BMSs can be backed up. If an event "
                "such as misdeletion or software or hardware fault occurs, "
                "data can be restored to any backup time point.",
            "catalogId": "2",
            "isDefault": "true"
        }
        # File set backup service product info.
        self.fsbs_product_info = {
            "proNameZh": "文件集备份",
            "proNameEn": "File Set Backup",
            "serviceId": self.__csbs_id,
            "iconId": "",
            "serviceParams": "{}",
            "proDesZh":
                "支持备份云服务器或者裸金属服务器中的单个或者多个文件，当发生人"
                "为误删除、软硬件故障等事件时，可将数据恢复到任意备份时间点。",
            "proDesEn":
                "One or more files on ECSs or BMSs can be backed up. "
                "If an event such as misdeletion or software or hardware "
                "fault occurs, data can be restored to any backup time point.",
            "catalogId": "2",
            "isDefault": "true"
        }

    def __quota_indicators(self):
        # CSBS and VBS quota.
        self.csbs_quota_indicators = {
            "quota_indicators": {
                "region_id": self.__region_id,
                "cloud_infra_type": "FUSION_CLOUD",
                "availability_zone_id": "",
                "indicators": [
                    {
                        "name": "backup_capacity",
                        "display_name":
                            '{"zh_cn": "备份空间",'
                            '"en_us": "Backup Space"}',
                        "unit": '{"zh_cn": "GB",'
                                '"en_us": "GB"}',
                        "data_type": "1",
                        "order_no": 140,
                        "min": 0,
                        "max": 2097151,
                        "capacity": 0,
                        "endpoint_id": self.__csbs_id,
                        "request_url": "",
                        "response_template": "",
                    },
                    {
                        "name": "copy_capacity",
                        "display_name":
                            '{"zh_cn": "复制空间",'
                            '"en_us": "Replication Space"}',
                        "unit": '{"zh_cn": "GB",'
                                '"en_us": "GB"}',
                        "data_type": "1",
                        "order_no": 141,
                        "min": 0,
                        "max": 2097151,
                        "capacity": 0,
                        "endpoint_id": self.__csbs_id,
                        "request_url": "",
                        "response_template": "",
                    }
                ]
            }
        }
        self.vbs_quota_indicators = {
            "quota_indicators": {
                "region_id": self.__region_id,
                "cloud_infra_type": "FUSION_CLOUD",
                "availability_zone_id": "",
                "indicators": [
                    {
                        "name": "volume_backup_capacity",
                        "display_name":
                            '{"zh_cn": "云硬盘备份",'
                            '"en_us": "Volume Backup Service"}',
                        "unit": '{"zh_cn": "GB",'
                                '"en_us": "GB"}',
                        "data_type": "1",
                        "order_no": 150,
                        "min": 0,
                        "max": 2097151,
                        "capacity": 0,
                        "endpoint_id": self.__vbs_id,
                        "request_url": "",
                        "response_template": "",
                    },
                    {
                        "name": "volume_copy_capacity",
                        "display_name":
                            '{"zh_cn": "复制空间",'
                            '"en_us": "Replication Space"}',
                        "unit": '{"zh_cn": "GB",'
                                '"en_us": "GB"}',
                        "data_type": "1",
                        "order_no": 151,
                        "min": 0,
                        "max": 2097151,
                        "capacity": 0,
                        "endpoint_id": self.__vbs_id,
                        "request_url": "",
                        "response_template": "",
                    }
                ]
            }
        }
        # Application backup service quota.
        self.app_backup_quota_indicators = {
            "quota_indicators": {
                "region_id": self.__region_id,
                "cloud_infra_type": "FUSION_CLOUD",
                "availability_zone_id": "",
                "indicators": [
                    {
                        "name": "database_backup_capacity",
                        "display_name":
                            '{"zh_cn": "数据库备份空间",'
                            '"en_us": "Database Backup Space"}',
                        "unit": '{"zh_cn": "GB",'
                                '"en_us": "GB"}',
                        "data_type": "1",
                        "order_no": 142,
                        "min": 0,
                        "max": 2097151,
                        "capacity": 0,
                        "endpoint_id": self.__csbs_id,
                        "request_url": "",
                        "response_template": "",
                    },
                    {
                        "name": "fileset_backup_capacity",
                        "display_name":
                            '{"zh_cn": "文件集备份空间",'
                            '"en_us": "File Set Backup Space"}',
                        "unit": '{"zh_cn": "GB",'
                                '"en_us": "GB"}',
                        "data_type": "1",
                        "order_no": 143,
                        "min": 0,
                        "max": 2097151,
                        "capacity": 0,
                        "endpoint_id": self.__csbs_id,
                        "request_url": "",
                        "response_template": "",
                    }
                ]
            }
        }

    def __metrics_info(self):
        self.csbs_metric_info = self._get_csbs_metric_info()
        self.vbs_metric_info = self._get_vbs_metric_info()
        self.app_backup_metric_info = self._get_app_backup_metric_info()

    def _get_csbs_metric_info(self):
        return {
            "metrics": [
                {"service_id": self.__csbs_id,
                 "name": "server_backup",
                 "display_name": "{\"zh_CN\":\"备份容量\",\"en_US\":\"Backup Capacity\"}",
                 "unit": "{\"zh_CN\":\"GB-小时\",\"en_US\":\"GB-Hour\"}",
                 "region_id": self.__region_id,
                 "type": "interval",
                 "cdr_parse_rule": f"parseFloat(Math.round((data.split(\"|\")[13].trim()"
                                   f"/{self.__byte_to_gb}) * Math.pow(10, 6)) / "
                                   f"Math.pow(10, 6).toFixed(6))",
                 "rate_parse_rule": "data.split(\"|\")[7].trim()",
                 "cloud_service_type_code": "hws.service.type.csbs",
                 "cloud_service_type_name": "{\"zh_CN\":\"云服务器备份\",\"en_US\":\"CSBS\"}",
                 "resource_type_code": f"hws.resource.type.{self.__csbs_id}.server_backup",
                 "resource_type_name": "{\"zh_CN\":\"云服务器备份容量\",\"en_US\":\"CSBS Backup Capacity\"}",
                 "cdr_folder_name": self.__csbs_id
                 },
                {"service_id": self.__csbs_id,
                 "name": "server_backup_copy",
                 "display_name": "{\"zh_CN\":\"复制容量\",\"en_US\":\"Duplication Capacity\"}",
                 "unit": "{\"zh_CN\":\"GB-小时\",\"en_US\":\"GB-Hour\"}",
                 "region_id": self.__region_id,
                 "type": "interval",
                 "cdr_parse_rule": f"parseFloat(Math.round((data.split(\"|\")[13].trim()"
                                   f"/{self.__byte_to_gb}) * Math.pow(10, 6)) / "
                                   f"Math.pow(10, 6).toFixed(6))",
                 "rate_parse_rule": "data.split(\"|\")[7].trim()",
                 "cloud_service_type_code": "hws.service.type.csbs",
                 "cloud_service_type_name": "{\"zh_CN\":\"云服务器备份\",\"en_US\":\"CSBS\"}",
                 "resource_type_code": f"hws.resource.type.{self.__csbs_id}.server_backup_copy",
                 "resource_type_name": "{\"zh_CN\":\"云服务器复制容量\",\"en_US\":\"CSBS Duplication Capacity\"}",
                 "cdr_folder_name": self.__csbs_id
                 }
            ]
        }

    def _get_vbs_metric_info(self):
        return {
            "metrics": [
                {"service_id": self.__vbs_id,
                 "name": "vbs",
                 "display_name": "{\"zh_CN\":\"备份容量\",\"en_US\":\"Backup Capacity\"}",
                 "unit": "{\"zh_CN\":\"GB-小时\",\"en_US\":\"GB-Hour\"}",
                 "region_id": self.__region_id,
                 "type": "interval",
                 "cdr_parse_rule": f"parseFloat(Math.round((data.split(\"|\")[13].trim()"
                                   f"/{self.__byte_to_gb}) * Math.pow(10, 6)) / "
                                   f"Math.pow(10, 6).toFixed(6))",
                 "rate_parse_rule": "data.split(\"|\")[7].trim()",
                 "cloud_service_type_code": "hws.service.type.vbs",
                 "cloud_service_type_name": "{\"zh_CN\":\"云硬盘备份\",\"en_US\":\"VBS\"}",
                 "resource_type_code": f"hws.resource.type.{self.__vbs_id}.vbs",
                 "resource_type_name": "{\"zh_CN\":\"云硬盘备份容量\",\"en_US\":\"VBS Backup Capacity\"}",
                 "cdr_folder_name": self.__vbs_id
                 },
                {"service_id": self.__vbs_id,
                 "name": "vbs_copy",
                 "display_name": "{\"zh_CN\":\"复制容量\",\"en_US\":\"Duplication Capacity\"}",
                 "unit": "{\"zh_CN\":\"GB-小时\",\"en_US\":\"GB-Hour\"}",
                 "region_id": self.__region_id,
                 "type": "interval",
                 "cdr_parse_rule": f"parseFloat(Math.round((data.split(\"|\")[13].trim()"
                                   f"/{self.__byte_to_gb}) * Math.pow(10, 6)) / "
                                   f"Math.pow(10, 6).toFixed(6))",
                 "rate_parse_rule": "data.split(\"|\")[7].trim()",
                 "cloud_service_type_code": "hws.service.type.vbs",
                 "cloud_service_type_name": "{\"zh_CN\":\"云硬盘备份\",\"en_US\":\"VBS\"}",
                 "resource_type_code": f"hws.resource.type.{self.__vbs_id}.vbs_copy",
                 "resource_type_name": "{\"zh_CN\":\"云硬盘复制容量\",\"en_US\":\"VBS Duplication Capacity\"}",
                 "cdr_folder_name": self.__vbs_id
                 }
            ]
        }

    def _get_app_backup_metric_info(self):
        return {
            "metrics": [
                {"service_id": self.__app_backup_id,
                 "name": "database_backup",
                 "display_name": "{\"zh_CN\":\"数据库备份\",\"en_US\":\"Database Backup\"}",
                 "unit": "{\"zh_CN\":\"GB-小时\",\"en_US\":\"GB-Hour\"}",
                 "region_id": self.__region_id,
                 "type": "interval",
                 "cdr_parse_rule": f"parseFloat(Math.round((data.split(\"|\")[13].trim()"
                                   f"/{self.__byte_to_gb}) * Math.pow(10, 6)) / "
                                   f"Math.pow(10, 6).toFixed(6))",
                 "rate_parse_rule": "data.split(\"|\")[7].trim()",
                 "cloud_service_type_code": "hws.service.type.csbs",
                 "cloud_service_type_name": "{\"zh_CN\":\"云服务器备份\",\"en_US\":\"CSBS\"}",
                 "resource_type_code": f"hws.resource.type.{self.__app_backup_id}.database_backup",
                 "resource_type_name": "{\"zh_CN\":\"云服务器数据库备份\",\"en_US\":\"Cloud Server Database Backup\"}",
                 "cdr_folder_name": self.__app_backup_id
                 },
                {"service_id": self.__app_backup_id,
                 "name": "fileset_backup",
                 "display_name": "{\"zh_CN\":\"文件集备份\",\"en_US\":\"File Set Backup\"}",
                 "unit": "{\"zh_CN\":\"GB-小时\",\"en_US\":\"GB-Hour\"}",
                 "region_id": self.__region_id,
                 "type": "interval",
                 "cdr_parse_rule": f"parseFloat(Math.round((data.split(\"|\")[13].trim()"
                                   f"/{self.__byte_to_gb}) * Math.pow(10, 6)) / "
                                   f"Math.pow(10, 6).toFixed(6))",
                 "rate_parse_rule": "data.split(\"|\")[7].trim()",
                 "cloud_service_type_code": "hws.service.type.csbs",
                 "cloud_service_type_name": "{\"zh_CN\":\"云服务器备份\",\"en_US\":\"CSBS\"}",
                 "resource_type_code": f"hws.resource.type.{self.__app_backup_id}.fileset_backup",
                 "resource_type_name": "{\"zh_CN\":\"云服务器文件集备份\",\"en_US\":\"Cloud Server File Set Backup\"}",
                 "cdr_folder_name": self.__app_backup_id
                 }
            ]
        }
