import utils.common.log as logger
from utils.PasswordManager.PasswordManager import FSPasswordManager
from utils.common.openstack_host import get_opentack_ssh_info_by_pod_id

from plugins.CSBS.common.model import Node
from plugins.CSBS.common.ssh_client import SshClient
from plugins.CSBS.common.util import auto_retry


class CPSInfo(object):
    cps_user = "fsp"

    def __init__(self, pod_id):
        self.pod_id = pod_id
        self.fsp_pwd_manager = FSPasswordManager(self.pod_id)

    def get_cps_ip(self):
        cps_ip = get_opentack_ssh_info_by_pod_id(self.pod_id).get("host_ip")
        if not cps_ip:
            raise Exception("Get cps_ip failed.")
        return cps_ip

    def get_cps_envpwd(self):
        cps_envpwd = self.fsp_pwd_manager.get_cloud_admin_password()
        if not cps_envpwd:
            raise Exception("Get cps cloud_admin password failed.")
        return cps_envpwd

    def get_cps_userpwd(self):
        cps_userpwd = self.fsp_pwd_manager.get_fsp_password()
        if not cps_userpwd:
            raise Exception("Get cps fsp password failed.")
        return cps_userpwd

    def get_cps_rootpwd(self):
        cps_rootpwd = self.fsp_pwd_manager.get_root_password()
        if not cps_rootpwd:
            raise Exception("Get cps root password failed.")
        return cps_rootpwd

    def get_cps_node(self):
        return Node(self.get_cps_ip(),
                    CPSInfo.cps_user,
                    self.get_cps_userpwd(),
                    'root',
                    self.get_cps_rootpwd(),
                    self.get_cps_envpwd())


class CpsHelper(object):
    def __init__(self, cps_node):
        self.node = cps_node
        self.ssh_client = SshClient()

    def get_cps_client(self):
        logger.info(f'Get cps ssh client, node:{self.node.node_ip}.')
        ssh_client = self.ssh_client.get_ssh_client(self.node, sudo_type="su")
        cmds = "source set_env"
        expect_str = "please choose:[1|2"
        self.ssh_client.ssh_send_command_expect(ssh_client, cmds, expect_str)
        cmds = "4"
        expect_str = "Enter OS_PASSWORD="
        self.ssh_client.ssh_send_command_expect(ssh_client, cmds, expect_str)
        self.ssh_client.ssh_send_command_expect(ssh_client, self.node.extra)
        return ssh_client

    def exec_cps_cmds(self, cmds, ret_success=False):
        ssh_client = self.get_cps_client()
        result = self.ssh_client.ssh_exec_command_return(ssh_client, cmds)
        self.ssh_client.ssh_close(ssh_client)
        if not ret_success:
            return result
        logger.info(f'Cmd result:{result}.')
        return self.ssh_client.is_ssh_cmd_executed(result)

    @auto_retry(max_retry_times=5, delay_time=60)
    def get_neutron_endpoint(self):
        neutron_cmd = "openstack endpoint list" \
                      "|grep $OS_REGION_NAME " \
                      "|grep neutron" \
                      "|grep admin" \
                      "|sed 's/ //g'" \
                      "|sed 's/\/$(tenant_id)s//g'"
        logger.info(f'Get neutron endpoint, cmd:{neutron_cmd}.')
        result_dict = self.exec_cps_cmds(neutron_cmd)
        logger.info(f'The result:{result_dict}.')
        for one_result in result_dict:
            ret = one_result.split("|")
            if len(ret) == 9 and ret[3] == "neutron":
                return ret[7]
        raise Exception("Failed to obtain neutron_endpoint.")

    def get_keystone_endpoint(self):
        keystone_cmd = "echo $OS_AUTH_URL"
        logger.info(f'Get keystone endpoint, cmd:{keystone_cmd}.')
        result_dict = self.exec_cps_cmds(keystone_cmd)
        logger.info(f'The result:{result_dict}.')
        if len(result_dict) != 3 or "https://" not in result_dict[1]:
            return ""
        return result_dict[1]

    def get_portid_macaddr(self, ip_address):
        cmds = "neutron port-list|grep \\\"%s\\\"|sed 's/ //g'" % ip_address
        logger.info(f'Get portid macaddr, cmd:{cmds}.')
        result_list = self.exec_cps_cmds(cmds)
        logger.info(f'The result:{result_list}.')
        if len(result_list) == 0:
            return dict()
        # 查询结果可能出现其他警告信息, 导致解析出错,增加包含'|'过滤
        find_list = [v for v in result_list if '|' in v]
        logger.info(f'Find data list:{find_list}.')
        if not find_list:
            logger.error('Return data error, errMsg: no "|" in data to split')
            return dict()
        port_list = [v for v in find_list if 'ip_address' in v]
        if not port_list:
            logger.error('Return data error, errMsg: no "ip_address" in data to split')
            return dict()
        port_result = port_list[0].split("|")
        if len(port_result) != 6:
            logger.error(f'Length of splited port result is not 6, port result:{port_result}.')
            return dict()
        return {"portid": port_result[1], "macaddr": port_result[3]}

    def allowed_address_pairs(self, ip_address, float_ip1, float_ip2):
        param = self.get_portid_macaddr(ip_address)
        if not param:
            return False
        update_cmd = "neutron port-update %s --allowed-address-pairs " \
                     "type=dict list=true ip_address=%s,mac_address=%s " \
                     "ip_address=%s,mac_address=%s" % (
                         param.get('portid'), float_ip1, param.get('macaddr'),
                         float_ip2,
                         param.get('macaddr'))
        logger.info(f'Allowed address pairs, cmd:{update_cmd}.')
        return self.exec_cps_cmds(update_cmd, True)

    def set_backup_quota(self):
        backup_quota = "cps template-params-update " \
                       "--parameter use_default_quota_class=false " \
                       "--service cinder cinder-api; " \
                       "cps template-params-update " \
                       "--service cinder cinder-api " \
                       "--parameter quota_gigabytes=-1 " \
                       "quota_snapshots=-1 " \
                       "quota_backup_gigabytes=-1 " \
                       "quota_backups=-1 " \
                       "quota_volumes=-1;" \
                       "cps commit"
        logger.info(f'Set backup quota, cmd:{backup_quota}.')
        return self.exec_cps_cmds(backup_quota, True)

    @auto_retry(max_retry_times=5, delay_time=60)
    def get_availability_zone_dict(self):
        az_cmd = "cinder availability-zone-list " \
                 "|grep -v 'Status' " \
                 "|grep -v '+--' " \
                 "|sed 's/ //g'"
        result_dict = self.exec_cps_cmds(az_cmd)
        logger.info(f'The result:{result_dict}.')
        az_dict = {}
        az_list_cli_output_item_len = 2
        az_id_index = 0
        az_status_index = 1
        for one_result in result_dict:
            ret = one_result.strip("|").split("|")
            if ret and len(ret) == az_list_cli_output_item_len:
                az_dict[ret[az_id_index]] = ret[az_status_index]
        if not az_dict:
            raise Exception("Failed to obtain availability zone info.")
        logger.info(f"Succeeded to get availability zone info, availability zone info:{az_dict}.")
        return az_dict

    def get_business_az_info(self):
        existence_az_dict = self.get_availability_zone_dict()
        # 去除管理AZ
        for manage_az in ["manage-az", "dr-manage-az"]:
            if manage_az in existence_az_dict:
                existence_az_dict.pop(manage_az)
        return existence_az_dict
