import time

import paramiko
import utils.common.log as logger
from utils.common.code2Msg import code2Msg

from plugins.CSBS.common.param_check.os_param_check import check_pwd_will_expired
from plugins.CSBS.common.dpa_login_check import DPALoginCheck
from plugins.CSBS.common.ssh_client import SshClient
from plugins.CSBS.common.util import to_str
from plugins.CSBS.common.upgrade.karborproxy import KarborProxyOperation


class ProxyParamsCheck(object):
    def __init__(self, project_id):
        self.project_id = project_id
        self.ssh_client = SshClient()
        self.proxy_operation = KarborProxyOperation(self.project_id)
        self.proxy_ip_list = self.proxy_operation.all_ip_list
        self.proxy_user = self.proxy_operation.root_name
        self.proxy_pwd = self.proxy_operation.root_pwd
        self.proxy_root_pwd = self.proxy_operation.get_karborproxy_user_info().root_pwd
        self.dpa_login_check = DPALoginCheck(project_id)

    def check_proxy_login_ability(self, check_result):
        check_result = self.check_proxy_dpamanager_login_ability(check_result)

        check_result = self.check_proxy_root_login_ability(check_result)
        return check_result

    def check_proxy_dpamanager_login_ability(self, check_result):
        dpamanager_pwd_expired = []
        dpamanager_pwd_will_expired = []
        dpamanager_pwd_no_ssh_ips = []

        check_list = ["csbs_karborproxy_dpamanager_password"]

        def _get_dpamanager_login_check_result():
            success_count = 0
            for proxy_ip in self.proxy_ip_list:
                ssh_client = None
                pwd_expired_flag = None
                try:
                    ssh_client = self.ssh_client.create_ssh_client(proxy_ip, self.proxy_user, self.proxy_pwd)
                except Exception as err:
                    logger.error(f"Failed to login to Proxy node: IP {proxy_ip}, User dpamanager, error:{str(err)}")
                    dpamanager_pwd_no_ssh_ips.append(proxy_ip)
                finally:
                    if ssh_client:
                        pwd_expired_flag = check_pwd_will_expired(self.ssh_client, ssh_client,
                                                                  "dpamanager")
                        self.ssh_client.ssh_close(ssh_client)
                if self._check_password_expired(proxy_ip):
                    logger.error(f"The {self.proxy_user} account password has expired, please check.")
                    dpamanager_pwd_expired.append(proxy_ip)
                elif pwd_expired_flag:
                    logger.error(f"Node {proxy_ip}, User {self.proxy_user} password will expire, "
                                 f"please change and retry.")
                    dpamanager_pwd_will_expired.append(proxy_ip)
                else:
                    success_count += 1
            return success_count

        check_success_count = _get_dpamanager_login_check_result()
        if check_success_count == len(self.proxy_ip_list):
            check_result.set_check_result(param_keys=check_list, status=200, error_msg='')
        else:
            if dpamanager_pwd_expired:
                logger.error(f"Failed to login to Proxy node: IP {','.join(dpamanager_pwd_expired)}, "
                             "User dpamanager passwd is expired.")
                error_msg = code2Msg(645055)
            elif dpamanager_pwd_will_expired:
                logger.error(f"Node IP {','.join(dpamanager_pwd_will_expired)}, "
                             "User dpamanager password will expire.")
                error_msg = code2Msg(645056)
            else:
                logger.error(f"Failed to login to Proxy node: IP {','.join(dpamanager_pwd_no_ssh_ips)}, "
                             "User dpamanager.")
                error_msg = code2Msg(642001)
            check_result.set_check_result(param_keys=check_list, status=500, error_msg=error_msg)
        return check_result

    def check_proxy_root_login_ability(self, check_result):
        check_list = ["csbs_karborproxy_root_password"]
        check_success_count = 0
        proxy_root_no_ssh_ips = []
        for proxy_ip in self.proxy_ip_list:
            ssh_client = None
            try:
                ssh_client = self._check_root_login(proxy_ip)
            except Exception as err:
                logger.error(
                    "Failed to login to Proxy node: IP {}, User {}, "
                    "error: {}".format(proxy_ip, 'root', str(err))
                )
                proxy_root_no_ssh_ips.append(proxy_ip)
            finally:
                if ssh_client:
                    self.ssh_client.ssh_close(ssh_client)
            check_success_count += 1

        if check_success_count == len(self.proxy_ip_list):
            check_result.set_check_result(param_keys=check_list, status=200, error_msg='')
        else:
            logger.error(
                "Failed to login to Proxy node: IP {}, User {}".format(
                    ','.join(proxy_root_no_ssh_ips), "root")
            )
            check_result.set_check_result(param_keys=check_list, status=500, error_msg=code2Msg(642001))
        return check_result

    def _check_root_login(self, _ip):
        ssh_client = self.ssh_client.create_ssh_client(_ip, self.proxy_user, self.proxy_pwd)
        self.ssh_client.ssh_send_command_expect(ssh_client, 'sudo su root', 'password for root:', 50)
        self.ssh_client.ssh_send_command_expect(ssh_client, self.proxy_root_pwd, '#', 50)
        return ssh_client

    @staticmethod
    def _check_expires_str(ssh_client):
        expire_str = "has expired"
        ssh_client['channel'].send('\n')
        recv_str = ""
        num = 0
        while not (recv_str.endswith("$ ") or recv_str.endswith("# ")):
            num += 1
            time.sleep(1)
            if expire_str in recv_str or num > 10:
                logger.error("Get expired str, received shell prompt: "
                             "{}, in times:{}.".format(recv_str, num))
                return True
            if not ssh_client['channel'].recv_ready():
                continue
            recv_str = recv_str + to_str(ssh_client['channel'].recv(65535))
        logger.info("Test login success, no passwd expired.")
        return False

    def _check_password_expired(self, ip_num, port=22, timeout=5, width=80):
        """create a SSH client, check whecher pwd is expired or not"""
        ssh_client = dict()
        trans = paramiko.Transport((ip_num, port))
        try:
            trans.connect(username=self.proxy_user, password=self.proxy_pwd)
        except Exception as outer_e:
            try:
                if trans:
                    trans.close()
            except Exception as inner_e:
                logger.error("Failed to close connection after creating SSH "
                             "client failed: {}.".format(inner_e))
            err_msg = "Create ssh client failed. ip:{}, username:{}, " \
                      "port:{}".format(ip_num, self.proxy_user, port)
            if not self.proxy_pwd:
                err_msg += ', and password is none'
            if str(outer_e):
                err_msg += '. the error is:{}'.format(str(outer_e))
            logger.error(err_msg)
            raise Exception(err_msg) from outer_e
        trans.set_keepalive(30)
        channel = trans.open_session()
        channel.settimeout(timeout)
        channel.get_pty(width=width)
        channel.invoke_shell()
        stdout = channel.makefile('r', -1)
        ssh_client['ip'] = ip_num
        ssh_client['port'] = port
        ssh_client['username'] = self.proxy_user
        ssh_client['timeout'] = timeout
        ssh_client['width'] = width
        ssh_client['client'] = trans
        ssh_client['channel'] = channel
        ssh_client['stdout'] = stdout
        time.sleep(2)
        try:
            check_result = self._check_expires_str(ssh_client)
        except Exception as ex:
            logger.error(f"Try to get the shell prompt failed, error info: {str(ex)}.")
            check_result = True
        return check_result

    def check_dpa_login_ability(self, check_result):
        check_list = ['DPA_Business_Address', 'DPA_Cluster_Address', 'DPA_Admin', 'DPA_Admin_Password', 'DPA_Sec_Admin',
                      'DPA_Sec_Admin_Password', 'DPA_General_User', 'DPA_General_User_Password']
        try:
            self.dpa_login_check.check_account_info()
            check_result.set_check_result(param_keys=check_list, status=200, error_msg='')
        except Exception as exception:
            logger.error(f"Failed to login to dpa, exception: {str(exception)}")
            check_result.set_check_result(param_keys=check_list, status=500, error_msg=code2Msg(645060))
        return check_result

