import utils.common.log as logger
from utils.common.exception import HCCIException
from utils.common.message import Message

from plugins.CSBS.common.karbor import KarborUtil
from plugins.CSBS.common.params_tool import ParamTool
from plugins.CSBS.common.ssh_client import SshClient
from plugins.CSBS.common.step_base import DeployBaseSubJob
from plugins.CSBS.scripts.deploy.karbor.cbs_installer import sc_server_config, cts_server_config

logger.init("CtsSyslogConfig")


class CtsSyslogConfig(DeployBaseSubJob):
    def __init__(self, project_id, pod_id, regionid_list=None):
        super(CtsSyslogConfig, self).__init__(project_id, pod_id, regionid_list)
        self.project_id = project_id
        self.pod_id = pod_id
        self.karbor_util = KarborUtil(project_id, pod_id)
        self.params_util = ParamTool(project_id, pod_id)
        self.ssh_client = SshClient()

    def execute(self, project_id, pod_id, regionid_list=None):
        logger.info('Start init ssh client.')
        node_list = self.karbor_util.get_karbor_node_list()
        karbor_client = self.ssh_client.get_ssh_client(node_list[0])
        if not karbor_client:
            return Message(500, HCCIException("640051"))

        logger.info('Start configuring CTSServer')
        try:
            if not cts_server_config.CTSServerConfig(project_id, pod_id, karbor_client).install():
                self.ssh_client.ssh_close(karbor_client)
                return Message(500, HCCIException("640051"))
        except Exception as err:
            logger.error(f"Configuring CTSServer execute error:{err}")
            self.ssh_client.ssh_close(karbor_client)
            return Message(500, HCCIException("640051"))
        logger.info('Succeed configuring CTSServer')

        logger.info('Start configuring SCServer')
        try:
            if not sc_server_config.SCServerConfig(project_id, pod_id, karbor_client).install():
                self.ssh_client.ssh_close(karbor_client)
                return Message(500, HCCIException("640053"))
        except Exception as err:
            logger.error(f"Configuring SCServer execute error:{err}")
            self.ssh_client.ssh_close(karbor_client)
            return Message(500, HCCIException("640053"))
        self.ssh_client.ssh_close(karbor_client)
        logger.info('Succeed configuring SCServer')
        logger.info('All finished')
        return Message(200)

    def rollback(self, project_id, pod_id, regionid_list=None):
        node_list = self.karbor_util.get_karbor_node_list()
        karbor_client = self.ssh_client.get_ssh_client(node_list[0])
        if not karbor_client:
            return Message(500, HCCIException("640051"))

        try:
            if not cts_server_config.CTSServerConfig(project_id, pod_id, karbor_client).rollback():
                return Message(500, HCCIException("640051"))
        except Exception as err:
            logger.error(f"Execute error:{err}")
            return Message(500, HCCIException("640051"))

        try:
            if not sc_server_config.SCServerConfig(project_id, pod_id, karbor_client).rollback():
                return Message(500, HCCIException("640053"))
        except Exception as err:
            logger.error(f"Execute error:{err}")
            return Message(500, HCCIException("640053"))
        return Message(200)
