import os.path
import threading

import utils.common.log as logger
from utils.business.manageone_cert_manager import MOCertManager
from utils.common.exception import HCCIException
from utils.common.message import Message
from utils.common.ssh_util import Ssh

from plugins.CSBS.common.karbor import KarborUtil
from plugins.CSBS.common.params_tool import ParamTool, Params
from plugins.CSBS.common.step_base import DeployBaseSubJob
from plugins.CSBS.scripts.deploy.karbor.cbs_installer import cbs_karborcerts

logger.init("KarborCerts")

CERT_TMP_PATH = "plugins/CSBS/conf/deploy/karbor/mo_apimlb_certs"


class KarborCerts(DeployBaseSubJob):
    def __init__(self, project_id, pod_id, regionid_list=None):
        super(KarborCerts, self).__init__(project_id, pod_id, regionid_list)
        self.project_id = project_id
        self.pod_id = pod_id
        params = Params(project_id, pod_id)
        self.db_param_dict = params.get_params()
        self.destination = "/home/djmanager/"
        self.root_passwd = self.db_param_dict.get('karbor_user_passwd')
        self.mo_cert_manager = MOCertManager(pod_id)
        self.param_tool = ParamTool(project_id, pod_id)
        self.karbor_util = KarborUtil(project_id, pod_id)
        self.cert_tmp_path = os.path.join(self.param_tool.get_project_plugins_path(), CERT_TMP_PATH)
        self.r_lock = threading.RLock()

    def execute(self, project_id, pod_id, regionid_list=None):
        node_list = self.karbor_util.get_karbor_node_list()
        logger.info('Remove original trust.cer and nava_ca.crt in karbor node')
        try:
            if not cbs_karborcerts.KarborCerts(project_id, pod_id).rollback():
                return Message(500, HCCIException("640046"))
        except Exception as err:
            logger.error('Execute error: {}'.format(err))
            return Message(500, HCCIException("640046"))
        logger.info('Succeed removing certificates')
        try:
            if node_list:
                logger.info("Start getting trust.cer from iam")
                self._upload_mo_apimlb_certs(node_list[0])
        except HCCIException as err:
            logger.error("Execute error:%s" % str(err))
            return Message(500, err)
        except Exception as err:
            logger.error("Execute error:%s" % str(err))
            return Message(500, HCCIException("640001", str(err)))
        logger.info('Start uploading and synchronizing certificates to karbor node')
        try:
            if not cbs_karborcerts.KarborCerts(project_id, pod_id).install():
                return Message(500, HCCIException("640048"))
        except Exception as err:
            logger.error('Execute error:{}'.format(str(err)))
            return Message(500, HCCIException("640048"))
        logger.info('Succeed uploading and synchronizing certificates')
        logger.info('All finished')
        return Message(200)

    def rollback(self, project_id, pod_id, regionid_list=None):
        try:
            if not cbs_karborcerts.KarborCerts(project_id, pod_id).rollback():
                return Message(500, HCCIException("640048"))
        except HCCIException as err:
            logger.error(f"Execute error:{str(err)}")
            return Message(500, err)
        except Exception as err:
            logger.error(f"Execute error:{str(err)}")
            return Message(500, HCCIException(640048))
        return Message(200)

    def _upload_mo_apimlb_certs(self, node):
        """下载MO集成网关的IAM/SC/OC证书，上传到karbor节点"""
        result = self.mo_cert_manager.get_mo_apimlb_certs(self.cert_tmp_path)
        if not result:
            logger.error("Failed to get mo apimlb certs.")
            raise HCCIException("640047")
        logger.info("Succeed to get mo apimlb certs.")
        try:
            self._upload_mo_apimlb_certs_to_karbor(node)
        except Exception as err:
            logger.error(f"Upload MO apimlb certs to karbor error:{str(err)}")
            raise err

    def _upload_mo_apimlb_certs_to_karbor(self, node):
        iam_file = "{}/trust_apimlb_iam.cer".format(self.cert_tmp_path)
        sc_file = "{}/trust_apimlb_sc.cer".format(self.cert_tmp_path)
        oc_file = "{}/trust_apimlb_oc.cer".format(self.cert_tmp_path)
        for cert_file in [iam_file, sc_file, oc_file]:
            with self.r_lock:
                upload_res = Ssh.put_file(node.node_ip, node.user, node.user_pwd, cert_file, self.destination)
            if not upload_res:
                err_msg = "Upload cert file to karbor failed."
                logger.error(err_msg)
                raise Exception("Upload cert file to karbor failed.")
