import json
import os.path
import shutil
import time

import requests
import utils.common.log as logger
from utils.business.manageone_cert_manager import MOCertManager
from utils.business.param_util import ParamUtil
from utils.common.exception import HCCIException
from utils.common.ssh_util import Ssh

from plugins.CSBS.common.iam_util import IamUtil
from plugins.CSBS.common.step_base import DeployBaseSubJob
from plugins.CSBS.scripts.deploy.karborproxy.config.agent_proxy_param import AgentProxy
from plugins.CSBS.scripts.deploy.karborproxy.config.file_path import FilePath
from plugins.CSBS.scripts.deploy.karborproxy.config.project_set import project_preset
from plugins.CSBS.scripts.deploy.karborproxy.config.shell_cmd import ShellCmd
from plugins.CSBS.scripts.deploy.karborproxy.config.ssh_cmd import SShCmd

logger.init("KarborProxy")

INSECURE = False


class RegisterCertToManageone(DeployBaseSubJob):
    def __init__(self, project_id, pod_id, regionid_list=None):
        super().__init__(project_id, pod_id, regionid_list)
        self.params = ParamUtil()
        self.project_id = project_id
        self.pod_id = pod_id
        self.ssh_obj = Ssh()
        self.headers = None
        self.mo_cert_manager = None
        self.auth_url = ""
        self.iam_token = ""
        self.agent_proxy = AgentProxy(project_id, pod_id)
        self.ssh = SShCmd()
        self.iam_util = IamUtil(self.project_id, self.pod_id)
        self.agent_proxy.op_service_account = self.iam_util.get_iam_account_info()

    @project_preset
    def execute(self, project_id, pod_id, regionid_list=None):
        """
        标准调用接口：执行安装前预检查&安装&配置
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        self.iam_token = self.iam_util.get_iam_token(self.agent_proxy.op_service_account)
        self.agentproxy_register_unify_cert()
        self._exec_upload_cert_to_karborproxy()
        self._update_cert_permissions()
        self._restart_agentproxy_service()

    def build_unity_cert_body(self, float_ip):
        oc_cert = self.get_cert_body("KarborProxy-OC", "", "ServiceNameOC",
                                     "DependCertNameOC", "pem", "SSL",
                                     "Global", float_ip)
        iam_cert = self.get_cert_body("KarborProxy-IAM", "", "ServiceNameIAM",
                                      "DependCertNameIAM", "pem", "SSL",
                                      "Global", float_ip)
        karborproxy_cert = self.get_cert_body("KarborProxy-Cert", "", "", "",
                                              "pem", "SSL", "", float_ip)
        data = {
            "serviceName": "KarborProxy",
            "region": self.agent_proxy.region_id,
            "serviceUrl": {
                "update": "%s:8088/v1/unity_cert/cert_replace" % float_ip,
                "checkStatus": "%s:8088/v1/unity_cert/cert_status" % float_ip,
                "notice": "%s:8088/v1/unity_cert/cert_notify" % float_ip
            },
            "certs":
                [
                    oc_cert, iam_cert, karborproxy_cert
                ]
        }
        return data

    def get_cert_body(self, cert_name, description, dep_service_name,
                      dep_cert_name, dep_cert_format, dep_cert_type,
                      dep_region, float_ip):
        not_before_utc, not_after_utc = "", ""
        if cert_name == "KarborProxy-Cert":
            not_before_utc, not_after_utc = self.get_cert_not_before_after_utc(self.agent_proxy.karbor_proxy_ip0)
        depend_cert = {
            "certName": cert_name,
            "use": "SSL",
            "certFormat": "pem",
            "subjectName": "C=CN, O=Huawei, ST=SiChuan, CN=%s" % float_ip,
            "strExtentions": "",
            "keyAlgorithm": "rsa",
            "keySize": 2048,
            "notBefore": not_before_utc,
            "notAfter": not_after_utc,
            "description": description,
            "dependServiceName": dep_service_name,
            "dependCertName": dep_cert_name,
            "dependCertFormat": dep_cert_format,
            "dependType": dep_cert_type
        }
        if dep_region != "":
            depend_cert.update({"extendMap": {"dependRegion": dep_region}})
        return depend_cert

    def get_cert_not_before_after_utc(self, node_ip):
        def str_to_time(time_str):
            tmp = time.strptime(time_str, "%b %d %H:%M:%S %Y GMT")
            return time.strftime("%Y-%m-%d %H:%M:%S", tmp)

        cert_path = "/opt/huawei-data-protection/agentproxy/certs/" \
                    "agentproxy/tcp_cert/ca-cert.pem"
        cmd = "openssl x509 -in %s -text" % cert_path
        cert_text = self.ssh_obj.ssh_cmds(node_ip, cmd, 'dpamanager',
                                          self.agent_proxy.dpamanager_pwd,
                                          self.agent_proxy.root_pwd, "", "")
        not_before_utc = ""
        not_after_utc = ""
        for line in cert_text:
            if "Not Before" in line:
                line = line.replace("Not Before: ", "")
                not_before_utc = str_to_time(line)
            elif "Not After" in line:
                line = line.replace("Not After : ", "")
                not_after_utc = str_to_time(line)
                break
        return not_before_utc, not_after_utc

    def agentproxy_register_unify_cert(self):
        """
        注册证书
        :param:
        :return:
        """
        req_body = self.build_unity_cert_body(
            self.agent_proxy.management_float_ip)
        headers = {
            'Content-Type': 'application/json;charset=utf8',
            'Accept': 'application/json',
            'X-Auth-Token': self.iam_token
        }
        url = "https://%s/rest/unify-cert/components" % \
              self.agent_proxy.om_url

        rsp = requests.post(url, headers=headers,
                            data=json.dumps(req_body), verify=False)
        if rsp.status_code != 200:
            raise HCCIException(640102)

    def _exec_upload_cert_to_karborproxy(self):
        """
        Download the IAM/SC/OC certificate of the MO integration gateway
        and upload it to the KarborProxy node.
        """
        logger.info("Start uploading cert to KarborProxy.")
        self.mo_cert_manager = MOCertManager(self.agent_proxy.pod_id)
        result = self.mo_cert_manager.get_mo_apimlb_certs(
            FilePath.CERT_TMP_PATH)
        if not result:
            logger.error("Failed to get mo apimlb certs.")
            raise HCCIException("640047")
        else:
            logger.info("Succeed to get mo apimlb certs.")
        # Renaming a Certificate File
        # iam证书重命名
        self._cert_rename(FilePath.CERT_TMP_PATH, "trust_apimlb_iam.cer", "trust_iam.cer")
        # OC证书重命名
        self._cert_rename(FilePath.CERT_TMP_PATH, "trust_apimlb_oc.cer", "trust_oc.cer")
        iam_file = os.path.join(FilePath.CERT_TMP_PATH, "trust_iam.cer")
        oc_file = os.path.join(FilePath.CERT_TMP_PATH, "trust_oc.cer")
        # 上传iam证书
        self.ssh.put_file_to_all_roots(self.agent_proxy, iam_file, FilePath.CERT_PATH)
        # 生成bak目录
        iam_bak_path = os.path.join(FilePath.CERT_PATH, "bak")
        # 上传iam证书到bak目录
        self.ssh.put_file_to_all_roots(self.agent_proxy, iam_file, iam_bak_path)
        # 上传oc证书
        self.ssh.put_file_to_all_roots(self.agent_proxy, oc_file, FilePath.OC_CERT_PATH)
        # 生成oc证书路径
        oc_bak_path = os.path.join(FilePath.OC_CERT_PATH, "bak")
        # 上传OC证书到bak目录。
        self.ssh.put_file_to_all_roots(self.agent_proxy, oc_file, oc_bak_path)

    @staticmethod
    def _cert_rename(dir_path, old_file_name, new_file_name):
        """
        Certificate Rename

        """
        old_file_path = os.path.join(dir_path, old_file_name)
        new_file_path = os.path.join(dir_path, new_file_name)
        shutil.copyfile(old_file_path, new_file_path)
        if not os.path.exists(new_file_path):
            raise Exception(f"Failed to rename the {old_file_name} file.")
        logger.info(f"The {old_file_name} file is renamed successfully.")

    def _update_cert_permissions(self):
        """
        Change the owner, owner group, and permission of the certificate
        uploaded to the KarborProxy node.
        """
        logger.info("Start to modify certificate permissions.")
        self.ssh.ssh_cmds_to_all_proxys(self.agent_proxy, ShellCmd.update_iam_cer_permission)
        self.ssh.ssh_cmds_to_all_proxys(self.agent_proxy, ShellCmd.update_oc_cer_permission)
        logger.info("Certificate permission changed successfully.")

    def _restart_agentproxy_service(self):
        """
        Restart the AgentProxy service.
        """
        logger.info("Start to restart the AgentProxy service.")
        self.ssh.ssh_cmds_to_all_proxys(self.agent_proxy,
                                        ShellCmd.restart_agentproxy_service)
        logger.info("The AgentProxy service is restarted successfully.")
