import time

import utils.common.log as logger
from utils.common.exception import HCCIException
from utils.common.ssh_util import Ssh

from plugins.CSBS.common.karbor import KarborUtil
from plugins.CSBS.common.model import Node
from plugins.CSBS.common.step_base import DeployBaseSubJob
from plugins.CSBS.scripts.deploy.karborproxy.config.agent_proxy_param import AgentProxy
from plugins.CSBS.scripts.deploy.karborproxy.config.file_path import FilePath
from plugins.CSBS.scripts.deploy.karborproxy.config.project_set import project_preset
from plugins.CSBS.scripts.deploy.karborproxy.config.shell_cmd import ShellCmd
from plugins.CSBS.scripts.deploy.karborproxy.config.ssh_cmd import SShCmd

logger.init("KarborProxy StandardHardening")


class StandardHardening(DeployBaseSubJob):
    def __init__(self, project_id, pod_id, regionid_list=None):
        super().__init__(project_id, pod_id, regionid_list)
        self.agent_proxy = AgentProxy(project_id, pod_id)
        self.ssh_obj = Ssh()
        self.ssh = SShCmd()
        self.node_list = []
        self.karbor_info = KarborUtil(project_id, pod_id)
        for proxy_ip in self.agent_proxy.all_ip_list:
            self.node_list.append(
                Node(proxy_ip, self.agent_proxy.dpamanager_name, self.agent_proxy.dpamanager_pwd, 'root',
                     self.agent_proxy.root_pwd))

    @project_preset
    def execute(self, project_id, pod_id, regionid_list=None):
        self._add_protected_files()
        self.upload_file_to_all_dpamanagers()
        self.execute_harden_cmd()
        self.wait_os_start()
        self.delete_harden_files()
        self.resetting_karborproxy_node_account()

    def delete_harden_files(self):
        try:
            for node in self.node_list:
                self.ssh_obj.ssh_cmds(node.node_ip, ShellCmd.delete_proxy_harden_files,
                                      node.user, node.user_pwd, node.root_pwd, "", "")
        except Exception as err:
            logger.error("Delete harden scripts failed, reason: %s" % str(err))
            raise HCCIException(640134, str(err)) from err

    def resetting_karborproxy_node_account(self):
        """
        Reset the O&M account. You need to manually log in to the O&M account.
        :return:
        """
        cmd = 'chage -d 0 dpamanager'
        try:
            self.ssh.ssh_cmds_to_all_proxys(self.agent_proxy, cmd)
        except Exception as err:
            logger.error("[KarborProxy] execute error:%s" % str(err))

    def upload_file_to_all_dpamanagers(self):
        try:
            for proxy_ip in self.agent_proxy.all_ip_list:
                self.ssh_obj.put_file(proxy_ip,
                                      self.agent_proxy.dpamanager_name,
                                      self.agent_proxy.dpamanager_pwd,
                                      FilePath.STANDARD_HARDEN_FILE,
                                      self.agent_proxy.dpamanager_path)
        except Exception as err:
            raise HCCIException(640110, str(err)) from err
        logger.info("put file succ.")

    def execute_harden_cmd(self):
        try:
            for node in self.node_list:
                self.ssh_obj.ssh_cmds(node.node_ip, ShellCmd.harden_cmd,
                                      node.user, node.user_pwd, node.root_pwd, "",
                                      "")
        except Exception as err:
            logger.error("execute harden scripts failed, reason: %s" % str(err))
            raise HCCIException(640111, str(err)) from err

    def wait_os_start(self):
        time.sleep(120)
        success_ip_list = []
        num = 0
        while num < 60:
            success_ip_list = []
            for node in self.node_list:
                logger.info("begin to connect:" + node.node_ip)
                if self.is_connected_server(node):
                    logger.info(node.node_ip + " can connected.")
                    success_ip_list.append(node.node_ip)
            if len(success_ip_list) == len(self.agent_proxy.all_ip_list):
                break
            time.sleep(20)
            num = num + 1
        if num >= 60:
            ip_str = ','.join(success_ip_list)
            raise HCCIException(640092, ip_str)

    def is_connected_server(self, node):
        self.agent_proxy.proxy_ip = node.node_ip
        try:
            result = self.ssh_obj.ssh_cmds(node.node_ip, 'uname -a', node.user, node.user_pwd, node.root_pwd, "", "")
        except Exception as err:
            logger.info(str(err))
            result = ['False']
        return "Linux" in result[0]

    def _add_protected_files(self):
        logger.info("[KarborProxy] Adding a protected file")
        self.ssh.ssh_cmds_to_all_proxys(
            self.agent_proxy, ShellCmd.add_protected_files)

        karbor_node_list = self.karbor_info.get_karbor_node_list()
        for karbor_node in karbor_node_list:
            self.ssh.ssh_cmd_to_karbor(karbor_node, ShellCmd.add_i_attr)
        logger.info("[Karbor] Succeeded "
                    "in uploading the assistant package to Kabor.")
