#!/bin/bash
if [[ ${UID} -eq 0 ]];then
    sudo -u openstack bash "$(readlink -f "$0")" "$@";exit $?
elif [[ ${UID} -ne 51001 ]];then
    echo "Please switch to user(openstack) for run." && exit 1
fi

source /opt/huawei/dj/inst/utils.sh
export LOG_TAG="mo_update_cert"

# 各证书路径
IAM_FILE="/opt/huawei/dj/DJSecurity/mo_certs/trust_iam.cer"
OC_FILE="/opt/huawei/dj/DJSecurity/mo_certs/trust_oc.cer"
SC_FILE="/opt/huawei/dj/DJSecurity/mo_certs/trust_sc.cer"
OPENSTACK_CA_FILE="/opt/huawei/dj/DJSecurity/server-cert/karbor/openstack/nova_ca.crt"
APICOM_CA_FILE="/opt/huawei/dj/DJSecurity/mo_certs/apicom_ca.cer"
ARB_CA_FILE="/opt/huawei/dj/DJSecurity/arb/arb_ca.crt"
APIGW_CA_FILE="/opt/huawei/dj/DJSecurity/mo_certs/apigw_ca.cer"

# 7天内即将过期发送严重告警，时间:604800s
CRITICAL_ALARM_TIME=604800

#告警时间
ALARM_TIME=$(get_cms_info "certificate.alarm.date")
if [[ -z "${ALARM_TIME}" ]];then
    ALARM_TIME="90"
fi

# 获取证书开关
CERT_VERIFY_SWITCH=$(get_features | grep Certificate_verification_switch | grep -c true)

# 检查证书存在性
# $1: 证书路径
function check_cert_existence() {
    cert_path="$1"
    alarm_id=1020760

    # 证书校验开关关闭，取消告警
    if [[ ${CERT_VERIFY_SWITCH} -lt 1 ]];then
        echo_info "${cert_path}:The certificate verify switch is off so will not check the certs."
        recover_not_exist_alarm "${cert_path}"; recover_not_exist=$?
        return ${recover_not_exist}
    fi

    # 检查证书存在性，不存在发送告警，否则恢复告警
    if [[ ! -f "${cert_path}" ]];then
        echo_info "The cert does not exist: ${cert_path}."
        send_not_exist_alarm "${cert_path}"; send_not_exist=$?
        return ${send_not_exist}
    else
        echo_info "The cert does exist: ${cert_path}."
        recover_not_exist_alarm "${cert_path}"; recover_not_exist=$?
        return ${recover_not_exist}
    fi
}


# 检查证书时间
# $1: 证书路径
function check_certificate_date()
{
    cert_path="$1"

    # 证书校验开关关闭，不进行检查，并取消告警
    if [[ ${CERT_VERIFY_SWITCH} -lt 1 ]];then
        echo_info "${cert_path}:The certificate verify switch is off so will not check the certs."
        recover_expired_alarm "${cert_path}"; recover_expired=$?
        recover_expiring_alarm "${cert_path}"; recover_expiring=$?
        return ${recover_expired} && ${recover_expiring}
    fi

    alarm_time_second=$((ALARM_TIME*86400))

    # 证书存在性已经单独检查，这里如果不存在则直接返回
    if [[ ! -f "${cert_path}" ]];then
        return 0
    fi

    # 如果无法用 x509 解析，说明证书有问题，后续会在 check_cert_error 中检查并处理告警
    if ! openssl x509 -subject_hash -fingerprint -in "${cert_path}" -noout -text &>/dev/null;then
        echo_warn "${cert_path} can not use openssl to check."
        return
    fi

    diff_time=$(get_cert_diff_time "${cert_path}")
    if [[ ${diff_time} -lt 0 ]];then
        # 证书已经过期，发布或者转换成已经过期告警（针对即将过期证书，要先撤销这条告警）
        echo_info "${cert_path}:The certificate has expired."
        recover_expiring_alarm "${cert_path}"; recover_ret=$?
        send_expired_alarm "${cert_path}"; send_ret=$?
        return ${recover_ret} && ${send_ret}
    elif [[ ${diff_time} -ge 0 ]] && [[ ${diff_time} -le ${CRITICAL_ALARM_TIME} ]];then
        # 7 天内即将过期发送紧急告警
        echo_info "${cert_path}:The certificate is about to expire in 7 days."
        recover_expired_alarm "${cert_path}"; recover_ret=$?
        send_expiring_alarm "${cert_path}"; send_ret=$?
        return ${recover_ret} && ${send_ret}
    elif [[ ${diff_time} -ge ${CRITICAL_ALARM_TIME} ]] && [[ ${diff_time} -le ${alarm_time_second} ]];then
        # 90 天内即将过期发送重要告警
        echo_info "${cert_path}:The certificate is about to expire."
        recover_expired_alarm "${cert_path}"; recover_ret=$?
        send_expiring_alarm "${cert_path}"; send_ret=$?
        return ${recover_ret} && ${send_ret}
    fi

    echo_info "${cert_path}:The certificate is normal."
    recover_expired_alarm "${cert_path}"; recover_expired=$?
    recover_expiring_alarm "${cert_path}"; recover_expiring=$?
    return ${recover_expired} && ${recover_expiring}
}

function get_cert_time()
{
    cert_path=$1
    cer_date=$(openssl x509 -subject_hash -fingerprint -in "${cert_path}" -noout -text | grep -w 'Not After')
    IFS=" " read -r -a cer_date_arr <<< "${cer_date}"
    cer_date_time=$(date --date="${cer_date_arr[3]} ${cer_date_arr[4]} ${cer_date_arr[6]} ${cer_date_arr[5]}" +%s)
    echo "${cer_date_time}"
}

function get_cert_diff_time()
{
    cer_date_time=$(get_cert_time "$1")
    cur_date_time=$(date +%s)
    diff_time=$((cer_date_time-cur_date_time))
    echo "${diff_time}"
}

function get_cer_time_show() 
{
    cer_date_time=$(get_cert_time "$1")
    cer_date_time_disp=$(date -d @"${cer_date_time}" "+%Y-%m-%d")
    echo "${cer_date_time_disp}"
}

# 检查证书是否错误，发送告警
function check_cert_error()
{
    cert_path=$1

    alarm_id=1020761

   # 证书校验开关关闭，取消告警
    if [[ ${CERT_VERIFY_SWITCH} -lt 1 ]];then
        echo_info "${cert_path}:The certificate verify switch is off so will not check the certs."
        clear_alarm ${alarm_id} "{\"FullCertificatePath\":\"${cert_path}\"}"
        ECHO_RETURN $? "${cert_path}:The certificate error recover alarm send failed"
        echo_info "${cert_path}:The certificate error recover alarm send success"
        return 0
    fi

    # 通过 check_karbor_connect 来检查证书校验是否有错误
    if [[ "$(basename "${cert_path}")" == "trust_iam.cer" ]];then
        ssl_count=$(check_karbor_connect | grep IAM | grep -c SSLError)
    elif [[ "$(basename "${cert_path}")" == "trust_sc.cer" ]];then
        ssl_count=$(check_karbor_connect | grep 'ManageOneSC' | grep -c SSLError)
    elif [[ "$(basename "${cert_path}")" == "trust_oc.cer" ]];then
        ssl_count=$(check_karbor_connect | grep 'ManageOneCMC' | grep -c SSLError)
    elif [[ "$(basename "${cert_path}")" == "nova_ca.crt" ]];then
        ssl_count=$(check_karbor_connect | grep Nova | grep -c SSLError)
    elif [[ "$(basename "${cert_path}")" == "apicom_ca.cer" ]];then
        ssl_count=$(check_karbor_connect --scope include_extend | grep 'ApiCom' | grep -c SSLError)
    elif [[ "$(basename "${cert_path}")" == "arb_ca.crt" ]];then
        ssl_count=$(check_karbor_connect --scope include_extend | grep 'Arb' | grep -c SSLError)
    elif [[ "$(basename "${cert_path}")" == "apigw_ca.cer" ]];then
        ssl_count=$(check_karbor_connect --scope include_extend | grep 'Kms' | grep -c SSLError)
    fi

    if [[ ${ssl_count} -gt 0 ]];then
        send_alarm ${alarm_id} "{\"FullCertificatePath\":\"${cert_path}\"}"
        ECHO_RETURN $? "${cert_path}:The certificate error alarm send failed"
        echo_info "${cert_path}:The certificate error alarm send success"
    else
        clear_alarm ${alarm_id} "{\"FullCertificatePath\":\"${cert_path}\"}"
        ECHO_RETURN $? "${cert_path}:The certificate error recover alarm send failed"
        echo_info "${cert_path}:The certificate error recover alarm send success"
    fi
    return 0
}

function send_not_exist_alarm()
{
    cert_path=$1
    if ! send_alarm 1020760 "{\"FullCertificatePath\":\"${cert_path}\"}";then
        echo_error "${cert_path}:The certificate not exist alarm send failed."
        return 1
    fi
    echo_info "${cert_path}:The certificate not exist alarm send success."
    return 0
}

function recover_not_exist_alarm()
{
    cert_path=$1
    if ! clear_alarm 1020760 "{\"FullCertificatePath\":\"${cert_path}\"}";then
        echo_error "${cert_path}:The certificate not exist recover alarm send failed"
        return 1
    fi
    echo_info "${cert_path}:The certificate not exist recover alarm send success"
    return 0
}

function send_expired_alarm()
{
    cert_path=$1
    if ! send_alarm 1023278 "{\"FullCertificatePath\":\"${cert_path}\"}";then
        echo_error "${cert_path}:The certificate expired alarm send failed"
        return 1
    fi
    echo_info "${cert_path}:The certificate expired alarm send success"
    return 0
}

function recover_expired_alarm()
{
    cert_path=$1
    if ! clear_alarm 1023278 "{\"FullCertificatePath\":\"${cert_path}\"}";then
        echo_error "${cert_path}:The certificate expired recover alarm send failed"
        return 1
    fi
    echo_info "${cert_path}:The certificate expired recover alarm send success"
    return 0
}

function send_expiring_alarm()
{
    cert_path=$1
    cer_date_time_disp=$(get_cer_time_show "${cert_path}")
    if ! send_alarm 1023279 "{\"FullCertificatePath\":\"${cert_path}\", \"ExpirationTime\":\"${cer_date_time_disp}\"}";then
        echo_error "${cert_path}:The certificate is about to expire alarm send failed"
        return 1
    fi
    echo_info "${cert_path}:The certificate is about to expire alarm send success"
    return 0
}

function recover_expiring_alarm()
{
    cert_path=$1
    cer_date_time_disp=$(get_cer_time_show "${cert_path}")
    if ! clear_alarm 1023279 "{\"FullCertificatePath\":\"${cert_path}\", \"ExpirationTime\":\"${cer_date_time_disp}\"}";then
        echo_error "${cert_path}:The certificate is about to expire recover alarm send failed"
        return 1
    fi
    echo_info "${cert_path}:The certificate is about to expire recover alarm send success"
    return 0
}

function main()
{
    cert_list="${IAM_FILE} ${OC_FILE} ${SC_FILE} ${OPENSTACK_CA_FILE} ${APICOM_CA_FILE}"

    # 管理面高可用场景，需检查云平台统一仲裁证书
    if [[ $(get_info profile) == "private_recover" ]];then
        cert_list="${cert_list} ${ARB_CA_FILE}"
    fi

    # 国密场景，需检查 apitgateway 证书
    if [[ $(get_info cipher_type) == "SMCompatible" ]];then
        cert_list="${cert_list} ${APIGW_CA_FILE}"
    fi

    # 对 iam, oc, sc, nova, apicom, arb(管理面高可用场景)， apigateway(国密场景) 检查其证书存在性，有效期，有效性
    for cert_path in ${cert_list}
    do
        check_cert_existence "${cert_path}"
        check_certificate_date "${cert_path}"
        check_cert_error "${cert_path}"
    done

    # 对于 DJSecurity 目录下的其他证书（以cert.pem结尾）进行有效期检查
    check_file=$(find /opt/huawei/dj/DJSecurity -type f)
    while IFS= read -r element;do
        file_name=$(basename "${element}")
        if [[ "${file_name##*-}" == "cert.pem" ]];then
            check_certificate_date "${element}"
        fi
    done <<< "${check_file}"
}

main